/*
 * File Name: FileUtils.java 
 * Created on 2014-4-4
 */
package net.iyouqu.video.basecommon.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.zip.ZipEntry;
import java.util.zip.ZipException;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

import net.iyouqu.video.basecommon.config.Constant;
import net.iyouqu.video.basecommon.utils.http.IOUtils;
import net.iyouqu.video.basecommon.utils.ui.UIUtils;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Environment;
import android.os.StatFs;
import android.util.Log;

/**
 * File Name: FileUtils.java
 * Create by:Bruce on 2014-12-12 下午4:59:09 
 * @version 1.0.0
 */
public class FileUtils {

	public static boolean sdAvailable() {
		return Environment.MEDIA_MOUNTED_READ_ONLY.equals(Environment
				.getExternalStorageState())
				|| Environment.MEDIA_MOUNTED.equals(Environment
						.getExternalStorageState());
	}
	
	/**
	 * 判断文件是否存在
	 * @param pathName 文件全路径名称
	 * @return 有返回TRUE，否则FALSE
	 */
	public static boolean exists(String pathName) {
		try {
			return !StringUtils.isEmpty(pathName) && new File(pathName).exists();
		} catch (Exception e) {
			LogUtils.e(e);
			return false;
		}
	}

	/**
	 * 创建目录
	 * @param dirPath 目录全路径名
	 * @return 是否创建成功
	 */
	public static boolean createDir(String dirPath) {
		File file = new File(dirPath);
		if (!file.exists() || !file.isDirectory()) {
			return file.mkdir();
		}
		return true;
	}

	/**
	 * 创建多级目录，如果上级目录不存在，会先创建上级目录
	 * @param dirPath 目录全路径名
	 * @return 是否创建成功
	 */
	public static boolean createDirs(String dirPath) {
		File file = new File(dirPath);
		if (!file.exists() || !file.isDirectory()) {
			return file.mkdirs();
		}
		return true;
	}

	/**
	 * 删除文件
	 * @param path 被删除的文件路径
	 * @return 是否删除成功
	 */
	public static boolean deleteFile(String path) {
		return new File(path).delete();
	}
	
	public static boolean delete(File path) {

        boolean result = true;
        if (path.exists()) {
            if (path.isDirectory()) {
                File[] files=path.listFiles();
                if(null!=files){
                    for (File child : files) {
                        result &= delete(child);
                    }
                    result &= path.delete(); // Delete empty directory.
                }
            }
            if (path.isFile()) {
                result &= path.delete();
            }
            if (!result) {
                Log.e(null, "Delete failed;");
            }
            return result;
        } else {
            Log.e(null, "File does not exist.");
            return false;
        }
    }

	/**
	 * 复制文件
	 * @param srcPath  被复制的源文件路径
	 * @param destPath 复制后的目标文件路径
	 * @return 是否复制成功
	 */
	public static boolean copyFile(String srcPath, String destPath) {
		File srcFile = new File(srcPath);
		File destFile = new File(destPath);
		return copyFile(srcFile, destFile);
	}

	/**
	 * 复制文件
	 * @param srcFile  被复制的源文件
	 * @param destFile 复制后的目标文件
	 * @return 是否复制成功
	 */
	public static boolean copyFile(File srcFile, File destFile) {
		if (!srcFile.exists() || !srcFile.isFile()) {
			return false;
		}
		InputStream in = null;
		OutputStream out = null;
		try {
			in = new FileInputStream(srcFile);
			out = new FileOutputStream(destFile);
			byte[] buffer = new byte[1024];
			int i = -1;
			while ((i = in.read(buffer)) > 0) {
				out.write(buffer, 0, i);
			}
			out.flush();
			srcFile.delete();
		} catch (Exception e) {
			LogUtils.e(e);
			return false;
		} finally {
			IOUtils.close(out);
			IOUtils.close(in);
		}
		return true;
	}

	/**
	 * 判断文件是否可读
	 * @param pathName 文件全路径名称
	 * @return 可读返回TRUE，否则FALSE
	 */
	public static boolean isReadable(String pathName) {
		try {
			if (StringUtils.isEmpty(pathName)) {
				return false;
			}
			File f = new File(pathName);
			return f.exists() && f.canRead();
		} catch (Exception e) {
			LogUtils.e(e);
			return false;
		}
	}

	/**
	 * 判断文件是否可写
	 * @param pathName 文件全路径名称
	 * @return 可写返回TRUE，否则FALSE
	 */
	public static boolean isWriteable(String pathName) {
		try {
			if (StringUtils.isEmpty(pathName)) {
				return false;
			}
			File f = new File(pathName);
			return f.exists() && f.canWrite();
		} catch (Exception e) {
			LogUtils.e(e);
			return false;
		}
	}

	/**
	 * 修改文件的权限
	 * @param path 被修改的文件路径
	 * @param mode 修改后的权限
	 */
	public static void chmod(String path, int mode) {
		chmod(path, mode + "");
	}

	/**
	 * 修改文件的权限
	 * @param path 被修改的文件路径
	 * @param mode 修改后的权限
	 */
	public static void chmod(String path, String mode) {
		try {
			String command = "chmod " + mode + " " + path;
			Runtime runtime = Runtime.getRuntime();
			runtime.exec(command);
		} catch (IOException e) {
			LogUtils.e(e);
		}
	}

	/**
	 * 获取文件的大小
	 * @param path 文件的路径名
	 * @return 如果返回-1，则代表文件是目录或者路径为null
	 */
	public static long getFileSize(String path) {
		if (null == path) {
			return -1;
		}
		File f = new File(path);
		if (f.isDirectory()) {
			return -1;
		} else {
			return f.length();
		}
	}

	/** 获取到目录下面文件的大小。包含了子目录。 */
	public static long getDirLength(File f) throws IOException {
		if (null == f) {
			return -1;
		}
		if (f.isFile()) return f.length();
		long size = 0;
		File[] files = f.listFiles();
		if (files != null) {
			for (File file : files) {
				long length = 0;
				if (file.isFile()) {
					length = file.length();
				} else {
					length = getDirLength(file);
				}
				size += length;
			}
		}
		return size;
	}

	/**
	 * 把数据写入文件
	 * @param is       数据流
	 * @param path     文件路径
	 * @param recreate 如果文件存在，是否需要删除重建
	 * @return 是否写入成功
	 */
	public static boolean writeFile(InputStream is, String path, boolean recreate) {
		boolean res = false;
		File f = new File(path);
		FileOutputStream fos = null;
		try {
			if (recreate && f.exists()) {
				f.delete();
			}
			if (!f.exists() && null != is) {
				int count = -1;
				byte[] buffer = new byte[1024];
				fos = new FileOutputStream(f);
				while ((count = is.read(buffer)) != -1) {
					fos.write(buffer, 0, count);
				}
				res = true;
			}
		} catch (FileNotFoundException e) {
			LogUtils.e(e);
		} catch (IOException e) {
			LogUtils.e(e);
		} catch (Exception e) {
			LogUtils.e(e);
		} finally {
			IOUtils.close(fos);
			IOUtils.close(is);
		}
		return res;
	}

	/**
	 * 把字符串数据写入文件
	 * @param content 需要写入的字符串
	 * @param path    文件路径名称
	 * @param append  是否以添加的模式写入
	 * @return 是否写入成功
	 */
	public static boolean writeFile(String content, String path, boolean append) {
		boolean res = false;
		File f = new File(path);
		RandomAccessFile raf = null;
		try {
			if (f.exists()) {
				if (!append) {
					f.delete();
					f.createNewFile();
				}
			} else {
				f.createNewFile();
			}

			if (f.canWrite()) {
				raf = new RandomAccessFile(f, "rw");
				raf.seek(raf.length());
				raf.write(content.getBytes());
				res = true;
			}
		} catch (FileNotFoundException e) {
			LogUtils.e(e);
		} catch (IOException e) {
			LogUtils.e(e);
		} catch (Exception e) {
			LogUtils.e(e);
		} finally {
			IOUtils.close(raf);
		}
		return res;
	}

	/**
	 * 把int型数据写入文件
	 * @param content int数据
	 * @param path    文件的路径
	 * @param append  是否以追加模式写入
	 * @return 是否写入成功
	 */
	public static boolean writeFile(int content, String path, boolean append) {
		boolean res = false;
		File f = new File(path);
		RandomAccessFile raf = null;
		try {
			if (f.exists()) {
				if (!append) {
					f.delete();
					f.createNewFile();
				}
			} else {
				f.createNewFile();
			}

			if (f.canWrite()) {
				raf = new RandomAccessFile(f, "rw");
				raf.seek(raf.length());
				raf.writeInt(content);
				res = true;
			}
		} catch (FileNotFoundException e) {
			LogUtils.e(e);
		} catch (IOException e) {
			LogUtils.e(e);
		} catch (Exception e) {
			LogUtils.e(e);
		} finally {
			IOUtils.close(raf);
		}
		return res;
	}

	/**
	 * 读取文件的32位数据，即一个int型数据
	 * @param path 文件的路径
	 * @return 读取的int数据
	 */
	public static Integer readInt(String path) {
		Integer res = null;
		File f = new File(path);
		RandomAccessFile raf = null;
		try {
			if (!f.exists()) {
				return null;
			}

			if (f.canRead()) {
				raf = new RandomAccessFile(f, "r");
				res = raf.readInt();
			}
		} catch (FileNotFoundException e) {
			LogUtils.e(e);
		} catch (IOException e) {
			LogUtils.e(e);
		} catch (Exception e) {
			LogUtils.e(e);
		} finally {
			IOUtils.close(raf);
		}
		return res;
	}

	/**
	 * 把键值对写入文件
	 * @param filePath 文件路径
	 * @param key      键
	 * @param value    值
	 * @param comment  该键值对的注释
	 */
	public static void writeProperties(String filePath, String key, String value, String comment) {
		if (StringUtils.isEmpty(key) || StringUtils.isEmpty(filePath)) {
			return;
		}
		FileInputStream fis = null;
		FileOutputStream fos = null;
		File f = new File(filePath);
		try {
			if (!f.exists() || !f.isFile()) {
				f.createNewFile();
			}
			fis = new FileInputStream(f);
			Properties p = new Properties();
			p.load(fis);// 先读取文件，再把键值对追加到后面
			p.setProperty(key, value);
			fos = new FileOutputStream(f);
			p.store(fos, comment);
		} catch (IllegalArgumentException e) {
			LogUtils.e(e);
		} catch (IOException e) {
			LogUtils.e(e);
		} finally {
			IOUtils.close(fis);
		}
	}

	/** 把字符串键值对的map写入文件 */
	public static void writeMap(String filePath, Map<String, String> map, boolean append, String comment) {
		if (map == null || map.size() == 0 || StringUtils.isEmpty(filePath)) {
			return;
		}
		FileInputStream fis = null;
		FileOutputStream fos = null;
		File f = new File(filePath);
		try {
			if (!f.exists() || !f.isFile()) {
				f.createNewFile();
			}
			Properties p = new Properties();
			if (append) {
				fis = new FileInputStream(f);
				p.load(fis);// 先读取文件，再把键值对追加到后面
			}
			p.putAll(map);
			fos = new FileOutputStream(f);
			p.store(fos, comment);
		} catch (IllegalArgumentException e) {
			LogUtils.e(e);
		} catch (IOException e) {
			LogUtils.e(e);
		} finally {
			IOUtils.close(fis);
			IOUtils.close(fos);
		}
	}

	/** 根据值读取 */
	public static String readProperties(String filePath, String key, String defaultValue) {
		if (StringUtils.isEmpty(key) || StringUtils.isEmpty(filePath)) {
			return null;
		}
		String value = null;
		FileInputStream fis = null;
		File f = new File(filePath);
		try {
			if (!f.exists() || !f.isFile()) {
				f.createNewFile();
			}
			fis = new FileInputStream(f);
			Properties p = new Properties();
			p.load(fis);
			value = p.getProperty(key, defaultValue);
		} catch (IOException e) {
			LogUtils.e(e);
		} finally {
			IOUtils.close(fis);
		}
		return value;
	}

	/** 把字符串键值对的文件读入map */
	@SuppressWarnings({"rawtypes", "unchecked"})
	public static Map<String, String> readMap(String filePath, String defaultValue) {
		if (StringUtils.isEmpty(filePath)) {
			return null;
		}
		Map<String, String> map = null;
		FileInputStream fis = null;
		File f = new File(filePath);
		try {
			if (!f.exists() || !f.isFile()) {
				f.createNewFile();
			}
			fis = new FileInputStream(f);
			Properties p = new Properties();
			p.load(fis);
			map = new HashMap<String, String>((Map) p);// 因为properties继承了map，所以直接通过p来构造一个map
		} catch (IllegalArgumentException e) {
			LogUtils.e(e);
		} catch (IOException e) {
			LogUtils.e(e);
		} finally {
			IOUtils.close(fis);
		}
		return map;
	}

	/**
	 * 返回文件名 如果flag为true，文件以/结时，返回上一级，否则返回空字符串，跟目录直接返回/
	 * @param path 文件路径
	 * @param flag 当文件路径以/结时，是否返回上一级
	 */
	public static String getPathName(String path, boolean flag) {
		int index = path.lastIndexOf('/');
		if (index == -1 || path.equals("/")) {// 没找到或者为根目录。直接返回path
			return path;
		}
		if (index == 0 && path.length() > 1) {// 如果只有一个'/'，且处于第一个，返回后面的字符串
			return path.substring(1);
		}
		if (index == (path.length() - 1) && flag) {// 如果最后一个'/'，获取上一级
			return getPathName(path.substring(0, path.length() - 1), flag);
		}
		return path.substring(index + 1);
	}

	/** 获取应用的路径，如果没有SD卡，则返回data/data目录下的应用目录，如果有，则返回SD卡上的应用目录 */
	public static String getAppFilesDir() {
		if (isSDcardExist()) {
			return getSDCardPath() + Constant.APP_CACHE_DIR_NAME;
		} else {
			return getAppFilesDirByData();
		}
	}

	/** 判断是否有存储卡，有返回TRUE，否则FALSE */
	public static boolean isSDcardExist() {
		return Environment.getExternalStorageState().equals(Environment.MEDIA_MOUNTED);
	}

	/** 获取安装在用户手机上的sdcard/下的chianandroid目录 */
	public static String getSDCardPath() {
		return Environment.getExternalStorageDirectory().getAbsolutePath();
	}
	
	/** 获取文件data/data目录 */
	public static String getAppFilesDirByData() {
		return UIUtils.getContext().getFilesDir().getAbsolutePath();
	}
	
	/**
	 * 获取手机外部空间大小
	 * @return 单位为byte
	 */
	@SuppressWarnings("deprecation")
	public static long getExternalTotalSpace() {
		long totalSpace = -1l;
		if (FileUtils.isSDcardExist()) {
			try {
				String path = Environment.getExternalStorageDirectory().getPath();// 获取外部存储目录即 SDCard
				StatFs stat = new StatFs(path);
				long blockSize = stat.getBlockSize();
				long totalBlocks = stat.getBlockCount();
				totalSpace = totalBlocks * blockSize;
			} catch (Exception e) {
				LogUtils.e(e);
			}
		}
		return totalSpace;
	}

	/**
	 * 获取外部存储可用空间
	 * @return 单位为byte
	 */
	@SuppressWarnings("deprecation")
	public static long getExternalSpace() {
		long availableSpace = -1l;
		if (FileUtils.isSDcardExist()) {
			try {
				String path = Environment.getExternalStorageDirectory().getPath();
				StatFs stat = new StatFs(path);
				availableSpace = stat.getAvailableBlocks() * (long) stat.getBlockSize();
			} catch (Exception e) {
				LogUtils.e(e);
			}
		}
		return availableSpace;
	}

	/**
	 * 获取手机内部空间大小
	 * @return 单位为byte
	 */
	@SuppressWarnings("deprecation")
	public static long getTotalInternalSpace() {
		long totalSpace = -1l;
		try {
			String path = Environment.getDataDirectory().getPath();
			StatFs stat = new StatFs(path);
			long blockSize = stat.getBlockSize();
			long totalBlocks = stat.getBlockCount();// 获取该区域可用的文件系统数
			totalSpace = totalBlocks * blockSize;
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return totalSpace;
	}

	/**
	 * 获取手机内部可用空间大小
	 * @return 单位为byte
	 */
	@SuppressWarnings("deprecation")
	public static long getAvailableInternalMemorySize() {
		long availableSpace = -1l;
		try {
			String path = Environment.getDataDirectory().getPath();// 获取 Android 数据目录
			StatFs stat = new StatFs(path);// 一个模拟linux的df命令的一个类,获得SD卡和手机内存的使用情况
			long blockSize = stat.getBlockSize();// 返回 Int ，大小，以字节为单位，一个文件系统
			long availableBlocks = stat.getAvailableBlocks();// 返回 Int ，获取当前可用的存储空间
			availableSpace = availableBlocks * blockSize;
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return availableSpace;
	}
	

	/**
	 * 使用zip格式压缩文件
	 * @param srcPath  源文件路径
	 * @param destPath 目标文件路径
	 * @param zipEntry 压缩后的zip实体名称
	 * @return 是否压缩成功
	 */
	public static boolean zip(String srcPath, String destPath, String zipEntry) {
		return zip(new File(srcPath), new File(destPath), zipEntry);
	}

	/**
	 * 使用zip格式压缩文件
	 * @param srcFile  源文件
	 * @param destFile 目标文件
	 * @param zipEntry 压缩后的zip实体名称
	 * @return 是否压缩成功
	 */
	public static boolean zip(File srcFile, File destFile, String zipEntry) {
		boolean res = false;
		if (null == srcFile || !srcFile.exists() || !srcFile.canRead()) {
			return false;
		}
		if (destFile.exists()) {
			destFile.delete();
		}
		ZipOutputStream zos = null;
		FileOutputStream fos = null;
		FileInputStream fis = null;
		try {
			fos = new FileOutputStream(destFile, false);
			zos = new ZipOutputStream(fos);
			ZipEntry entry = new ZipEntry(zipEntry);
			zos.putNextEntry(entry);
			fis = new FileInputStream(srcFile);
			byte[] buffer = new byte[32];
			int cnt = 0;
			while ((cnt = fis.read(buffer)) != -1) {
				zos.write(buffer, 0, cnt);
			}
			zos.flush();
			res = true;
		} catch (FileNotFoundException e) {
			LogUtils.e(e);
		} catch (Exception e) {
			LogUtils.e(e);
		} finally {
			IOUtils.close(fis);
			IOUtils.close(fos);
			IOUtils.close(zos);
		}
		return res;
	}

	/**
	 * 解压缩zip文件
	 * @param srcPath  需要被解压的文件路径
	 * @param destPath 被解压后的文件存放的目录路径
	 * @return 是否解压成功
	 */
	public static boolean unZip(String srcPath, String destPath) {
		return unZip(new File(srcPath), new File(destPath));
	}

	/**
	 * 解压缩zip文件
	 * @param srcFile  需要被解压的文件
	 * @param destFile 被解压后的文件存放的目录
	 * @return 是否解压成功
	 */
	@SuppressWarnings("resource")
	public static boolean unZip(File srcFile, File destFile) {
		boolean res = false;
		if (null == srcFile || null == destFile || !srcFile.exists() || !srcFile.canRead()) {
			return false;
		}
		FileOutputStream fos = null;
		InputStream is = null;
		try {
			ZipFile zipFile = new ZipFile(srcFile);
			Enumeration<? extends ZipEntry> entries = zipFile.entries();
			byte[] buffer = new byte[1024 * 8];
			int len = 0;
			while (entries.hasMoreElements()) {
				ZipEntry zipEntry = entries.nextElement();
				String entryName = zipEntry.getName();
				String unZipFileName = destFile.getAbsolutePath() + File.separator + entryName;
				if (zipEntry.isDirectory()) { // 没有执行此代码
					new File(unZipFileName).mkdirs();
				} else { // 总是执行该代码.因为压缩的时候是对每个文件进行压缩的.
					new File(unZipFileName).getParentFile().mkdirs();
				}
				File unZipedFile = new File(unZipFileName);
				if (unZipedFile.isDirectory()) {
					File[] files = unZipedFile.listFiles();
					for (File file : files) {
						fos = new FileOutputStream(file);
						is = zipFile.getInputStream(zipEntry);
						while ((len = is.read(buffer)) != -1) {
							fos.write(buffer, 0, len);
						}
					}
				} else {
					fos = new FileOutputStream(unZipedFile);
					is = zipFile.getInputStream(zipEntry);
					while ((len = is.read(buffer)) != -1) {
						fos.write(buffer, 0, len);
					}
				}
			}
			fos.flush();
			res = true;
		} catch (ZipException e) {
			LogUtils.e(e);
		} catch (IOException e) {
			LogUtils.e(e);
		} finally {
			IOUtils.close(fos);
			IOUtils.close(is);
		}
		return res;
	}
	
	public static Bitmap decodeFile(File f){
	    Bitmap b = null;
	    try {
	        //Decode image size
//	        BitmapFactory.Options o = new BitmapFactory.Options();
//	        o.inJustDecodeBounds = true;
//
//	        FileInputStream fis = new FileInputStream(f);
//	        BitmapFactory.decodeStream(fis, null, o);
//	        fis.close();

	        int scale = 1;
//	        if (o.outHeight > IMAGE_MAX_SIZE || o.outWidth > IMAGE_MAX_SIZE) {
//	            scale = Math.pow(2, (int) Math.round(Math.log(IMAGE_MAX_SIZE / (double) Math.max(o.outHeight, o.outWidth)) / Math.log(0.5)));
//	        }

	        //Decode with inSampleSize
	        BitmapFactory.Options o2 = new BitmapFactory.Options();
	        o2.inSampleSize = scale;
	        FileInputStream fis = new FileInputStream(f);
	        b = BitmapFactory.decodeStream(fis, null, o2);
	        fis.close();
	    } catch (FileNotFoundException e) {
	    	e.printStackTrace();
	    } catch (IOException e) {
			e.printStackTrace();
		}
	    return b;
	}
	
	public static Bitmap doParse(byte[] data) {
        BitmapFactory.Options decodeOptions = new BitmapFactory.Options();
        Bitmap bitmap = null;
        bitmap = BitmapFactory.decodeByteArray(data, 0, data.length, decodeOptions);


        return bitmap;
    }
}
