/*
 * File Name: ReflectUtils.java 
 */
package net.iyouqu.video.basecommon.utils;

import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

public class ReflectUtils {

	//==========================================================================
	// 方法
	//==========================================================================

	/**
	 * 反射执行静态方法
	 * @param receiverType 执行对象类型
	 * @param methodName   方法名
	 * @param paramTypes   方法参数类型
	 * @param params       方法参数
	 * @return 执行结果
	 */
	public static <T> T invokeStatic(Class<?> receiverType, String methodName, Class<?>[] paramTypes, Object[] params) {
		return invoke(receiverType, methodName, paramTypes, receiverType, params, false);
	}

	/**
	 * 反射执行静态方法
	 * @param receiverType        执行对象类型
	 * @param methodName          方法名
	 * @param paramTypes          方法参数类型
	 * @param params              方法参数
	 * @param requestAccessiblity 是否采用暴力反射
	 * @return 执行结果
	 */
	public static <T> T invokeStatic(Class<?> receiverType, String methodName, Class<?>[] paramTypes, Object[] params, boolean requestAccessiblity) {
		return invoke(receiverType, methodName, paramTypes, receiverType, params, requestAccessiblity);
	}

	/**
	 * 反射执行某方法
	 * @param receiverType 执行对象类型
	 * @param methodName   执行方法名
	 * @param paramTypes   方法参数类型
	 * @param receiver     执行对象
	 * @param params       方法参数
	 * @return 返回执行结果
	 */
	public static <T> T invoke(Class<?> receiverType, String methodName, Class<?>[] paramTypes, Object receiver, Object[] params) {
		return invoke(receiverType, methodName, paramTypes, receiver, params, false);
	}

	/**
	 * 反射执行某方法
	 * @param receiverType        执行对象类型
	 * @param methodName          执行方法名
	 * @param paramTypes          方法参数类型
	 * @param receiver            执行对象
	 * @param params              方法参数
	 * @param requestAccessiblity 是否采用暴力反射
	 * @return 返回执行结果
	 */
	public static <T> T invoke(Class<?> receiverType, String methodName, Class<?>[] paramTypes, Object receiver, Object[] params, boolean requestAccessiblity) {
		T res = null;
		try {
			Method method = receiverType.getDeclaredMethod(methodName, paramTypes);
			if (requestAccessiblity) {
				method.setAccessible(true);
			}
			res = (T) method.invoke(receiver, params);
		} catch (SecurityException e) {
			LogUtils.e(e);
		} catch (NoSuchMethodException e) {
			LogUtils.e(e);
		} catch (IllegalArgumentException e) {
			LogUtils.e(e);
		} catch (IllegalAccessException e) {
			LogUtils.e(e);
		} catch (InvocationTargetException e) {
			LogUtils.e(e);
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return res;
	}

	/**
	 * 反射获取字段值
	 * @param receiverType        执行对象类型
	 * @param fieldName           字段名称
	 * @param receiver            执行对象
	 * @param requestAccessiblity 是否采用暴力反射
	 * @return 返回字段值
	 */
	public static <T> T getValue(Class<?> receiverType, String fieldName, Object receiver, boolean requestAccessiblity) {
		T res = null;
		try {
			Field f = receiverType.getDeclaredField(fieldName);
			if (requestAccessiblity) {
				f.setAccessible(true);
			}
			res = (T) f.get(receiver);
		} catch (SecurityException e) {
			LogUtils.e(e);
		} catch (NoSuchFieldException e) {
			LogUtils.e(e);
		} catch (IllegalArgumentException e) {
			LogUtils.e(e);
		} catch (IllegalAccessException e) {
			LogUtils.e(e);
		}
		return res;
	}

	/**
	 * 反射设置字段值
	 * @param receiverType        执行对象类型
	 * @param fieldName           字段名称
	 * @param receiver            执行对象
	 * @param value               字段设置值
	 * @param requestAccessiblity 是否采用暴力反射
	 */
	public static void setValue(Class<?> receiverType, String fieldName, Object receiver, Object value, boolean requestAccessiblity) {
		try {
			Field f = receiverType.getDeclaredField(fieldName);
			if (requestAccessiblity) {
				f.setAccessible(true);
			}
			f.set(receiver, value);
		} catch (SecurityException e) {
			LogUtils.e(e);
		} catch (NoSuchFieldException e) {
			LogUtils.e(e);
		} catch (IllegalArgumentException e) {
			LogUtils.e(e);
		} catch (IllegalAccessException e) {
			LogUtils.e(e);
		} catch (Throwable e) {
			LogUtils.e(e);
		}
	}
	//==========================================================================
	// 内部类、内部接口
	//==========================================================================
}
