/*
 * File Name: StringUtisl.java 
 * Created on 2014-4-4
 */
package net.iyouqu.video.basecommon.utils;

import android.content.Context;
import android.text.*;
import android.text.style.CharacterStyle;
import android.text.style.ForegroundColorSpan;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Random;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * 字符串操作类 (Description)
 */
public class StringUtils {
	//==========================================================================
	// 静态常亮
	//==========================================================================
	private final static Pattern mEmailPattern = Pattern.compile("\\w+([-+.]\\w+)*@\\w+([-.]\\w+)*\\.\\w+([-.]\\w+)*");
	private final static Pattern mDoublePattern = Pattern.compile("^[-\\+]?[.\\d]*$");
	private final static Pattern mIntegerPattern = Pattern.compile("^[-\\+]?[\\d]*$");
	private final static Pattern mNumberPattern = Pattern.compile("[0-9]*");

	public final static String UTF_8 = "utf-8";

	public static final String EMPTY = "";
	
	/**返回第一个标签*/
	public static String getFirstMatchString(String s){
		if(TextUtils.isEmpty(s)){
			return "";
		}
		try {
			String[] split = s.split(",");
			return TextUtils.isEmpty(split[0]) ? "" : split[0];
		} catch (PatternSyntaxException e) {
			e.printStackTrace();
			return "";
		}
	}

	/**
	 * 返回一个高亮spannable
	 * @param content 文本内容
	 * @param color   高亮颜色
	 * @param start   起始位置
	 * @param end     结束位置
	 * @return 高亮spannable
	 */
	public static CharSequence getHighLightText(String content, int color, int start, int end) {
		if (TextUtils.isEmpty(content)) {
			return "";
		}
		start = start >= 0 ? start : 0;
		end = end <= content.length() ? end : content.length();
		SpannableString spannable = new SpannableString(content);
		CharacterStyle span = new ForegroundColorSpan(color);
		spannable.setSpan(span, start, end, Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
		return spannable;
	}

	/**
	 * 获取链接样式的字符串，即字符串下面有下划线
	 * @param context 资源上下文
	 * @param resId   文字资源
	 * @return 返回链接样式的字符串
	 */
	public static Spanned getHtmlStyleString(Context context, int resId) {
		StringBuilder sb = new StringBuilder();
		sb.append("<a href=\"\"><u><b>").append(context.getString(resId)).append(" </b></u></a>");
		return Html.fromHtml(sb.toString());
	}

	/**
	 * 对指定的字符串进行URL转码
	 * @param src 需要转码的字符串
	 * @param def 转码失败后默认的字符串
	 * @return 转码成功后的字符串
	 */
	public static String getUrlEncodedString(String src, String def) {
		try {
			src = URLEncoder.encode(src, "UTF-8");
		} catch (UnsupportedEncodingException e) {
			LogUtils.e(e);
			src = def;
		}
		return src;
	}
	
	public static int getRandomInt(){
		Random random = new Random();
		random.nextInt(10000);
		return random.nextInt(10000);
	}

	/**
	 * 产生一个随机的字符串
	 * @param length 需要产生的字符串长度
	 * @return 返回一个指定长度的字符串
	 */
	public static String getRandomString(int length) {
		String base = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
		Random random = new Random();
		StringBuilder builder = new StringBuilder();
		for (int i = 0; i < length; i++) {
			int number = random.nextInt(base.length());
			builder.append(base.charAt(number));
		}
		return builder.toString();
	}

	/**
	 * 判断字符串是否有值，如果为null或者是空字符串或者只有空格或者为"null"字符串，则返回true，否则则返回false
	 * @param value 需要判断的字符串
	 * @return 判断结果
	 */
	public static boolean isEmpty(String value) {
		return !(value != null && !"".equalsIgnoreCase(value.trim()) && !"null".equalsIgnoreCase(value.trim()));
	}

	/**
	 * 判断是否为数字
	 * @param str 传入的字符串
	 * @return 是数字返回true, 否则返回false
	 */
	public static boolean isNumber(String str) {
		if (str == null || str.trim().length() == 0) {
			return false;
		}
		return mNumberPattern.matcher(str).matches();
	}

	/**
	 * 判断是否为整数
	 * @param str 传入的字符串
	 * @return 是整数返回true, 否则返回false
	 */
	public static boolean isInteger(String str) {
		if (str == null || str.trim().length() == 0) {
			return false;
		}
		return mIntegerPattern.matcher(str).matches();
	}

	/**
	 * 判断是否为浮点数，包括double和float
	 * @param str 传入的字符串
	 * @return 是浮点数返回true, 否则返回false
	 */
	public static boolean isDouble(String str) {
		if (str == null || str.trim().length() == 0) {
			return false;
		}
		return mDoublePattern.matcher(str).matches();
	}

	/**
	 * 字符串转整数
	 * @param str 需要转换的字符串
	 * @return 转换异常返回 0
	 */
	public static int string2Int(String str) {
		return string2Int(str, 0);
	}

	/**
	 * 字符串转整数
	 * @param str      需要转换的字符串
	 * @param defValue 转换失败后的默认值
	 * @return 转换后的值
	 */
	public static int string2Int(String str, int defValue) {
		try {
			return Integer.parseInt(str);
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return defValue;
	}

	/**
	 * 字符串转长整数
	 * @param str 需要转换的字符串
	 * @return 转换异常返回 0
	 */
	public static long string2Long(String str) {
		return string2Long(str, 0);
	}

	/**
	 * 字符串转长整数
	 * @param str      需要转换的字符串
	 * @param defValue 转换失败后的默认值
	 * @return 转换后的值
	 */
	public static long string2Long(String str, long defValue) {
		try {
			return Long.parseLong(str);
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return defValue;
	}

	/**
	 * 字符串转布尔值
	 * @param b 需要转换的字符串
	 * @return 转换异常返回 false
	 */
	public static boolean toBool(String b) {
		try {
			return Boolean.parseBoolean(b);
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return false;
	}

	/**
	 * 判断是不是一个合法的电子邮件地址
	 */
	public static boolean isEmail(String str) {
		if (str == null || str.trim().length() == 0) {
			return false;
		}
		return mEmailPattern.matcher(str).matches();
	}
	
	//==========================================================================
	// 内部类、内部接口
	//==========================================================================
}
