package net.iyouqu.video.basecommon.utils.http;
import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;
import javax.crypto.KeyGenerator;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

import net.iyouqu.video.basecommon.utils.LogUtils;


import java.io.*;
import java.security.*;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.UUID;
/**
 * File Name: EncryptionUtils.java
 * Create by:Bruce on 2014-12-12 下午5:22:27 
 * @version 1.0.0
 */
public class EncryptionUtils {

	// ==========================================================================
	// BASE64，一种编码方式，把字节转换成64种常见字符
	// ==========================================================================

	/** 用于转换成Base64的字符 */
	private static final byte[] encodingTable = {(byte) 'A', (byte) 'B', (byte) 'C', (byte) 'D', (byte) 'E', (byte) 'F', (byte) 'G', (byte) 'H', (byte) 'I', (byte) 'J', (byte) 'K', (byte) 'L', (byte) 'M', (byte) 'N', (byte) 'O', (byte) 'P', (byte) 'Q', (byte) 'R', (byte) 'S', (byte) 'T', (byte) 'U', (byte) 'V', (byte) 'W', (byte) 'X', (byte) 'Y', (byte) 'Z', (byte) 'a', (byte) 'b', (byte) 'c', (byte) 'd', (byte) 'e', (byte) 'f', (byte) 'g', (byte) 'h', (byte) 'i', (byte) 'j', (byte) 'k', (byte) 'l', (byte) 'm', (byte) 'n', (byte) 'o', (byte) 'p', (byte) 'q', (byte) 'r', (byte) 's', (byte) 't', (byte) 'u', (byte) 'v', (byte) 'w', (byte) 'x', (byte) 'y', (byte) 'z', (byte) '0', (byte) '1', (byte) '2', (byte) '3', (byte) '4', (byte) '5', (byte) '6', (byte) '7', (byte) '8', (byte) '9', (byte) '+', (byte) '/'};

	/**
	 * 把指定的字节数组转换成Base64字节数组
	 * @param data 用于转换的字节数组
	 * @return Base64字节数组
	 */
	private static byte[] encodeBase64Inner(byte[] data) {
		byte[] bytes;
		int modulus = data.length % 3;
		if (modulus == 0) {
			bytes = new byte[4 * data.length / 3];
		} else {
			bytes = new byte[4 * ((data.length / 3) + 1)];
		}
		int dataLength = (data.length - modulus);
		int a1, a2, a3;
		for (int i = 0, j = 0; i < dataLength; i += 3, j += 4) {
			a1 = data[i] & 0xff;
			a2 = data[i + 1] & 0xff;
			a3 = data[i + 2] & 0xff;
			bytes[j] = encodingTable[(a1 >>> 2) & 0x3f];
			bytes[j + 1] = encodingTable[((a1 << 4) | (a2 >>> 4)) & 0x3f];
			bytes[j + 2] = encodingTable[((a2 << 2) | (a3 >>> 6)) & 0x3f];
			bytes[j + 3] = encodingTable[a3 & 0x3f];
		}
		// 处理剩余的部分字节
		int b1, b2, b3;
		int d1, d2;
		switch (modulus) {
			case 0: // 没有剩余部分
				break;
			case 1:
				d1 = data[data.length - 1] & 0xff;
				b1 = (d1 >>> 2) & 0x3f;
				b2 = (d1 << 4) & 0x3f;
				bytes[bytes.length - 4] = encodingTable[b1];
				bytes[bytes.length - 3] = encodingTable[b2];
				bytes[bytes.length - 2] = (byte) '=';
				bytes[bytes.length - 1] = (byte) '=';
				break;
			case 2:
				d1 = data[data.length - 2] & 0xff;
				d2 = data[data.length - 1] & 0xff;
				b1 = (d1 >>> 2) & 0x3f;
				b2 = ((d1 << 4) | (d2 >>> 4)) & 0x3f;
				b3 = (d2 << 2) & 0x3f;
				bytes[bytes.length - 4] = encodingTable[b1];
				bytes[bytes.length - 3] = encodingTable[b2];
				bytes[bytes.length - 2] = encodingTable[b3];
				bytes[bytes.length - 1] = (byte) '=';
				break;
		}
		return bytes;
	}

	/**
	 * 把指定的字节数组转换成Base64字符串
	 * @param data 需要转换的字节数组
	 * @return 转换成Base64的字符串
	 */
	public static String encodeBase64ToString(byte[] data) {
		if (null == data) {
			return null;
		}
		return new String(encodeBase64Inner(data));
	}

	/**
	 * 把指定的字符串转换成Base64字符串
	 * @param data 需要转换的字符串
	 * @return 转换成Base64的字符串
	 */
	public static String encodeToString(String data) {
		if (null == data) {
			return null;
		}
		return encodeBase64ToString(data.getBytes());
	}

	/** 用于解码的字节数组 */
	private static final byte[] decodingTable;

	/** 把用于解码的数组各个角标位填充上合适的字符 */
	static {
		decodingTable = new byte[128];
		for (int i = 0; i < 128; i++) {
			decodingTable[i] = (byte) -1;
		}
		for (int i = 'A'; i <= 'Z'; i++) {
			decodingTable[i] = (byte) (i - 'A');
		}
		for (int i = 'a'; i <= 'z'; i++) {
			decodingTable[i] = (byte) (i - 'a' + 26);
		}
		for (int i = '0'; i <= '9'; i++) {
			decodingTable[i] = (byte) (i - '0' + 52);
		}
		decodingTable['+'] = 62;
		decodingTable['/'] = 63;
	}

	/**
	 * 把Base64字节数组解码成普通字节数组
	 * @param data 需要解密的Base64字节数组
	 * @return 解密后的字节数组
	 */
	public static byte[] decodeBase64(byte[] data) {
		if (null == data) {
			return null;
		}
		byte[] bytes;
		byte b1, b2, b3, b4;
		data = discardNonBase64Bytes(data);
		if (data[data.length - 2] == '=') {
			bytes = new byte[(((data.length / 4) - 1) * 3) + 1];
		} else if (data[data.length - 1] == '=') {
			bytes = new byte[(((data.length / 4) - 1) * 3) + 2];
		} else {
			bytes = new byte[((data.length / 4) * 3)];
		}
		for (int i = 0, j = 0; i < data.length - 4; i += 4, j += 3) {
			b1 = decodingTable[data[i]];
			b2 = decodingTable[data[i + 1]];
			b3 = decodingTable[data[i + 2]];
			b4 = decodingTable[data[i + 3]];

			bytes[j] = (byte) ((b1 << 2) | (b2 >> 4));
			bytes[j + 1] = (byte) ((b2 << 4) | (b3 >> 2));
			bytes[j + 2] = (byte) ((b3 << 6) | b4);
		}
		if (data[data.length - 2] == '=') {
			b1 = decodingTable[data[data.length - 4]];
			b2 = decodingTable[data[data.length - 3]];

			bytes[bytes.length - 1] = (byte) ((b1 << 2) | (b2 >> 4));
		} else if (data[data.length - 1] == '=') {
			b1 = decodingTable[data[data.length - 4]];
			b2 = decodingTable[data[data.length - 3]];
			b3 = decodingTable[data[data.length - 2]];

			bytes[bytes.length - 2] = (byte) ((b1 << 2) | (b2 >> 4));
			bytes[bytes.length - 1] = (byte) ((b2 << 4) | (b3 >> 2));
		} else {
			b1 = decodingTable[data[data.length - 4]];
			b2 = decodingTable[data[data.length - 3]];
			b3 = decodingTable[data[data.length - 2]];
			b4 = decodingTable[data[data.length - 1]];

			bytes[bytes.length - 3] = (byte) ((b1 << 2) | (b2 >> 4));
			bytes[bytes.length - 2] = (byte) ((b2 << 4) | (b3 >> 2));
			bytes[bytes.length - 1] = (byte) ((b3 << 6) | b4);
		}
		return bytes;
	}

	/**
	 * 把Base64字符串解码成普通字节数组
	 * @param data Base64字符串
	 * @return 普通数组
	 */
	public static byte[] decodeBase64(String data) {
		if (null == data) {
			return null;
		}
		byte[] bytes;
		byte b1, b2, b3, b4;
		data = discardNonBase64Chars(data);
		if (data.charAt(data.length() - 2) == '=') {
			bytes = new byte[(((data.length() / 4) - 1) * 3) + 1];
		} else if (data.charAt(data.length() - 1) == '=') {
			bytes = new byte[(((data.length() / 4) - 1) * 3) + 2];
		} else {
			bytes = new byte[((data.length() / 4) * 3)];
		}
		for (int i = 0, j = 0; i < data.length() - 4; i += 4, j += 3) {
			b1 = decodingTable[data.charAt(i)];
			b2 = decodingTable[data.charAt(i + 1)];
			b3 = decodingTable[data.charAt(i + 2)];
			b4 = decodingTable[data.charAt(i + 3)];

			bytes[j] = (byte) ((b1 << 2) | (b2 >> 4));
			bytes[j + 1] = (byte) ((b2 << 4) | (b3 >> 2));
			bytes[j + 2] = (byte) ((b3 << 6) | b4);
		}
		if (data.charAt(data.length() - 2) == '=') {
			b1 = decodingTable[data.charAt(data.length() - 4)];
			b2 = decodingTable[data.charAt(data.length() - 3)];

			bytes[bytes.length - 1] = (byte) ((b1 << 2) | (b2 >> 4));
		} else if (data.charAt(data.length() - 1) == '=') {
			b1 = decodingTable[data.charAt(data.length() - 4)];
			b2 = decodingTable[data.charAt(data.length() - 3)];
			b3 = decodingTable[data.charAt(data.length() - 2)];

			bytes[bytes.length - 2] = (byte) ((b1 << 2) | (b2 >> 4));
			bytes[bytes.length - 1] = (byte) ((b2 << 4) | (b3 >> 2));
		} else {
			b1 = decodingTable[data.charAt(data.length() - 4)];
			b2 = decodingTable[data.charAt(data.length() - 3)];
			b3 = decodingTable[data.charAt(data.length() - 2)];
			b4 = decodingTable[data.charAt(data.length() - 1)];

			bytes[bytes.length - 3] = (byte) ((b1 << 2) | (b2 >> 4));
			bytes[bytes.length - 2] = (byte) ((b2 << 4) | (b3 >> 2));
			bytes[bytes.length - 1] = (byte) ((b3 << 6) | b4);
		}
		return bytes;
	}

	/**
	 * 把Base64字符串解码后按照指定的编码集进行字符串转换
	 * @param data    需要解码的Base64字符串
	 * @param charset 用于编码的编码集
	 * @return 返回的字符串
	 */
	public static String decodeBase64(String data, String charset) {
		if (null == data) {
			return null;
		}
		if (charset == null) {
			return new String(decodeBase64(data));
		}
		try {
			return new String(decodeBase64(data), charset);
		} catch (UnsupportedEncodingException ex) {
			LogUtils.e("Charset: " + charset + " not supported. Using default.");
			return new String(decodeBase64(data));
		}
	}

	/**
	 * 把Base64字节数组解码后按照指定的编码集进行字符串转换
	 * @param data    需要解码的Base64字符串
	 * @param charset 用于编码的编码集
	 * @return 返回的字符串
	 */
	public static String decodeBase64(byte[] data, String charset) {
		if (null == data) {
			return null;
		}
		if (charset == null) {
			return new String(decodeBase64(data));
		}
		try {
			return new String(decodeBase64(data), charset);
		} catch (UnsupportedEncodingException ex) {
			LogUtils.e("Charset: " + charset + " not supported. Using default.");
			return new String(decodeBase64(data));
		}
	}

	/**
	 * 判断字节数组中的字符是否不属于Base64字节，如果不属于，将被舍去
	 * @param data 需要判断的字节数组
	 * @return 过滤后的字节数组
	 */
	private static byte[] discardNonBase64Bytes(byte[] data) {
		byte temp[] = new byte[data.length];
		int bytesCopied = 0;
		for (byte aData : data) {
			if (isValidBase64Byte(aData)) {
				temp[bytesCopied++] = aData;
			}
		}
		byte newData[] = new byte[bytesCopied];
		System.arraycopy(temp, 0, newData, 0, bytesCopied);
		return newData;
	}

	/**
	 * 判断字符串的字符是否不属于Base64字节，如果不属于，将被舍去
	 * @param data 需要判断的字符串
	 * @return 过滤后的字节数组
	 */
	private static String discardNonBase64Chars(String data) {
		StringBuilder sb = new StringBuilder();
		int length = data.length();
		for (int i = 0; i < length; i++) {
			if (isValidBase64Byte((byte) (data.charAt(i)))) {
				sb.append(data.charAt(i));
			}
		}
		return sb.toString();
	}

	/**
	 * 检查给定的byte是否为Base64字节
	 * @param b 给定的byte
	 * @return 是否为Base64字节
	 */
	private static boolean isValidBase64Byte(byte b) {
		if (b == '=') {
			return true;
		} else if (b < 0 || b >= 128) {
			return false;
		} else if (decodingTable[b] == -1) {
			return false;
		}
		return true;
	}

	// ==========================================================================
	// MD5加密，一种不可逆的加密方式，一般用于验证文件是否被篡改或者用于账号密码的记录
	// ==========================================================================

	/**
	 * 对给定的字节数组进行MD5加密
	 * @param data 需要加密的字节数组
	 * @return 加密后的字节数组
	 */
	public static byte[] encodeMD5(byte[] data) {
		byte[] encoded = null;
		try {
			MessageDigest md = MessageDigest.getInstance("MD5");
			encoded = md.digest(data);
		} catch (NoSuchAlgorithmException e) {
			LogUtils.e(e);
		}
		return encoded;
	}

	/**
	 * 对给定的字符串进行MD5加密
	 * @param str 需要加密的字符串
	 * @return 加密后的字符串
	 */
	public static String encodeMD5ToString(String str) {
		byte[] encoded = encodeMD5(str.getBytes());
		if (null == encoded) {
			return null;
		} else {
			StringBuilder builder = new StringBuilder();
			for (byte anEncoded : encoded) {
				String hex = Integer.toHexString(0xFF & anEncoded);
				if (hex.length() == 1) {
					builder.append('0').append(hex);
				} else {
					builder.append(hex);
				}
			}
			return builder.toString();
		}
	}

	/**
	 * 获取指定文件的MD5值
	 * @param f 指定的文件
	 * @return 获取的MD5字节数组
	 */
	public static byte[] encodeMD5(File f) {
		FileInputStream fis = null;
		byte[] encoded = null;
		try {
			fis = new FileInputStream(f);
			MessageDigest md = MessageDigest.getInstance("MD5");
			byte[] buffer = new byte[1024];
			int numRead;
			while ((numRead = fis.read(buffer)) > 0) {
				md.update(buffer, 0, numRead);
			}
			encoded = md.digest();
		} catch (Exception e) {
			LogUtils.e(e);
		} finally {
			if (fis != null) {
				try {
					fis.close();
				} catch (IOException e) {
					LogUtils.e(e);
				}
				fis = null;
			}
		}
		return encoded;
	}

	/**
	 * 获取指定文件的MD5值
	 * @param f 指定的文件
	 * @return 获取的MD5字符串
	 */
	public static String encodeMD5ToString(File f) {
		byte[] encoded = encodeMD5(f);
		if (null == encoded) {
			return null;
		} else {
			StringBuilder builder = new StringBuilder();
			for (byte anEncoded : encoded) {
				String hex = Integer.toHexString(0xFF & anEncoded);
				if (hex.length() == 1) {
					builder.append('0').append(hex);
				} else {
					builder.append(hex);
				}
			}
			return builder.toString();
		}
	}

	// ==========================================================================
	// MD5加密，和MD5一样，为不可逆加密
	// ==========================================================================

	/**
	 * 对给定的字节数组进行SHA加密
	 * @param data 需要加密的字节数组
	 * @return 加密后的字节数组
	 */
	public static byte[] encodeSHA(byte[] data) {
		byte[] encoded = null;
		try {
			MessageDigest md = MessageDigest.getInstance("SHA");
			encoded = md.digest(data);
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return encoded;
	}

	/**
	 * 对给定的字符串进行SHA加密
	 * @param str 需要加密的字符串
	 * @return 加密后的字符串
	 */
	public static String encodeSHAToString(String str) {
		byte[] encoded = encodeSHA(str.getBytes());
		if (null == encoded) {
			return null;
		} else {
			StringBuilder builder = new StringBuilder();
			for (byte anEncoded : encoded) {
				String hex = Integer.toHexString(0xFF & anEncoded);
				if (hex.length() == 1) {
					builder.append('0').append(hex);
				} else {
					builder.append(hex);
				}
			}
			return builder.toString();
		}
	}

	/**
	 * 获取指定文件的SHA值
	 * @param f 指定的文件
	 * @return 获取的SHA字节数组
	 */
	public static byte[] encodeSHA(File f) {
		FileInputStream fis = null;
		byte[] encoded = null;
		try {
			fis = new FileInputStream(f);
			MessageDigest md = MessageDigest.getInstance("SHA");
			byte[] buffer = new byte[1024];
			int numRead = 0;
			while ((numRead = fis.read(buffer)) > 0) {
				md.update(buffer, 0, numRead);
			}
			encoded = md.digest();
		} catch (Exception e) {
			LogUtils.e(e);
		} finally {
			if (fis != null) {
				try {
					fis.close();
				} catch (IOException e) {
					LogUtils.e(e);
				}
				fis = null;
			}
		}
		return encoded;
	}

	/**
	 * 获取指定文件的SHA值
	 * @param f 指定的文件
	 * @return 获取的SHA字符串
	 */
	public static String encodeSHAToString(File f) {
		byte[] encoded = encodeSHA(f);
		if (null == encoded) {
			return null;
		} else {
			StringBuilder builder = new StringBuilder();
			for (byte anEncoded : encoded) {
				String hex = Integer.toHexString(0xFF & anEncoded);
				if (hex.length() == 1) {
					builder.append('0').append(hex);
				} else {
					builder.append(hex);
				}
			}
			return builder.toString();
		}
	}

	// ==========================================================================
	// DES加密，使用秘钥进行加密解密，是一种对称加密解密方式
	// ==========================================================================

	/**
	 * DES加密
	 * @param text   需要加密的字符串
	 * @param keyStr 加密的key
	 * @return 加密后的字符串
	 */
	public static String encodeDES(String text, String keyStr) {
		try {
			Cipher cipher = getDESCipher(keyStr, Cipher.ENCRYPT_MODE);
			byte[] byteMing = text.getBytes("UTF8");
			byte[] byteMi = cipher.doFinal(byteMing);
			return encodeBase64ToString(byteMi);
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return null;
	}

	/**
	 * DES解密
	 * @param text   需要解密的字符串
	 * @param keyStr 解密的key
	 */
	public static String decodeDES(String text, String keyStr) {
		try {
			Cipher cipher = getDESCipher(keyStr, Cipher.DECRYPT_MODE);
			byte[] byteMi = decodeBase64(text);
			byte[] byteMing = cipher.doFinal(byteMi);
			return new String(byteMing, "UTF8");
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return null;
	}

	/**
	 * DES加密文件
	 * @param srcPath 需要加密的文件路径
	 * @param desPath 加密后的文件路径
	 * @param keyStr  加密的key
	 */
	public void encodeFileDES(String srcPath, String desPath, String keyStr) {
		codeFileDES(srcPath, desPath, keyStr, Cipher.ENCRYPT_MODE);
	}

	/**
	 * DES解密文件
	 * @param srcPath 需要解密的文件路径
	 * @param desPath 解密后的文件路径
	 * @param keyStr  解密的key
	 */
	public void decodeFileDES(String srcPath, String desPath, String keyStr) {
		codeFileDES(srcPath, desPath, keyStr, Cipher.DECRYPT_MODE);
	}

	private void codeFileDES(String srcPath, String desPath, String keyStr, int mode) {
		Cipher cipher = getDESCipher(keyStr, mode);
		InputStream is = null;
		OutputStream out = null;
		CipherInputStream cis = null;
		try {
			is = new FileInputStream(srcPath);
			out = new FileOutputStream(desPath);
			cis = new CipherInputStream(is, cipher);
			byte[] buffer = new byte[1024];
			int r;
			while ((r = cis.read(buffer)) > 0) {
				out.write(buffer, 0, r);
			}
		} catch (Exception e) {
			LogUtils.e(e);
		} finally {
			try {
				if (is != null) {
					is.close();
					is = null;
				}
				if (out != null) {
					out.close();
					out = null;
				}
				if (cis != null) {
					cis.close();
					cis = null;
				}
			} catch (Exception e) {
				LogUtils.e(e);
			}
		}
	}

	private static Cipher getDESCipher(String keyStr, int mode) {
		Cipher cipher = null;
		try {
			KeyGenerator generator = KeyGenerator.getInstance("DES");
			generator.init(new SecureRandom(keyStr.getBytes()));
			Key key = generator.generateKey();
			cipher = Cipher.getInstance("DES");
			cipher.init(mode, key);
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return cipher;
	}

	// ==========================================================================
	// RSA加密，非对称加密，如果使用私钥加密，则需要使用公钥解密，如果使用公钥加密，则需要使用私钥解密
	// 也可以用来对文件生成签名，使用私钥签名，通过公钥验证
	// ==========================================================================

	/**
	 * RSA私钥加密
	 * @param text   需要加密的字符串
	 * @param keyStr 加密的key
	 * @return 加密后的字符串
	 */
	public static String encodeRSAByPrivate(String text, String keyStr) {
		return encodeRSA(text, keyStr, 0);
	}

	/**
	 * RSA公钥加密
	 * @param text   需要加密的字符串
	 * @param keyStr 加密的key
	 * @return 加密后的字符串
	 */
	public static String encodeRSAByPublic(String text, String keyStr) {
		return encodeRSA(text, keyStr, 1);
	}

	private static String encodeRSA(String text, String keyStr, int keyMode) {
		try {
			Cipher cipher = getRSACipher(keyStr, Cipher.ENCRYPT_MODE, keyMode);
			byte[] byteMing = text.getBytes("UTF8");
			byte[] byteMi = cipher.doFinal(byteMing);
			return encodeBase64ToString(byteMi);
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return null;
	}

	/**
	 * RSA私钥解密
	 * @param text   需要解密的字符串
	 * @param keyStr 解密的key
	 */
	public static String decodeRSAByPrivate(String text, String keyStr) {
		return decodeRSA(text, keyStr, 0);
	}

	/**
	 * RSA公钥解密
	 * @param text   需要解密的字符串
	 * @param keyStr 解密的key
	 */
	public static String decodeRSAByPublic(String text, String keyStr) {
		return decodeRSA(text, keyStr, 1);
	}

	private static String decodeRSA(String text, String keyStr, int keyMode) {
		try {
			Cipher cipher = getRSACipher(keyStr, Cipher.DECRYPT_MODE, keyMode);
			byte[] byteMi = decodeBase64(text);
			byte[] byteMing = cipher.doFinal(byteMi);
			return new String(byteMing, "UTF8");
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return null;
	}

	/**
	 * RSA私钥签名
	 * @param data       需要生成签名的数据
	 * @param privateKey 生成签名的私钥
	 */
	public static String createSign(byte[] data, String privateKey) {
		String sign = null;
		try {
			PrivateKey key = createPrivateKey(privateKey);
			// 用私钥对信息生成数字签名
			Signature signature = Signature.getInstance("MD5withRSA");
			signature.initSign(key);
			signature.update(data);
			sign = encodeBase64ToString(signature.sign());
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return sign;
	}

	/**
	 * RSA公钥检查签名
	 * @param data      需要检查签名的数据
	 * @param publicKey 检查签名的公钥
	 * @param sign      签名
	 */
	public static boolean checkSign(byte[] data, String publicKey, String sign) {
		boolean result = false;
		try {
			PublicKey key = createPublicKey(publicKey);
			Signature signature = Signature.getInstance("MD5withRSA");
			signature.initVerify(key);
			signature.update(data);
			// 验证签名是否正常
			result = signature.verify(decodeBase64(sign));
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return result;
	}

	private static Cipher getRSACipher(String keyStr, int mode, int keyMode) {
		Cipher cipher = null;
		try {
			cipher = Cipher.getInstance("RSA");
			if (keyMode == 0) {// 秘钥
				cipher.init(mode, createPrivateKey(keyStr));
			} else {// 公钥
				cipher.init(mode, createPublicKey(keyStr));
			}
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return cipher;
	}

	private static PrivateKey createPrivateKey(String key) {
		PrivateKey privateKey = null;
		try {
			byte[] keyBytes = decodeBase64(key);// 把字符串秘钥还原成字节数组
			// 取得私钥
			PKCS8EncodedKeySpec pkcs8KeySpec = new PKCS8EncodedKeySpec(keyBytes);
			KeyFactory keyFactory = KeyFactory.getInstance("RSA");
			privateKey = keyFactory.generatePrivate(pkcs8KeySpec);
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return privateKey;
	}

	private static PublicKey createPublicKey(String key) {
		PublicKey publicKey = null;
		try {
			byte[] keyBytes = decodeBase64(key);// 把字符串秘钥还原成字节数组
			// 取得公钥
			X509EncodedKeySpec x509KeySpec = new X509EncodedKeySpec(keyBytes);
			KeyFactory keyFactory = KeyFactory.getInstance("RSA");
			publicKey = keyFactory.generatePublic(x509KeySpec);
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return publicKey;
	}

	/** 生成一对RSA秘钥 */
	public static String[] createKeys() {
		try {
			KeyPairGenerator keyGen = KeyPairGenerator.getInstance("RSA");
			keyGen.initialize(1024);// 设置密钥长度
			KeyPair keyPair = keyGen.generateKeyPair();
			String publicKeyStr = encodeBase64ToString(keyPair.getPublic().getEncoded());
			String privateKeyStr = encodeBase64ToString(keyPair.getPublic().getEncoded());
			return new String[]{publicKeyStr, privateKeyStr};
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return null;
	}

	/** AES加密，可以对任意长度进行一次加密 */
	public static byte[] encryptAES(byte[] content, byte[] key) {
		int count = (content.length + 15) % 16;
		byte[] newContent = new byte[count * 16];
		for (int i = 0; i < content.length; i++) {
			newContent[i] = content[i];
		}
		Cipher cipher;
		byte[] encrypted = null;
		try {
			SecretKeySpec skeySpec = new SecretKeySpec(key, "AES");
			IvParameterSpec ivSpec = new IvParameterSpec(new byte[16]);
			cipher = Cipher.getInstance("AES/CBC/NoPadding");
			cipher.init(Cipher.ENCRYPT_MODE, skeySpec, ivSpec);
			encrypted = cipher.doFinal(newContent);
		} catch (Exception e) {
			LogUtils.e(e);
		}
		return encrypted;
	}

	/** AES加密，如果数据长度大于16，则会拆分成多次加密 */
	public static byte[] encryptAES2(byte[] content, byte[] key) {
		int count = (content.length + 15) % 16;
		if (count == 1) {
			return encryptAES(content, key);
		} else {
			byte[] result = new byte[16 * count];
			for (int i = 0; i < count; i++) {
				byte[] bytes = new byte[16];
				if (i != count - 1) {
					System.arraycopy(content, i * 16, bytes, 0, 16);
				} else {
					System.arraycopy(content, i * 16, bytes, 0, content.length - 16 * i);
				}
				byte[] encrypt = encryptAES(bytes, key);
				System.arraycopy(encrypt, 0, result, i * 16, 16);
			}
			return result;
		}
	}

	// ==========================================================================
	// UUID 全局唯一标示，会根据时间戳以及硬件信息生成一个唯一的标记，理论上生成的两个UUID都不会重复
	// ==========================================================================
	public static String createUUID() {
		return UUID.randomUUID().toString();
	}

	public static void main(String[] args) {
		byte[] b = {(byte)161,(byte)226,(byte)157};
		byte[] bytes = encodeBase64Inner(b);
		for (int i = 0; i < bytes.length; i++) {
			byte aByte = bytes[i];
		}
	}
}
