package net.iyouqu.video.basecommon.utils.ui;

import android.content.Context;
import android.content.res.ColorStateList;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Handler;
import android.os.Looper;
import android.os.Process;
import android.util.DisplayMetrics;
import android.util.TypedValue;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.Toast;

import net.iyouqu.video.basecommon.BaseActivity;
import net.iyouqu.video.basecommon.manager.EsBaseApplication;

/**
 * File Name: UIUtils.java
 * Create by:Bruce on 2014-12-12 下午4:40:08 
 * @version 1.0.0
 */
public class UIUtils {
	
	public static int dipToPx(Context c,float dipValue) {
		DisplayMetrics metrics = c.getResources().getDisplayMetrics();
		return (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, dipValue, metrics);
	}
	
	public static int spToPx(Context context, float spValue) {  
		DisplayMetrics metrics = context.getResources().getDisplayMetrics();
		return (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_SP, spValue, metrics);
    }
	
	/** 获取Context */
	public static Context getContext() {
		return EsBaseApplication.getApplication();
	}

	/** 获取主线程id */
	public static int getMainThreadId() {
		return EsBaseApplication.getMainThreadId();
	}

	/** 获取主线程 */
	public static Thread getMainThread() {
		return EsBaseApplication.getMainThread();
	}

	/** 判断是否运行在主线程 */
	public static boolean isRunInMainThread() {
		return Thread.currentThread().getId() == getMainThreadId();
	}

	/** 获取主线程的handler */
	public static Handler getMainThreadHandler() {
		return EsBaseApplication.getMainThreadHandler();
	}

	/** 获取主线程的looper */
	public static Looper getMainThreadLooper() {
		return EsBaseApplication.getMainThreadLooper();
	}

	/** 把任务post主线程中 */
	public static boolean post(Runnable runnable) {
		boolean result = false;
		Handler handler = getMainThreadHandler();
		if (handler != null) {
			result = handler.post(runnable);
		}
		return result;
	}

	/** 把任务延迟post到主线程中 */
	public static boolean postDelayed(Runnable runnable, long delay) {
		boolean result = false;
		Handler handler = getMainThreadHandler();
		if (handler != null) {
			result = handler.postDelayed(runnable, delay);
		}
		return result;
	}

	/** 把任务post到主线程的消息队列最前面 */
	public static boolean postAtFrontOfQueue(Runnable runnable) {
		boolean result = false;
		Handler handler = getMainThreadHandler();
		if (handler != null) {
			result = handler.postAtFrontOfQueue(runnable);
		}
		return result;
	}

	/** 把任务从主线程的消息队列中删除 */
	public static void removeCallbacks(Runnable runnable) {
		Handler handler = getMainThreadHandler();
		if (handler != null) {
			handler.removeCallbacks(runnable);
		}
	}

	/** 在主线程执行 */
	public static void runInMainThread(Runnable runnable) {
		if (isRunInMainThread()) {
			runnable.run();
		} else {
			post(runnable);
		}
	}

	/** 获取应用资源对象 */
	public static Resources getResources() {
		if (getContext() != null) {
			return getContext().getResources();
		} else {
			return null;
		}
	}

	/** dip转换px */
	public static int dip2px(int dip) {
		if (getResources() != null) {
			final float scale = getResources().getDisplayMetrics().density;
			return (int) (dip * scale + 0.5f);
		} else {
			return 0;
		}
	}

	/** pxz转换dip */
	public static int px2dip(int px) {
		if (getResources() != null) {
			final float scale = getResources().getDisplayMetrics().density;
			return (int) (px / scale + 0.5f);
		} else {
			return 0;
		}
	}

	/** 获取dimen值 */
	public static int getDimens(int resId) {
		if (getResources() != null) {
			return getResources().getDimensionPixelSize(resId);
		} else {
			return 0;
		}
	}

	/** 获取图片 */
//	public static Drawable getDrawable(int resId) {
//		if (getResources() != null) {
//			return getResources().getDrawable(resId);
//		} else {
//			return null;
//		}
//	}
	
	/**获取图片*/
	public static Bitmap getBitmap(int resId){
		if(getResources() != null){
			return BitmapFactory.decodeResource(getResources(), resId);
		}else{
			return null;
		}
	}

	/** 获取颜色 */
	public static int getColor(int resId) {
		if (getResources() != null) {
			return getResources().getColor(resId);
		} else {
			return 0;
		}
	}

	/** 获取颜色选择器 */
	public static ColorStateList getColorStateList(int resId) {
		if (getResources() != null) {
			return getResources().getColorStateList(resId);
		} else {
			return null;
		}
	}

	/** 获取文字 */
	public static String getString(int resId) {
		if (getResources() != null) {
			return getResources().getString(resId);
		} else {
			return null;
		}
	}

	/** 获取文字，并按照后面的参数进行格式化 */
	public static String getString(int resId, Object... formatAgrs) {
		if (getResources() != null) {
			return getResources().getString(resId, formatAgrs);
		} else {
			return null;
		}
	}

	/**
	 * 根据指定的layout索引，创建一个View
	 * @param resId 指定的layout索引
	 * @return 新的View
	 */
	public static View inflate(int resId) {
		Context context = UIUtils.getContext();
		if (context != null) {
			return LayoutInflater.from(context).inflate(resId, null);
		}
		return null;
	}

	/**
	 * 对toast的简易封装。线程安全，可以在非UI线程调用。
	 * @param resId    Toast内容的资源id
	 * @param duration Toast的持续时间
	 */
	public static void showToastSafe(final int resId, final int duration) {
		if (Process.myTid() == getMainThreadId()) {
			// 调用在UI线程
			if (BaseActivity.getForegroundActivity() != null) {
				Toast.makeText(BaseActivity.getForegroundActivity(), resId, duration).show();
			}
		} else {
			// 调用在非UI线程
			post(new Runnable() {
				@Override
				public void run() {
					if (BaseActivity.getForegroundActivity() != null) {
						Toast.makeText(BaseActivity.getForegroundActivity(), resId, duration).show();
					}
				}
			});
		}
	}

	/**
	 * 对toast的简易封装。线程安全，可以在非UI线程调用。
	 * @param text     Toast内容
	 * @param duration Toast的持续时间
	 */
	public static void showToastSafe(final CharSequence text, final int duration) {
		if (Process.myTid() == getMainThreadId()) {
			// 调用在UI线程
			if (BaseActivity.getForegroundActivity() != null) {
				Toast.makeText(BaseActivity.getForegroundActivity(), text, duration).show();
			}
		} else {
			// 调用在非UI线程
			post(new Runnable() {
				@Override
				public void run() {
					if (BaseActivity.getForegroundActivity() != null) {
						Toast.makeText(BaseActivity.getForegroundActivity(), text, duration).show();
					}
				}
			});
		}
	}
}
