/**
 * Finds an eligible loop from the given boolean road grid
 * Always returns an array - may be empty if no path found
 */
const RACE_PATH_MIN_LENGTH = 40;
const RACE_PATH_VERY_MIN_LENGTH = 5;

// Make TS happy
declare function postMessage(msg);

// returns possible starting positions
let findStartPoints = (booleanGrid:[number, number][],
                      townHallIndex:[number, number]): [number,number][] => {
    // Look for all eligible road tiles around town hall, prefer the tile with lowest distance from base tile
    let baseRow = townHallIndex[0];
    let baseCol = townHallIndex[1];
    let gridSize = booleanGrid.length;
    let foundPoints = {};

    let eligibleIndices: [number,number][] = [];
    let tlR = Math.max(baseRow - 2, 0);
    let tlC = Math.max(baseCol - 2, 0);
    let trC = Math.min(gridSize - 1, baseCol + 1);
    let blR = Math.min(gridSize - 1, baseRow + 1);

    let checkAndAdd = (r, c) => {
        let k = r+"_"+c;
        if (booleanGrid[r][c] && !foundPoints[k]){
            foundPoints[k] = 1;
            eligibleIndices.push([r,c]);
        }
    };

    let col, row;
    // left to right
    for (col = tlC; col<=trC; col++){
        checkAndAdd(tlR, col);
        if (tlR !== blR) {
            checkAndAdd(blR, col);
        }
    }
    // top to bottom
    if (tlR + 1 <= blR - 1) {
        for (row = tlR + 1; row <= blR - 1; row++) {
            checkAndAdd(row, tlC);
            if (tlC !== trC) {
                checkAndAdd(row, trC);
            }
        }
    }

    let preferred: [number, number] = [baseRow + 1, baseCol];
    if (eligibleIndices.filter(i => {
            return i[0] === preferred[0] &&
                i[1] === preferred[1];
        }).length){
        eligibleIndices.unshift(preferred);
    }
    return eligibleIndices;
};

// track marked roads + spawns ever
let globalMarked = [];

// main loop finder
let findRacePath = (booleanGrid:[number, number][],
                    townHallPoint:[number, number]): [number, number][] => {
    // initialize global mark tracker
    for (let i = 0; i < booleanGrid.length; i++){
        globalMarked[i] = [];
    }

    // flood fill from starting point until loop found!
    let possibleStarts:[number,number][] = findStartPoints(booleanGrid, townHallPoint);

    while (possibleStarts.length){
        let start = possibleStarts.shift();
        if (!globalMarked[start[0]][start[1]]){
            let path = findRacePathFromStart(booleanGrid, start);
            if (path.length) return path;
        }
    }

    return [];
};

interface pathLink{
    index: [number, number],
    prev: pathLink
}
let findRacePathFromStart = (booleanGrid:[number, number][], startRoadTile:[number,number]):[number,number][] => {
    let paths:pathLink[] = [];
    let marked:pathLink[][] = [];
    for (let i = 0; i < booleanGrid.length; i++){
        marked[i] = [];
    }

    // floodfill from startRoadTile
    let queue: [number, number,number][] = [[startRoadTile[0], startRoadTile[1], 0]];
    marked[startRoadTile[0]][startRoadTile[1]] = {
        index: [startRoadTile[0], startRoadTile[1]],
        prev: null
    };

    let lastLink = null;
    while (queue.length){
        let index = queue.shift();
        let depth = index[2];
        mapTouchingTiles(index[0], index[1], booleanGrid.length, (r, c) => {
            if (!booleanGrid[r][c]){
                return;
            }
            globalMarked[r][c] = 1; // for starting points
            if (marked[r][c]){
                return;
            }
            let link: pathLink = {
                index: [r,c],
                prev: marked[index[0]][index[1]]
            };
            lastLink = link;
            marked[r][c] = link;
            if (depth >= RACE_PATH_MIN_LENGTH){
                paths.push(link);
                return;
            }
            queue.push([r,c, depth+1]);
        });
    }

    if (!paths.length){
        // no suitable length path found, just use closest
        paths = [lastLink];
    }

    // convert loops from linked list to array
    let loopsConverted = [];
    paths.forEach((node:pathLink) => {
        let fullPath = [];
        while(node){
            fullPath.unshift([node.index[0], node.index[1]]);
            node = node.prev;
        }
        if (fullPath.length > RACE_PATH_VERY_MIN_LENGTH){
            loopsConverted.push(fullPath);
        }
    });

    // todo: decide which path to use (best fit for min path and min laps)

    if (loopsConverted.length){
        return randomChoice(loopsConverted);
    } else {
        return [];
    }
};

function randomChoice(arr){
    if (arr.length === 0){
        return null;
    }
    return arr[randomInRange(0, arr.length - 1)];
}
function randomInRange(min, max){
    return Math.min(min + Math.floor(Math.random() * (max - min + 1)), max);
}
function mapTouchingTiles(baseRow, baseCol, size, fn: (newRow:number, newCol:number) => any, allowInvalidIndex=false){
    [[0, -1], [0, 1], [1, 0], [-1, 0]].map(function(d){
        let newRow = baseRow + d[0];
        let newCol = baseCol +d[1];
        if ((newRow >= 0 && newRow < size && newCol >= 0 && newCol < size) ||
            allowInvalidIndex) {
            fn(newRow, newCol);
        }
    });
}

onmessage = function(e) {
    let obj = JSON.parse(e.data);
    postMessage(JSON.stringify(findRacePath(obj.booleanGrid, obj.startPoint)));
};
