package com.android.bitmapfun.util;

import java.io.File;
import java.io.FileDescriptor;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.SocketTimeoutException;
import java.net.URL;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.drawable.BitmapDrawable;
import android.provider.MediaStore;
import android.text.TextUtils;
import android.util.Log;

import com.android.bitmapfun.util.ImageCache.ImageCacheParams;
import com.pingan.frame.http.listener.TokenCallback;
import com.pingan.frame.util.AppLog;
import com.pingan.frame.util.BitmapUtils;

public class PAImageFetcher extends ImageFetcher {

	// Set memory cache to 25% of app memory
    private static final float MEM_CACHE_SIZE_PERCENT = 0.25f;
    private static final String IMAGE_HEAD_ICON_CACHE_DIR = "thumbs";
    private static final String TAG = "PAImageFetcher";

    private ImageCacheParams headIconCacheParams;

    private static PAImageFetcher mPAImageFetcher;

    private TokenCallback mTokenCallback;

    private PAImageFetcher(Context context, TokenCallback tokenCallback) {
        super(context);
        mTokenCallback = tokenCallback;
        headIconCacheParams = new ImageCacheParams(context, IMAGE_HEAD_ICON_CACHE_DIR);
        headIconCacheParams.setMemCacheSizePercent(MEM_CACHE_SIZE_PERCENT);
        addImageCache(headIconCacheParams);
    }

    public static void init(Context context, TokenCallback tokenCallback) {
        if (mPAImageFetcher == null) {
            mPAImageFetcher = new PAImageFetcher(context, tokenCallback);
        }
    }

    public static PAImageFetcher getInstance() {
        return mPAImageFetcher;
    }

    /**
     * 获取网络图片的本地保存地址
     */
    public String getLocalPathByUrl(String url) {
        if (TextUtils.isEmpty(url)) {
            return null;
        }
        String key = ImageCache.hashKeyForDisk(url);
        File file = new File(mHttpDiskCache.getDirectory(), key + "." + DISK_CACHE_INDEX);
        if (file != null && file.exists()) {
            return file.getAbsolutePath();
        }
        return null;
    }

    /**
     * 获取缩略图
     */
    public Bitmap getThumbsBitmap(LoadImage data) {
        BitmapDrawable drawable = mImageCache.getBitmapFromMemCache(data.getCacheKey());
        if (drawable != null) {
            return drawable.getBitmap();
        }
        Bitmap bitmap = mImageCache.getBitmapFromDiskCache(data.getCacheKey());
        if (bitmap != null) {
            mImageCache.addBitmapToMemCache(data.getCacheKey(), new RecyclingBitmapDrawable(mResources, bitmap));
        }
        return bitmap;
    }

    @Override
    protected Bitmap processBitmap(LoadImage data, Callback callback) {
    	Bitmap bitmap = null;
    	try{
            synchronized (mHttpDiskCacheLock) {
                // Wait for disk cache to initialize
                while (mHttpDiskCacheStarting) {
                    try {
                        mHttpDiskCacheLock.wait();
                    } catch (InterruptedException e) {
                    }
                }
                AppLog.d(TAG, "imagecache manager 正在从大图缓存中获取图片");
                // 优先从SDCARD 缓存中取
                bitmap = processLocalCache(data);
                // 如果从缓存中取不到
                if (bitmap == null || bitmap.isRecycled()) {
                    AppLog.d(TAG, "imagecache manager 大图缓存中获取图片失败");
                    // 如果是加载的本地图片
                    if (data instanceof LoadImageFile) {
                        bitmap = processFileBitmap((LoadImageFile) data);
                    } 
                    else if(data instanceof LoadImageThumb){
                    	bitmap = processThumbBitmap((LoadImageThumb)data);
                    }
                    else {
                        // 如果加载的是网络图片
                        // 需要下载，才从服务器下载
                        if (data.isDownloadAvailable()) {
                            bitmap = processUrlBitmap((LoadImageUrl) data, callback);
                        } else {
                            AppLog.d(TAG, "imagecache manager 获取网络图片失败，未开启网络下载");
                        }
                    }
                }
            }
    	}catch(Exception ex){
    		ex.printStackTrace();
    	}catch (OutOfMemoryError e) {
    		e.printStackTrace();
		}
        return bitmap;
    }

    @Override
	protected boolean hasLocalCache(LoadImageUrl data) {
    	try {
			if(mHttpDiskCache!=null&&!mHttpDiskCacheStarting){
				String cacheKey = data.getSourceCacheKey();
		    	final String key = ImageCache.hashKeyForDisk(cacheKey);
				final DiskLruCache.Snapshot snapshot = mHttpDiskCache.get(key);
				if(snapshot!=null){
					AppLog.d(TAG, "imagecache manager 查找：在从大图缓存中存在缓存");
					return true;
				}
			}
    	} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
    	AppLog.d(TAG, "imagecache manager 查找：在从大图缓存中没有缓存");
    	return false;
	}
    
    /**
     * 根据图片文件获取bitmap
     * 
     * @param data 请求资源信息
     * @return 图片
     */
    private Bitmap processThumbBitmap(LoadImageThumb data) {
    	Bitmap bitmap = MediaStore.Images.Thumbnails.getThumbnail(data.getApplication().getContentResolver(),
    			data.getSourceId(), data.getTyle(), null);
    	bitmap = BitmapUtils.changeBitmapSize(data.getWidth(), data.getHeight(), bitmap);
        return bitmap;
    }
    
    /**
     * 从SDCARD缓存中取图片
     * 
     * @param data 资源信息
     * @return 图片
     */
    private Bitmap processLocalCache(LoadImage data) {
        FileInputStream fileInputStream = null;
        if (mHttpDiskCache != null) {
            try {
                DiskLruCache.Snapshot snapshot = mHttpDiskCache
                        .get(ImageCache.hashKeyForDisk(data.getSourceCacheKey()));
                // 如果缓存中存在相应图片
                if (snapshot != null) {
                    fileInputStream = (FileInputStream) snapshot.getInputStream(DISK_CACHE_INDEX);

                    FileDescriptor fileDescriptor = fileInputStream.getFD();
                    return decodeSampledBitmapFromDescriptor(fileDescriptor, data.getWidth(), data.getHeight(),
                            getImageCache());
                }
            } catch (Exception e) {
                AppLog.e(TAG, "httpFrame processLocalCache  获取缓存出错");
                Log.e(TAG, "Failed to processLocalCache", e);
            } finally {
                if (fileInputStream != null) {
                    try {
                        fileInputStream.close();
                    } catch (IOException e) {
                    }
                }
            }
        }
        return null;
    }

    /**
     * 根据图片文件获取bitmap
     * 
     * @param data 请求资源信息
     * @return 图片
     */
    private Bitmap processFileBitmap(LoadImageFile data) {
        String localPath = data.getLocalPath();
        AppLog.d(TAG, "imagecache manager 正在从本地获取图片  localPath：" + localPath);
        File file = new File(localPath);
        Bitmap bitmap = null;
        // 如果文件存在
        if (file.isFile()) {
            bitmap = saveFileBimtapToCache(data, file);
            
            if (bitmap == null || bitmap.isRecycled()) {
                bitmap = getFileBitmap(data, file);
            }
            
            int angle = BitmapUtils.readPictureDegree(localPath);
            if(angle>0){
            	bitmap = BitmapUtils.rotaingImageView(angle, bitmap);
			}
        }
        return bitmap;
    }

    /**
     * 将图片文件 移动到缓存文件夹中 并且返回用户所需要的bitmap
     * 
     * @param data 请求资源信息
     * @param file 图片文件
     * @return bitmap
     */
    private Bitmap saveFileBimtapToCache(LoadImageFile data, File file) {
        FileInputStream fileInputStream = null;
        try {
            String key = ImageCache.hashKeyForDisk(data.getSourceCacheKey());
            // 如果需要将此文件移动到缓存文件夹中 并且
            if (data.isCacheStoregeAvailable()) {
                DiskLruCache.Editor editor = mHttpDiskCache.edit(key);
                if (editor != null) {
                    if (fileToStream(file, editor.newOutputStream(DISK_CACHE_INDEX))) {
                        editor.commit();
                    } else {
                        editor.abort();
                    }
                }
                DiskLruCache.Snapshot snapshot = mHttpDiskCache.get(key);

                fileInputStream = (FileInputStream) snapshot.getInputStream(DISK_CACHE_INDEX);
                FileDescriptor fileDescriptor = fileInputStream.getFD();
                return decodeSampledBitmapFromDescriptor(fileDescriptor, data.getWidth(), data.getHeight(),
                        getImageCache());
            }
        } catch (Exception ex) {
            AppLog.e(TAG, "httpFrame saveFileBimtapToCache  获取本地图片出错");
            Log.e(TAG, "Failed to saveFileBimtapToCache", ex);
        } finally {
            if (fileInputStream != null) {
                try {
                    fileInputStream.close();
                } catch (IOException e) {
                }
            }
        }
        return null;
    }

    /**
     * 获取图片文件
     * 
     * @param data 请求资源信息
     * @param file 图片文件
     * @return bitmap
     */
    private Bitmap getFileBitmap(LoadImageFile data, File file) {
        FileInputStream fileInputStream = null;
        try {
            fileInputStream = new FileInputStream(file);
            FileDescriptor fileDescriptor = fileInputStream.getFD();
            return decodeSampledBitmapFromDescriptor(fileDescriptor, data.getWidth(), data.getHeight(), getImageCache());
        } catch (Exception ex) {
            AppLog.e(TAG, "httpFrame processLocalCache  获取本地图片出错");
            Log.e(TAG, "Failed to getFileBitmap", ex);
        } finally {
            if (fileInputStream != null) {
                try {
                    fileInputStream.close();
                } catch (IOException e) {
                }
            }
        }
        return null;
    }

    /**
     * 根据网络地址获取资源
     * 
     * @param data 请求资源信息
     * @return bitmap
     */
    private Bitmap processUrlBitmap(LoadImageUrl data, Callback callback) {
        String url = data.getDownloadUrl(mTokenCallback);
        AppLog.d(TAG, "imagecache manager 正在获取网络图片  url:" + url);
        Bitmap bitmap = null;
        if (url != null) {
            try {
                // 下载网络原图
                bitmap = saveUrlBitmapToCache(data, url, callback);
                if (bitmap == null || bitmap.isRecycled()) {
                    // 下载网络的缩略图
                    bitmap = getUrlBitmap(data, url);
                }
            } catch (Throwable e) {
                // 网络相关错误不做处理
                Log.e(TAG, "Failed to processUrlBitmap " + data, e);
                return null;
            }
        }
        return bitmap;
    }

    /**
     * 将网络图片 移动到缓存文件夹中 并且返回用户所需要的bitmap
     * 
     * @param data 请求资源信息
     * @param file 图片文件
     * @return bitmap
     */
    private Bitmap saveUrlBitmapToCache(LoadImageUrl data, String url, Callback callback) throws SocketTimeoutException {
        // 如果是下载缩略图
        if (data.isSmallUrl()) {
            return null;
        }
        FileInputStream fileInputStream = null;
        try {
            // 下载原图
            String key = ImageCache.hashKeyForDisk(data.getSourceCacheKey());
            DiskLruCache.Editor editor = mHttpDiskCache.edit(key);
            if (editor != null) {
                if (downloadUrlToStream(url, editor.newOutputStream(DISK_CACHE_INDEX), data, callback)) {
                    editor.commit();
                } else {
                    editor.abort();
                }
            }
            DiskLruCache.Snapshot snapshot = mHttpDiskCache.get(key);
            fileInputStream = (FileInputStream) snapshot.getInputStream(DISK_CACHE_INDEX);
            FileDescriptor fileDescriptor = fileInputStream.getFD();
            return decodeSampledBitmapFromDescriptor(fileDescriptor, data.getWidth(), data.getHeight(), getImageCache());
        } catch (SocketTimeoutException socketException) {
            throw socketException;
        } catch (Exception ex) {
            AppLog.e(TAG, "httpFrame saveFileBimtapToCache  获取本地图片出错");
            Log.e(TAG, "Failed to saveUrlBitmapToCache " + data, ex);
        } finally {
            if (fileInputStream != null) {
                try {
                    fileInputStream.close();
                } catch (IOException e) {
                }
            }
        }
        return null;
    }

    /**
     * @param url
     * @param file
     */
    public void addCacheToStorage(String url, File file) {
        synchronized (mHttpDiskCacheLock) {
            // Wait for disk cache to initialize
            while (mHttpDiskCacheStarting) {
                try {
                    mHttpDiskCacheLock.wait();
                } catch (InterruptedException e) {
                }
            }

            if (mHttpDiskCache != null && file.isFile()) {
                final String key = ImageCache.hashKeyForDisk(url);
                OutputStream out = null;
                try {
                    DiskLruCache.Snapshot snapshot = mHttpDiskCache.get(key);
                    if (snapshot == null) {
                        final DiskLruCache.Editor editor = mHttpDiskCache.edit(key);
                        if (editor != null) {
                            if (fileToStream(file, editor.newOutputStream(DISK_CACHE_INDEX))) {
                                editor.commit();
                            } else {
                                editor.abort();
                            }
                        }
                    } else {
                        snapshot.getInputStream(DISK_CACHE_INDEX).close();
                    }
                } catch (final IOException e) {
                    AppLog.e(TAG, "httpFrame addBitmapToCache - " + e);
                } catch (Exception e) {
                    AppLog.e(TAG, "httpFrame addBitmapToCache - " + e);
                } finally {
                    try {
                        if (out != null) {
                            out.close();
                        }
                    } catch (IOException e) {
                    }
                }
            }
        }
    }
    
    private Bitmap getUrlBitmap(LoadImageUrl data, String urlString) {
        disableConnectionReuseIfNecessary();
        HttpURLConnection urlConnection = null;
        InputStream is = null;
        try {
            final URL url = new URL(urlString);
            urlConnection = (HttpURLConnection) url.openConnection();
            is = urlConnection.getInputStream();
            Bitmap bmBitmap =  BitmapFactory.decodeStream(is);
            bmBitmap = BitmapUtils.changeBitmapSize(data.getWidth(), data.getHeight(), bmBitmap);
            return bmBitmap;
        } catch (final IOException e) {
            AppLog.e(TAG, "httpframe  getUrlBitmap 写入缓存到硬盘失败");
            Log.e(TAG, "Failed to getUrlBitmap " + data, e);
        } finally {
            if (urlConnection != null) {
                urlConnection.disconnect();
            }
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                }
            }
        }
        return null;
    }

}
