/*
 * 文  件   名： HttpConnector.java
 * 版          权： Copyright GoPawPaw All Rights Reserved.
 * 描          述：[该类的简要描述]
 * 创  建   人： luochun
 * 
 * 修   改  人：
 * 修改时间：
 * 修改内容：[修改内容]
 */
package com.pingan.frame.http;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.lang.Thread.State;
import java.net.HttpCookie;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import javax.net.ssl.HttpsURLConnection;

import android.os.Handler;

import com.pingan.frame.http.listener.HttpSimpleListener;
import com.pingan.frame.util.AppLog;

/**
 * Http连接器<br>
 * 处理多线程队列请求，连接器包含两个线程池：1、数据报文处理线程池；2、文件下载处理线程池<br>
 * 两个线程池中拥有独立的线程数，用于分开处理数据报文和文件下载请求。
 * 
 * @version [Android 1.0.0, 2012-4-18]
 * @description
 */
public class HttpConnector {
	private static final String TAG = HttpConnector.class.getSimpleName();

	/**
	 * 最大的数据报文请求线程数
	 */
	private static final int MAX_THREAD_ACTION = 1;

	/**
	 * 最大的文件下载请求线程数
	 */
	private static final int MAX_THREAD_DOWN = 3;
	
	/**
	 * 最大的文件上传请求线程数
	 */
	private static final int MAX_THREAD_UPLOAD = 3;

	/**
	 * HttpConnector对象的实例
	 */
	private static HttpConnector instance;

	/**
	 * 同步对象Action
	 */
	private static Object object = new Object();

	/**
	 * 请求队列
	 */
	private Vector<HttpRequest> mRequestQueue;

	/**
	 * 当前正在处理的请求队列
	 */
	private Vector<HttpRequest> mRequestQueueCurr;

	/**
	 * Http请求线程池
	 */
	private ArrayList<HttpThread> mThreadList;
	
	private HttpSimpleListener mFilterListener;

	/**
	 * 单例模式
	 */
	private HttpConnector(HttpSimpleListener listener) {
		super();

		mRequestQueue = new Vector<HttpRequest>();
		mRequestQueueCurr = new Vector<HttpRequest>();
		mThreadList = new ArrayList<HttpThread>();
		mFilterListener = listener;
	}
	
	public static HttpSimpleListener getHttpFilterListener(){
		if (instance != null) {
			// 单例模式初始化，同时初始化队列
			return instance.mFilterListener;
		}
		return null;
	}

	public static void init(HttpSimpleListener listener){
		instance = new HttpConnector(listener);
	}
	
	/**
	 * 发送Http请求，单线程请求排队
	 * @param request
	 */
	public static void sendHttpRequest(HttpRequest httpRequest) {
		AppLog.w(TAG, "httpFrame sendHttpRequest  新建一个任务 ");
		//如果任务不合法
		if (httpRequest == null || !httpRequest.isComplete()) {
			if(httpRequest!=null){
				try {
					//直接回调  finish()
					HttpThread.postHttpFinish(httpRequest, 
							httpRequest.createErrorResponse(HttpResponse.STATE_CREATE_REQUEST_ERROR, httpRequest));
				} catch (HttpOperateException e) {
					e.printStackTrace();
					throw new RuntimeException("程序崩溃");
				}
			}
			return;
		}
		
		if(instance == null){
			instance = new HttpConnector(null);
		}
		
		// 将请求插入队列中
		instance.sendSynHttpRequest(httpRequest);

	}
	
	private void sendSynHttpRequest(HttpRequest httpRequest){
		if(httpRequest.getRequestType() == HttpRequest.REQUEST_TYPE_ACTION){
			synchronized (mHttpThreadAction) {
				// 将请求插入队列中
				if(instance.handleHttpRequest(httpRequest)){
					// 请求线程处理
					instance.requestThread(httpRequest.getRequestType());
				}
			}
		}
		else if(httpRequest.getRequestType() == HttpRequest.REQUEST_TYPE_DOWNLOAD){
			synchronized (mHttpThreadDown) {
				// 将请求插入队列中
				if(instance.handleHttpRequest(httpRequest)){
					// 请求线程处理
					instance.requestThread(httpRequest.getRequestType());
				}
			}
		}
		else if(httpRequest.getRequestType() == HttpRequest.REQUEST_TYPE_UPLOAD){
			synchronized (mHttpThreadUpload) {
				// 将请求插入队列中
				if(instance.handleHttpRequest(httpRequest)){
					// 请求线程处理
					instance.requestThread(httpRequest.getRequestType());
				}
			}
		}
		else{
			AppLog.e(TAG, "httpframe 无法识别的getRequestType："+httpRequest.getRequestType());
		}
	}
	
	/**
	 * 处理请求<br>
	 * 将请求插入到请求队列中，并发送申请网络请求线程进行处理，<br>
	 * 若线程忙，则按请求规则排队处理请求
	 * 
	 * @param httpRequest
	 *            新添加的请求
	 * @return 如果添加成功
	 */
	private boolean handleHttpRequest(HttpRequest httpRequest) {
		synchronized (object) {
			// 数据报文请求
			switch (httpRequest.getRequestType()) {
				case HttpRequest.FIRST_LEVEL_CLEAR_TOP:
					// 清空和该请求类型一致队列后再插入请求
					int size = mRequestQueue.size();
					for (int i=0;i<size;i++) {
						HttpRequest hr = mRequestQueue.get(i);
						if(hr.getRequestType() == httpRequest.getRequestType()){
							//移除和该请求类型一致的请求队列
							mRequestQueue.remove(hr);
							i--;
							size--;
						}
					}
					mRequestQueue.add(httpRequest);
					return true;
				case HttpRequest.FIRST_LEVEL_TOP:
					// 将请求插入队列最前位置
					return addHttpRequest(false, httpRequest);
				case HttpRequest.FIRST_LEVEL_BOTTOM:
					// 将请求插入队列最后位置
					return addHttpRequest(true, httpRequest);
				default:
					// 将请求插入队列最后位置
					return addHttpRequest(true, httpRequest);
			}
		}
	}
	
	/**
	 * 添加任务到队列中
	 * @param isBottom		是否叠加到最底部
	 * @param httpRequest	任务
	 * @return	如果添加成功返回true
	 */
	private boolean addHttpRequest(boolean isBottom,HttpRequest httpRequest){
		//获取所有相同请求
		List<HttpRequest> sameRequestList = getAllSameRequestList(httpRequest);
		
		//重复请求不给处理
		if(isRepeatRequest(sameRequestList, httpRequest)){
			AppLog.w(TAG, "httpFrame isRepeatRequest  存在重复请求  "+httpRequest.getClass().getSimpleName());
			return false;
		}
		
		if (sameRequestList==null||sameRequestList.size()<=0) {
			AppLog.i(TAG, "httpFrame 成功添加请求：" + httpRequest.getClass().getSimpleName());
			if(isBottom){
				mRequestQueue.add(httpRequest);
			}else{
				mRequestQueue.add(0, httpRequest);
			}
			return true;
		} else {
			AppLog.w(TAG, "httpFrame 出现相同请求  进行累加监听");
			for(int i=0;i<sameRequestList.size();i++){
				sameRequestList.get(i).addSameRequest(httpRequest);
			}
			if(sameRequestList.size()>1){
				AppLog.e(TAG, "httpFrame 程序出现BUG,为什么出现两个相同请求存在队列中?");
			}
			return false;
		}
		
	}
	/**
	 * 是否是重复任务
	 * @param sameRequestList	相同的任务 通过getAllSameRequestList方法  获取
	 * @param httpRequest	要添加进请求队列的任务
	 * @return
	 */
	private boolean isRepeatRequest(List<HttpRequest> sameRequestList,HttpRequest httpRequest){
		if(sameRequestList==null){
			return false;
		}
		for(int i=0;i<sameRequestList.size();i++){
			HttpRequest requestItem = sameRequestList.get(i);
			if(requestItem !=null){
				if(requestItem.isRepeatRequest(httpRequest)){
					return true;
				}
			}else{
				AppLog.e(TAG, "httpFrame isRepeatRequest  请求队列中存在为空的请求");
			}
		}
		return false;
	}
	
	
	private List<HttpRequest> getSameRequestListFromList(HttpRequest httpRequest,Vector<HttpRequest> requestQueue){
		List<HttpRequest> sameRequestList = null;
		for(int i=0;i<requestQueue.size();i++){
			HttpRequest requestItem = requestQueue.get(i);
			if(requestItem !=null){
				if(requestItem.equals(httpRequest)){
					if(sameRequestList == null){
						sameRequestList = new ArrayList<HttpRequest>();
					}
					sameRequestList.add(requestItem);
				}
			}else{
				AppLog.e(TAG, "httpFrame getSameRequestListFromList  请求队列中存在为空的请求");
			}
			
		}
		
		return sameRequestList;
		
	}
	
	private List<HttpRequest> getAllSameRequestList(HttpRequest httpRequest){
		List<HttpRequest> sameWaitRequestList = getSameRequestListFromList(httpRequest, mRequestQueue);
		List<HttpRequest> sameDoingRequestList = getSameRequestListFromList(httpRequest, mRequestQueueCurr);
		List<HttpRequest> sameRequstList = new ArrayList<HttpRequest>();
		if(sameWaitRequestList!=null&&sameWaitRequestList.size()>0){
			sameRequstList.addAll(sameWaitRequestList);
			
		}
		if(sameDoingRequestList!=null&&sameDoingRequestList.size()>0){
			sameRequstList.addAll(sameDoingRequestList);
		}
		if(sameRequstList.size()>1){
			AppLog.e(TAG, "httpFrame getAllSameRequestList  相同的任务出现了"+sameRequstList.size()+"个    其中正在处理的任务"+sameDoingRequestList.size()+"个    等待处理的任务"+sameWaitRequestList.size()+"个");
		}
		
		return sameRequstList;
	}
	
	/**
	 * 
	 * 请求线程处理
	 * 
	 * @param requestType
	 */
	private void requestThread(int requestType) {
		if (requestType == HttpRequest.REQUEST_TYPE_ACTION) {
			// 数据报文请求

			int tsize = mThreadList.size();
			int tAlivesize = 0;
			for (int i=0;i<tsize;i++) {
				HttpThread ht = mThreadList.get(i);
				
				if (ht.getHttpThreadListener() == mHttpThreadAction) {
					// 该线程属于用来处理文件下载请求
					
					if(ht.getState() == State.WAITING){
						//若存在线程处于等待状态
						synchronized (mHttpThreadAction) {
							mHttpThreadAction.notify();
						}
						AppLog.i(TAG, "httpFrame Thread : " +ht.getName() + " 已经唤醒...");
						//一次只唤醒一个线程
						return;
					}
					
					tAlivesize++;
					
				}
			}
			
			if (tAlivesize < MAX_THREAD_ACTION) {
				// 可用线程数未达到最大值，则增加线程
				HttpThread ht = new HttpThread(mHttpThreadAction);
				ht.start();
				mThreadList.add(ht);
			}

		} else if (requestType == HttpRequest.REQUEST_TYPE_DOWNLOAD) {
			// 文件下载请求

			int tsize = mThreadList.size();
			int tAlivesize = 0;
			for (int i=0;i<tsize;i++) {
				HttpThread ht = mThreadList.get(i);
				
				if (ht.getHttpThreadListener() == mHttpThreadDown) {
					// 该线程属于用来处理文件下载请求
					
					if(ht.getState() == State.WAITING){
						//若存在线程处于等待状态
						synchronized (mHttpThreadDown) {
							mHttpThreadDown.notify();
						}
						
						AppLog.i(TAG, "httpFrame Thread : " +ht.getName() + " 已经唤醒...");
						
						//一次只唤醒一个线程
						return;
					}
					tAlivesize++;
				}
			}
			
			if (tAlivesize < MAX_THREAD_DOWN) {
				// 可用线程数未达到最大值，则增加线程
				HttpThread ht = new HttpThread(mHttpThreadDown);
				ht.start();
				
				mThreadList.add(ht);
			}

		} else if (requestType == HttpRequest.REQUEST_TYPE_UPLOAD) {
			// 文件下载请求
			int tsize = mThreadList.size();
			int tAlivesize = 0;
			for (int i=0;i<tsize;i++) {
				HttpThread ht = mThreadList.get(i);
				
				if (ht.getHttpThreadListener() == mHttpThreadUpload) {
					// 该线程属于用来处理文件下载请求
					
					if(ht.getState() == State.WAITING){
						//若存在线程处于等待状态
						synchronized (mHttpThreadUpload) {
							mHttpThreadUpload.notify();
						}
						
						AppLog.i(TAG, "httpFrame Thread : " +ht.getName() + " 已经唤醒...");
						
						//一次只唤醒一个线程
						return;
					}
					tAlivesize++;
				}
			}
			
			if (tAlivesize < MAX_THREAD_UPLOAD) {
				// 可用线程数未达到最大值，则增加线程
				HttpThread ht = new HttpThread(mHttpThreadUpload);
				ht.start();
				
				mThreadList.add(ht);
			}

		}
		
	}
	
	/**
	 * 清楚所有请求
	 */
	public static void clearRequest() {
		clearRequest(0);
	}
	
	/**
	 * 根据请求类型、清楚请求
	 * @param requestType
	 */
	public static void clearRequest(int requestType) {
		if(instance == null){
			return;
		}
		
		if(requestType == 0){
			instance.mRequestQueue.removeAllElements();
			return;
		}
		
		int size = instance.mRequestQueue.size();
		for (int i=0;i<size;i++) {
			HttpRequest hr = instance.mRequestQueue.get(i);
			if(hr.getRequestType() == requestType){
				//移除和该请求类型一致的请求队列
				instance.mRequestQueue.remove(hr);
				i--;
				size--;
			}
		}
	}

	/**
	 * 数据报文请求线程，获取请求对象的接口
	 */
	private HttpThreadListener mHttpThreadAction = new HttpThreadListener() {

		@Override
		public HttpRequest getHttpRequest() {

			synchronized (object) {

				if (mRequestQueue.size() > 0) {
					for (HttpRequest hr : mRequestQueue) {
						if (hr.getRequestType() == HttpRequest.REQUEST_TYPE_ACTION) {
							// 数据报文请求
							mRequestQueue.remove(hr);
							mRequestQueueCurr.add(hr);
							return hr;
						}
					}
				}
			}

			return null;
		}

		@Override
		public void finishHttpRequest(HttpRequest httpRequest) {
			// TODO Auto-generated method stub
			synchronized (object) {
				AppLog.w(TAG, "httpFrame finishHttpRequest 当前正在处理的任务总共有"+mRequestQueueCurr.size());
				mRequestQueueCurr.remove(httpRequest);
				AppLog.w(TAG, "httpFrame finishHttpRequest 处理完此任务还剩下"+mRequestQueueCurr.size());
			}
		}

		@Override
		public void requestWait(HttpThread httpThread) {
			synchronized (mHttpThreadAction) {
				try {
					AppLog.i(TAG, "httpFrame Thread : " +httpThread.getName() + " 正在等待中...");
					mHttpThreadAction.wait();

				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
			
		}

	};

	/**
	 * 文件下载请求线程，获取请求对象的接口
	 */
	private HttpThreadListener mHttpThreadDown = new HttpThreadListener() {

		@Override
		public HttpRequest getHttpRequest() {

			synchronized (object) {

				if (mRequestQueue.size() > 0) {
					for (HttpRequest hr : mRequestQueue) {
						if (hr.getRequestType() == HttpRequest.REQUEST_TYPE_DOWNLOAD) {
							// 文件下载请求
							mRequestQueue.remove(hr);
							mRequestQueueCurr.add(hr);
							return hr;
						}
					}
				}
			}

			return null;
		}

		@Override
		public void finishHttpRequest(HttpRequest httpRequest) {
			// TODO Auto-generated method stub
			synchronized (object) {
				mRequestQueueCurr.remove(httpRequest);
			}
		}

		@Override
		public void requestWait(HttpThread httpThread) {
			synchronized (mHttpThreadDown) {
				try {
					AppLog.i(TAG, "httpFrame Thread : " +httpThread.getName() + " 正在等待中...");
					mHttpThreadDown.wait();

				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		}
	};
	
	/**
	 * 文件下载请求线程，获取请求对象的接口
	 */
	private HttpThreadListener mHttpThreadUpload = new HttpThreadListener() {

		@Override
		public HttpRequest getHttpRequest() {

			synchronized (object) {

				if (mRequestQueue.size() > 0) {
					for (HttpRequest hr : mRequestQueue) {
						if (hr.getRequestType() == HttpRequest.REQUEST_TYPE_UPLOAD) {
							// 文件下载请求
							mRequestQueue.remove(hr);
							mRequestQueueCurr.add(hr);
							return hr;
						}
					}
				}
			}

			return null;
		}

		@Override
		public void finishHttpRequest(HttpRequest httpRequest) {
			synchronized (object) {
				mRequestQueueCurr.remove(httpRequest);
			}
		}

		@Override
		public void requestWait(HttpThread httpThread) {
			synchronized (mHttpThreadUpload) {
				try {
					AppLog.i(TAG, "httpFrame Thread : " +httpThread.getName() + " 正在等待中...");
					mHttpThreadUpload.wait();

				} catch (InterruptedException e) {
					e.printStackTrace();
				}
			}
		}
	};

}
