package com.pingan.frame.http.download;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URLDecoder;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import android.content.Context;
import android.os.Handler;
import android.text.TextUtils;

import com.pingan.frame.http.HttpHelper;
import com.pingan.frame.http.HttpOperateException;
import com.pingan.frame.http.HttpRequest;
import com.pingan.frame.http.HttpResponse;
import com.pingan.frame.http.HttpThread;
import com.pingan.frame.http.bitmap.HttpBitmapRequest;
import com.pingan.frame.http.listener.HttpProgressListener;
import com.pingan.frame.http.listener.TokenCallback;
import com.pingan.frame.http.upload.AuthPolicy;
import com.pingan.frame.util.AppFluxLog;
import com.pingan.frame.util.AppLog;
import com.pingan.frame.util.FileUtil;
import com.pingan.frame.util.DownloadFileSaveUtil;
import com.pingan.frame.util.Tools;
import com.pingan.frame.util.DownloadFileSaveUtil.DownloadFileSave;

/**
 * 文件下载相关逻辑代码
 * @author ex-luochun001
 */
public class HttpDownloadRequest extends HttpRequest{
	/**
	 * TAG
	 */
	private final String TAG = "HttpDownloadRequest";
	
	/**
	 * 保存路径
	 */
	private final String mSavePath;
	
	private String mSaveFilePath;
	/**
	 * 保存文件名
	 */
	private String mSaveFileName;
	/**
	 * 是否需要持续下载
	 */
	private boolean isNeedContinueDownload = false;

	/**
	 * 资源文件名,当是公有资源是此变量为下载地址
	 */
	private final String mFileName;
	/**
	 * 样式 (缩略图等等相关)
	 */
	private final String mStyleName;
	/**
	 * token获取callback
	 */
	private final TokenCallback mTokenCallback;
	/**
	 * save
	 */
	private DownloadFileSave mDownloadFileSave;
	
	protected long mContentLength = -1;
	/**
	 * stylename:android_high		
	 * 视频编码器:libx264	
	 * 视频格式:mp4	
	 * 解析率:480x320	
	 * 视频码率:700k	
	 * 音频编码器:libfaac	
	 * 音频码率:128k	
	 * 音频采样率:48k
	 */
	public static final String STYLE_NAME_VIDEO_HIGH = "android_high";
	/**
	 * stylename:android_low
	 * 视频编码器:libx264	
	 * 视频格式:mp4	
	 * 解析率:480x320	
	 * 视频码率:96k
	 * 音频编码器:libfaac	
	 * 音频码率:64k
	 * 音频采样率:48k
	 */
	public static final String STYLE_NAME_VIDEO_LOW = "android_low";
	/**
	 * stylename:android
	 * 视频编码器:libx264	
	 * 视频格式:mp4	
	 * 解析率:480x320	
	 * 视频码率:512k
	 * 音频编码器:libfaac	
	 * 音频码率:128k
	 * 音频采样率:48k
	 */
	public static final String STYLE_NAME_VIDEO_NORMAL = "android";

	protected Context mContext;
	
	/**
	 * 图片请求成功会以此url进行缓存到内存当中
	 */
	protected String mCacheXmlUrl;
	
	/**
	 * 非七牛下载
	 * 下载原本不带样式的资源的  构造函数
	 * @param downloadUrl	资源下载地址
	 * @param savePath		资源下载完成后保存地址
	 * @param tokenCallback	tokenCallbak
	 */
	public HttpDownloadRequest(Context context,String downloadUrl,String savePath) {
		super(null,HttpRequest.REQUEST_METHOD_GET);
		this.mContext = context;
		this.mSavePath = savePath==null?"":savePath;
		this.mFileName = downloadUrl==null?"":downloadUrl;
		this.mStyleName = "";
		this.mTokenCallback = null;
		this.mCacheXmlUrl = this.mFileName;
	}
	
	/**
	 * 七牛下载
	 * 下载原本不带样式的资源的  构造函数
	 * @param downloadUrl	资源下载地址
	 * @param savePath		资源下载完成后保存地址
	 * @param tokenCallback	tokenCallbak
	 */
	public HttpDownloadRequest(Context context,String downloadUrl,String savePath,TokenCallback tokenCallback) {
		super(null,HttpRequest.REQUEST_METHOD_GET);
		this.mContext = context;
		this.mSavePath = savePath==null?"":savePath;
		this.mFileName = downloadUrl==null?"":downloadUrl;
		this.mStyleName = "";
		this.mTokenCallback = tokenCallback;
		this.mCacheXmlUrl = this.mFileName;
	}
	
	/**
	 * 七牛下载
	 * 下载带有样式的资源的构造函数
	 * @param downloadUrl	资源下载地址
	 * @param savePath		资源下载完成后保存地址
	 * @param styleName		资源样式(高清低清视频)
	 * @param tokenCallback	toenCallback
	 */
	public HttpDownloadRequest(Context context,String downloadUrl,String savePath,String styleName,TokenCallback tokenCallback) {
		super(null,HttpRequest.REQUEST_METHOD_GET);
		this.mContext = context;
		this.mSavePath = savePath==null?"":savePath;
		this.mFileName = downloadUrl==null?"":downloadUrl;
		this.mStyleName = styleName==null?"":styleName;
		this.mTokenCallback = tokenCallback;
		if(!Tools.isEmpty(styleName)){
			this.mCacheXmlUrl = ""+this.mFileName+styleName;
		}else{
			this.mCacheXmlUrl = ""+this.mFileName;
		}
		
	}
	/**
	 * GET 
	 * @return 资源样式(缩略图,高清低清视频)
	 */
	public String getStyleName() {
		return mStyleName;
	}
	/**
	 * GET 
	 * @return 资源保存地址
	 */
	public String getSavePath() {
		return mSavePath;
	}
	/**
	 * GET
	 * @return	资源下载地址
	 */
	public String getFileName(){
		return mFileName;
	}
	
	@Override
	public boolean isComplete() {
		if(Tools.isEmpty(mFileName)){
			AppLog.e(TAG, "httpFrame 下载地址不能为空");
			return false; 
		}
		else if(Tools.isEmpty(mSavePath)){
			AppLog.e(TAG, "httpFrame 下载完成后的保存地址不能为空");
			return false; 
		}
		return true;
	}


	@Override
	public long getReadTimeout() {
		return 40000;
	}

	@Override
	public long getConnectTimeout() {
		return 40000;
	}


	@Override
	public Object getParamData() {
		return null;
	}


	@Override
	public int getFirstLevel() {
		return mFirstLevel;
	}

	@Override
	public int getRequestType() {
		return HttpRequest.REQUEST_TYPE_DOWNLOAD;
	}

	@Override
	public HttpResponse createErrorResponse(int state, HttpRequest httpTask) {
		return new HttpDownloadResponse(state,httpTask);
	}


	@Override
	public HttpResponse onPreExecute() throws Exception {
		if(!(this instanceof HttpBitmapRequest)){
			String filePath = DownloadFileSaveUtil.getUsableCachePath(mContext, mCacheXmlUrl);
			if(filePath!=null){
				AppLog.i(TAG, "httpFrame  threadName:"+Thread.currentThread().getName()+" 已经找到本地路径的文件无需下载");
				return new HttpDownloadResponse(HttpResponse.STATE_SUCCESS, this, filePath);
			}
		}
		
		if(HttpHelper.hasProtocol(mFileName)){
			if(mStyleName!=null&&mStyleName.length()>0){
				setUrl(mFileName+"/"+mStyleName);
			}else{
				setUrl(mFileName);
			}
			
		}else{
			setUrl(URLDecoder.decode(getTokenStr()));
		}
		return null;
		
	}
	
	/**
	 * 获取tokenMap
	 * @return	token
	 * @throws Exception
	 */
	private HashMap<String, String> getToken() throws Exception{
		if(mTokenCallback==null){
			return AuthPolicy.getDefaultToken();
		}else{
			HashMap<String, String> tokenMap = mTokenCallback.getToken(this);
			if(tokenMap==null){
				AppLog.e(TAG, "httpFrame  threadName:"+Thread.currentThread().getName()+"  threadName:"+Thread.currentThread().getName()+" getToken  token 获取失败");
				throw new Exception("httpFrame getToken  token 获取失败");
			}
			return tokenMap;
		}
	} 
	
	/**
	 * 获取Token String
	 * @return	token
	 * @throws Exception
	 */
	private String getTokenStr() throws Exception{
		HashMap<String, String> tokenMap = getToken();
		
		if(tokenMap!=null){
			return tokenMap.get(TokenCallback.KEY_TOKEN);
		}
		AppLog.e(TAG, "httpFrame threadName:"+Thread.currentThread().getName()+" getTokenStr  token 获取失败");
		return null;
		
	}
	
	/**
	 * 图片请求成功会以此url进行缓存到内存当中
	 * @return
	 */
	public String getCacheUrl(){
		return mCacheXmlUrl;
	}
	
	public String getExtension(HttpURLConnection connection) {
		String field=connection.getHeaderField("Content-Disposition");
		String fileName = "";
		if(field!=null){
			fileName = field.substring(field.indexOf("filename")+10, field.length()-1); //获取filename="xxx"中的xxx
		}else{
			fileName = getFileName();
		}
		
		return FileUtil.getExtensionName(fileName);
	}
	
	@Override
	public HttpResponse onDoingExecute(HttpURLConnection connection)
			throws Exception {
		InputStream is = connection.getInputStream();
		mContentLength = connection.getContentLength();
		
		InputStream httpInStream = is;
		//图片下载开始回调
		//每次向流里面读取的字节数  1KB
		byte[] buffer = new byte[1024];
		//当前读取的长度
		int len = -1;
		//已经读取流的总大小
		int readBytes = 0;
		
		String extension = getExtension(connection);
		
		if(FileUtil.getSDFreeSize(mSavePath)>mContentLength){
			AppLog.i(TAG, "httpFrame threadName:"+Thread.currentThread().getName()+" onDoingExecute 手机拥有足够内存，先保存进内存卡中");
			
			if(mSaveFileName != null&&!mSaveFileName.trim().equals("")){
				mSaveFilePath = mSavePath +File.separator+ mSaveFileName;
			}else{
				mSaveFilePath = mSavePath +File.separator+FileUtil.getTimeMillisFileName();
				if(extension!=null&&extension.length()>0){
					mSaveFilePath = mSaveFilePath+"."+extension;
				}
			}
			
			File file=new File(mSaveFilePath);
			file.createNewFile();
			FileOutputStream fos=null;
			try{
				fos = new FileOutputStream(file);
				while ((len = httpInStream.read(buffer)) != -1) {
					readBytes+= len;
					
					fos.write(buffer,0,len);
					
					//回调 汇报进度
					HttpThread.postUploadProgress(this, readBytes, mContentLength);
				}
			}catch(Exception ex){
				throw new Exception("下载失败   IO流处理出错");
			}finally{
				if(fos!=null){
					fos.flush();
					fos.close();
				}
				
			}
			
			AppFluxLog.i("AppFluxLog", "上行:未知      下行:"+file.length()+"   时间:"+AppFluxLog.LOG_TIME_FORMAT.format(new Date(System.currentTimeMillis()))+"   url:"+mUrl);
			return executeContent(mSaveFilePath,extension,readBytes);
		}else{
			//如果是下载图片
			if(this instanceof HttpBitmapRequest){
				AppLog.i(TAG, "httpFrame threadName:"+Thread.currentThread().getName()+" onDoingExecute 手机内存卡空间不足，只能采用内存");
				//sdcard内存不足
				ByteArrayOutputStream byteArraryStream = new ByteArrayOutputStream();
				//每次读取1kb
				while ((len = httpInStream.read(buffer)) != -1) {
					readBytes+= len;
					//回调 汇报进度
					HttpThread.postUploadProgress(this, readBytes, mContentLength);
					byteArraryStream.write(buffer, 0, len);
				}
				
				byte[] bytes = byteArraryStream.toByteArray();
				
				//不再用到的资源进行回收
				byteArraryStream.flush();
				byteArraryStream.close();
				
				AppFluxLog.i("AppFluxLog", "上行:未知      下行:"+bytes.length+"   时间:"+AppFluxLog.LOG_TIME_FORMAT.format(new Date(System.currentTimeMillis()))+"   url:"+mUrl);
				return executeContent(bytes,extension,readBytes);
			}
			//如果是下载文件，则必须依赖存储空间
			else{
				return new HttpDownloadResponse(HttpResponse.STATE_MEMORY_NOT_ENOUGH, this);
			}
			
			
		}
		
		
		
	}
	
	/**
	 * 处理inputStream的 数据
	 * @param content	数据
	 * @return	response
	 */
	protected HttpResponse executeContent(byte[] content,String extension,long fileLength){
		String filePath = FileUtil.saveFile(mSavePath, extension, content);
		return executeContent(filePath, extension,fileLength);
	}
	
	/**
	 * 处理inputStream的 数据
	 * @param content	数据
	 * @return	response
	 */
	protected HttpResponse executeContent(String filePath,String extension,long fileLength){
		
		DownloadFileSaveUtil.saveFile(mContext, mCacheXmlUrl, 
				new DownloadFileSave(filePath));
		return new HttpDownloadResponse(HttpResponse.STATE_SUCCESS, this, filePath);
	}
	
	@Override
	public void onPostExecute(boolean isSuccess) {
		if(!isSuccess){
			if(isNeedContinueDownload){
				DownloadFileSaveUtil.saveFile(mContext, mUrl, 
						new DownloadFileSave(mSaveFilePath, mContentLength)) ;
			}
			
		}
	}
	
	@Override
	public boolean equals(Object o) {
		// TODO Auto-generated method stub
		if(o instanceof HttpDownloadRequest){
			HttpDownloadRequest tempRequest = (HttpDownloadRequest)o;
			if(mFileName.equals(tempRequest.getFileName())&&
				mStyleName.equals(tempRequest.getStyleName())&&
				mSavePath.equals(tempRequest.getSavePath())){
				return true;
			}
		}
		return false;
	}
	
	public void setSaveFileName(String saveFileName){
		mSaveFileName = saveFileName;
		mCacheXmlUrl += saveFileName;
	}

	private void setNeedContinueDownload(boolean isNeedContinueDownload) {
		this.isNeedContinueDownload = isNeedContinueDownload;
	}

	@Override
	public void onPreExecute(HttpURLConnection connection)
			throws Exception {
		mDownloadFileSave = DownloadFileSaveUtil.getFile(mContext, mCacheXmlUrl);
		if(mDownloadFileSave!=null&&
				mDownloadFileSave.getDownloaStatus()==DownloadFileSave.DOWNLOAD_STATUS_DOWNLOADING){
			connection.setRequestProperty("Range", "bytes="+mDownloadFileSave.mFileLength);
		}
	}
	
}
