package com.pingan.frame.http.upload;


import java.io.File;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.net.HttpURLConnection;
import java.util.HashMap;
import java.util.List;
import java.util.zip.CRC32;

import org.json.JSONException;
import org.json.JSONObject;

import android.graphics.Bitmap;
import android.graphics.Bitmap.Config;
import android.os.Handler;

import com.pingan.frame.http.HttpOperateException;
import com.pingan.frame.http.HttpRequest;
import com.pingan.frame.http.HttpResponse;
import com.pingan.frame.http.HttpThread;
import com.pingan.frame.http.listener.HttpProgressListener;
import com.pingan.frame.http.listener.TokenCallback;
import com.pingan.frame.util.AppLog;
import com.pingan.frame.util.BitmapUtils;
import com.pingan.frame.util.FileUtil;
import com.pingan.frame.util.Tools;

/**
 * 文件上传
 * @author ex-luochun001
 */
public class HttpUploadContinueRequest extends HttpRequest {
	/**
	 * TAG
	 */
	private final String TAG = "HttpUploadContinueRequest";
	/**
	 * 上传文件的路径
	 */
	private String mFilePath ;
	/**
	 * IO读写文件流
	 */
	private RandomAccessFile mFileIO;
	/**
	 * IO流操作只读写
	 */
	private final String FILE_ONLY_READ = "r";
	/**
	 * 已经上传文件多少字节
	 */
	private long mHasUploadBytes;
	/**
	 * 被上传的文件总字节数
	 */
	private long mUpLoadFileLength;
	/**
	 * 被分为多少大块进行上传
	 */
	private int mBlockCount;
	/**
	 * 当前上传到第几块
	 */
	private int mCurBlockIndex;
	/**
	 * 压缩最大高度
	 */
	private int mMaxMeasureHeight = 960;
	/**
	 * 压缩最大宽度
	 */
	private int mMaxMeasureWidth = 960;
	/**
	 * 创建文件需要的数组 ,标示着将服务器的那几大块进步合并
	 */
	private String[] mCheckSumArrary;
	/**
	 * 上传的参数
	 */
	private byte[] mParameter;
	/**
	 * 初始化的json String
	 */
	private String mInitJsonMessage;
	/**
	 * token获取callback
	 */
	private TokenCallback mTokenCallback;
	/**
	 * 空间名称
	 */
	private String mBucketName;
	/**
	 * file name
	 */
	private String mUploadFileName;
	/**
	 * 文件类型
	 * {@link #FILE_TYPE_OTHER} 等等
	 */
	private final String mFileType;
	/**
	 * 资源类型
	 * {@link #BIZ_TYPE_COMM} 等等
	 */
	private final String mBizType;
	/**
	 * {@link #com.pingan.frame.http.listener.TokenCallback.KEY_TOKEN}
	 */
	private String mHostUrl;;
	/**
	 * 临时保存目录
	 */
	private String mTempFilePath;
	/**
	 * 其他
	 */
	public static final String FILE_TYPE_OTHER = "0";
	/**
	 * 视频 
	 */
	public static final String FILE_TYPE_VIDEO = "3";
	/**
	 * 图片
	 */
	public static final String FILE_TYPE_IMAGE = "1";
	/**
	 * 音频文件
	 */
	public static final String FILE_TYPE_SOUND = "2";
	/**
	 * 公共资源  如头像等等
	 */
	public static final String BIZ_TYPE_COMM = "1";
	/**
	 * 私有资源
	 */
	public static final String BIZ_TYPE_PRIVATE = "0";
	
	/**
	 * 上传文件的构造函数
	 * @param filePath		要上传得文件文件路径
	 * @param fileType		{@link #FILE_TYPE_IMAGE} 等等
	 * @param bizType		{@link #BIZ_TYPE_COMM} 等等
	 * @param tokenCallback	tokenCallback
	 */
	public HttpUploadContinueRequest(String filePath,String fileType,String bizType,TokenCallback tokenCallback) {
		super(null, HttpRequest.REQUEST_METHOD_POST);
		this.mFilePath = filePath==null?"":filePath;
		this.mHasUploadBytes = 0;
		this.mFileType = fileType==null?"":fileType;
		this.mBizType = bizType==null?"":bizType;
		this.mTokenCallback = tokenCallback;
	}
	
	
	/**
	 * 文件续传的构造函数
	 * @param filePath			要上传得文件文件路径
	 * @param initJsonMessage	上次上传的进度 详细信息json
	 * @param fileType			{@link #FILE_TYPE_IMAGE} 等等
	 * @param bizType			{@link #BIZ_TYPE_COMM} 等等
	 * @param tokenCallback		tokenCallback
	 */
	public HttpUploadContinueRequest(String filePath,String initJsonMessage,String fileType,String bizType,TokenCallback tokenCallback) {
		super(null, HttpRequest.REQUEST_METHOD_POST);
		this.mFilePath = filePath;
		this.mInitJsonMessage = initJsonMessage;
		this.mFileType = fileType;
		this.mBizType = bizType;
		this.mTokenCallback = tokenCallback;
	}
	
	/**
	 * 设置空间名
	 * @param bucketName	空间名
	 */
	private void setBucketName(String bucketName){
		this.mBucketName = bucketName;
	}
	/**
	 * GET
	 * @return {@link #FILE_TYPE_IMAGE} 等等
	 */
	public String getFileType() {
		return mFileType;
	}
	/**
	 * GET
	 * @return	{@link #BIZ_TYPE_COMM} 等等
	 */
	public String getBizType() {
		return mBizType;
	}
	/**
	 * SET
	 * @param callback	TokenCallback
	 */
	public void setTokenCallback(TokenCallback callback){
		mTokenCallback = callback;
	}
	
	@Override
	public boolean isComplete() {
		if(Tools.isEmpty(mFilePath)){
			AppLog.e(TAG, "httpFrame 上传的文件不能为空");
			return false;
		}
		return true;
	}

	@Override
	public long getReadTimeout() {
		return 40000;
	}

	@Override
	public long getConnectTimeout() {
		return 40000;
	}

	@Override
	public Object getParamData() throws Exception {
		return mParameter;
	}

	@Override
	public int getFirstLevel() {
		return mFirstLevel;
	}

	@Override
	public int getRequestType() {
		// TODO Auto-generated method stub
		return HttpRequest.REQUEST_TYPE_UPLOAD;
	}
	
	@Override
	public HttpResponse createErrorResponse(int state, HttpRequest httpTask) {
		// TODO Auto-generated method stub
		return new HttpUploadContinueResponse(state,httpTask);
	}

	@Override
	public HttpResponse onPreExecute() throws Exception{
		if(mFileIO==null){
			File tempFile = new File(mFilePath);
			if(mFileType == FILE_TYPE_IMAGE&&tempFile.length()/1024>150){
				String extension = FileUtil.getExtensionName(mFilePath);
				Bitmap bitmap = BitmapUtils.getBitmap(mFilePath, mMaxMeasureWidth,mMaxMeasureHeight,null);
				mTempFilePath = FileUtil.compressImagePath(bitmap, FileUtil.getFolderPath(mFilePath), extension);
				mFileIO = new RandomAccessFile(mTempFilePath, FILE_ONLY_READ);
				
			}else{
				mFileIO = new RandomAccessFile(mFilePath, FILE_ONLY_READ);
			}
			
			mUpLoadFileLength = mFileIO.length();
			mBlockCount = HttpUploadUtil.getBlockCount(mUpLoadFileLength);
			mCheckSumArrary = new String[mBlockCount];
			mCurBlockIndex = 0;
			mHasUploadBytes = 0;
			
			
			HashMap<String, String> tokenHashMap = getToken();
			String bucketName = getBucketName(tokenHashMap.get(TokenCallback.KEY_ETC));
			String jid = tokenHashMap.get(TokenCallback.KEY_JID);
			String fileName = jid+FileUtil.getTimeMillisFileName()+"."+FileUtil.getExtensionName(getFilePath());
			setBucketName(bucketName);
			setFileName(fileName);
			setUploadHeader(tokenHashMap.get(TokenCallback.KEY_TOKEN), null);
			//如果是公有文件
			if(!mBizType.equals("0")){
				mHostUrl = tokenHashMap.get(TokenCallback.KEY_ETC);
			}
			
			mParameter = HttpUploadUtil.nextChunk(mFileIO, mHasUploadBytes, mUpLoadFileLength, 
					mCurBlockIndex, mBlockCount);
			
			setUrl(HttpUploadUtil.UP_HOST+"/mkblk/"+HttpUploadUtil.makeBlockLength(mHasUploadBytes, mUpLoadFileLength, mCurBlockIndex, mBlockCount));
		}
		return null;
	}
	
	/**
	 * 
	 * 根据{@link #com.pingan.frame.http.listener.TokenCallback.KEY_TOKEN}获取空间名
	 * @param hostUrl	 TokenCallback.KEY_TOKEN
	 * @return	BucketName
	 */
	private String getBucketName(String hostUrl){
		//http://pingantxt.qiniudn.com  
		//http://pinganimg.qiniudn.com  
		int startIndex = hostUrl.indexOf("//");
		int endIndex = hostUrl.indexOf(".");
		return hostUrl.substring(startIndex+2,endIndex);
	}
	
	@Override
	public HttpResponse onDoingExecute(HttpURLConnection connection)
			throws Exception {
		InputStream is = connection.getInputStream();
		long contentLength = connection.getContentLength();
		HttpUploadContinueResponse httpResponse = null;
		String jsonStr = Tools.inputStream2String(is);
		JSONObject jsonObject = new JSONObject(jsonStr);
		if(mUrl.contains("/rs-mkfile/")){
			jsonObject.put("key", getFileName());
			httpResponse = new HttpUploadContinueResponse(HttpUploadContinueResponse.STATE_SUCCESS, 
					this, jsonObject,mHostUrl);
		}else{
			String ctx = jsonObject.getString("ctx");
			String checksum = jsonObject.getString("checksum");
			String host = jsonObject.getString("host");
			
			
			if(!checkUploadSrc32(jsonObject, mParameter)){
				throw new Exception("onDoingExecute  实质上传的文件和想要上传得文件不一致");
			}
			
			//上传成功  累计上传成功
			mHasUploadBytes += mParameter.length;
			mParameter = null;
			
			//如果是一个大块的结束
			if(HttpUploadUtil.isBlockEnd(mHasUploadBytes, mUpLoadFileLength, mBlockCount, mCurBlockIndex)){
				mCheckSumArrary[mCurBlockIndex] = checksum;
				//如果读取完毕  准备makeFile
				if(mHasUploadBytes == mUpLoadFileLength){
					AppLog.i(TAG, "httpFrame  threadName:"+Thread.currentThread().getName()+" onDoingExecute  上传文件成功,正在请求服务器创建文件");
					//这里写makefile
					httpResponse = getMakeFileResponse(host, mCheckSumArrary, mBucketName, mFileIO, mUpLoadFileLength);
					
				}else{
					AppLog.i(TAG, "httpFrame  threadName:"+Thread.currentThread().getName()+" onDoingExecute  已经成功地处理了"+mCurBlockIndex+"大块,正在构建下一块");
					httpResponse = getNextBlockResponse(mHasUploadBytes, mUpLoadFileLength, 
							mCurBlockIndex, mBlockCount,mFileIO);
				}
			}else{
				AppLog.i(TAG, "httpFrame  threadName:"+Thread.currentThread().getName()+" onDoingExecute  已经成功地处理了一个小块,正在构建下一个小块");
				httpResponse = getNextChunkResponse(host, ctx, mHasUploadBytes, mUpLoadFileLength, 
						mCurBlockIndex, mBlockCount, mFileIO);
			}
			AppLog.i(TAG, "httpFrame  threadName:"+Thread.currentThread().getName()+" onDoingExecute  已上传:"+mHasUploadBytes+"  总大小:"+mUpLoadFileLength+"  当前块:"+mCurBlockIndex+"  总块数:"+mBlockCount);
		}
		postUploadProgress();
		return httpResponse;
	}
	
	/**
	 * 检查实质上传的文件crc32和想要上传得文件crc32是否一致
	 * @param jsonObject	实际上传的文件crc32
	 * @param uploadBytes	想要上传的文件的字节数组
	 * @return	如果一致返回true
	 * @throws JSONException
	 */
	private boolean checkUploadSrc32(JSONObject jsonObject,byte[] uploadBytes) throws JSONException{
		Object crc32Object = jsonObject.get("crc32");
		long crc32 = 0;
		if (crc32Object instanceof Long) {
			crc32 = (long)(Long)crc32Object;
		} else if (crc32Object instanceof Integer) {
			crc32 = (long)(int)(Integer)crc32Object;
		}
		
		CRC32 uploadSrc32 = new CRC32();
		uploadSrc32.update(mParameter, 0, mParameter.length);
		if (crc32 != uploadSrc32.getValue()) {
			// Something wrong!
			AppLog.e(TAG, "httpFrame  threadName:"+Thread.currentThread().getName()+" onDoingExecute  实质上传的文件和想要上传得文件不一致");
			return false;
		}
		return true;
	}
	
	/**
	 * 
	 * @param host
	 * @param ctx
	 * @param hasUploadBytes
	 * @param uploadFileLength
	 * @param blockIndex
	 * @param blockCount
	 * @param fileIO
	 * @return
	 * @throws Exception
	 */
	private HttpUploadContinueResponse getNextChunkResponse(String host,String ctx,long hasUploadBytes,
			long uploadFileLength,int blockIndex,int blockCount,RandomAccessFile fileIO) 
			throws Exception{
		String tokenStr = getTokenStr();
        setUploadHeader(tokenStr, null);
		
		setUrl(host+"/bput/"+ctx+"/"+HttpUploadUtil.getBlockOffset(hasUploadBytes, 
				uploadFileLength, blockIndex, blockCount));
		mParameter = HttpUploadUtil.nextChunk(fileIO, hasUploadBytes, uploadFileLength, 
				blockIndex, blockCount);
		return new HttpUploadContinueResponse(HttpResponse.STATE_SUCCESS,
				this,this);
	}
	
	private HttpUploadContinueResponse getNextBlockResponse(long hasUploadBytes,long upLoadFileLength,
			int blockIndex,int blockCount,RandomAccessFile fileIO) throws Exception{
		String tokenStr = getTokenStr();
        setUploadHeader(tokenStr, null);
		
		setUrl(HttpUploadUtil.UP_HOST+"/mkblk/"+HttpUploadUtil.makeBlockLength(hasUploadBytes, upLoadFileLength, blockIndex, blockCount));
		mParameter = HttpUploadUtil.nextChunk(fileIO, hasUploadBytes, upLoadFileLength, 
				blockIndex, blockCount);
		mCurBlockIndex++;
		return new HttpUploadContinueResponse(HttpResponse.STATE_SUCCESS,this,this);
	}
	
	private HttpUploadContinueResponse getMakeFileResponse(String host,String[] checkSumArrary,String bucketName,
			RandomAccessFile fileIO,long upLoadFileLength) throws Exception{
		fileIO.close();
		
		String entryUri = bucketName + ":" + getFileName();
		String mimeType = "application/octet-stream";
		String params = "/mimeType/" + HttpUploadUtil.urlsafeEncodeString(mimeType.getBytes());
		
		
        String tokenStr = getTokenStr();
        String contentType = "application/octet-stream";
        setUploadHeader(tokenStr, contentType);
		
		setUrl(host+"/rs-mkfile/"+HttpUploadUtil.urlsafeEncodeString(entryUri.getBytes())+"/fsize/" +
				""+ String.valueOf(upLoadFileLength)+params);
		
		mParameter = HttpUploadUtil.getCheckSums(checkSumArrary);
		
		return new HttpUploadContinueResponse(HttpResponse.STATE_SUCCESS,
				this,this);
	}
	
	private void setFileName(String uploadFileName){
		this.mUploadFileName = uploadFileName;
	}
	
	private String getFileName(){
		return this.mUploadFileName;
	}
	
	private void setUploadHeader(String token,String contentType){
		HashMap<String, Object> headerMap = new HashMap<String, Object>();
		if(token!=null){
			headerMap.put(HttpUploadUtil.REQEUST_HEADER_AUTHORIZTION, "UpToken " + token);
		}
		if(contentType!=null){
			headerMap.put("Content-Type", contentType);
		}
		
		setHeaderMap(headerMap);
	}
	
	
	private HashMap<String, String> getToken() throws Exception{
		if(mTokenCallback==null){
			return AuthPolicy.getDefaultToken();
		}else{
			HashMap<String, String> tokenMap = mTokenCallback.getToken(this);
			if(tokenMap==null){
				AppLog.e(TAG, "httpFrame  threadName:"+Thread.currentThread().getName()+" getToken  token 获取失败");
				throw new Exception("httpFrame getToken  token 获取失败");
			}
			return tokenMap;
		}
	} 
	
	
	private String getTokenStr() throws Exception{
		HashMap<String, String> tokenMap = getToken();
		
		if(tokenMap!=null){
			return tokenMap.get(TokenCallback.KEY_TOKEN);
		}
		AppLog.e(TAG, "httpFrame  threadName:"+Thread.currentThread().getName()+" getTokenStr  token 获取失败");
		return null;
		
	}
	
	
	private void postUploadProgress() throws HttpOperateException{
		final long sumLength = mUpLoadFileLength;
		final long downloadLength = mHasUploadBytes;
		HttpThread.postUploadProgress(this, downloadLength, sumLength);
	}
	
	@Override
	public void onPostExecute(boolean isSuccess) {
		FileUtil.deleteFile(mTempFilePath,true);
	}
	
	@Override
	public boolean equals(Object o) {
		if(o instanceof HttpUploadContinueRequest){
			HttpUploadContinueRequest tempRequest = (HttpUploadContinueRequest)o;
			if(mFilePath.equals(tempRequest.getFilePath())&&
				mFileType.equals(tempRequest.getFileType())&&
				mBizType.equals(tempRequest.getBizType())){
				return true;
			}
		}
		return false;
	}
	
	
	private String getFilePath(){
		return mFilePath;
	}


	@Override
	public void onPreExecute(HttpURLConnection connection)
			throws Exception {
	}
}
