package com.pingan.frame.http.upload;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import org.json.JSONObject;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Bitmap.Config;
import android.graphics.BitmapFactory;
import android.os.Handler;

import com.android.bitmapfun.util.PAImageFetcher;
import com.pingan.frame.http.HttpOperateException;
import com.pingan.frame.http.HttpRequest;
import com.pingan.frame.http.HttpResponse;
import com.pingan.frame.http.listener.HttpProgressListener;
import com.pingan.frame.http.listener.HttpWriteParamCallback;
import com.pingan.frame.http.listener.TokenCallback;
import com.pingan.frame.util.AppFluxLog;
import com.pingan.frame.util.AppLog;
import com.pingan.frame.util.BitmapUtils;
import com.pingan.frame.util.DownloadFileSaveUtil;
import com.pingan.frame.util.DownloadFileSaveUtil.DownloadFileSave;
import com.pingan.frame.util.FileUtil;
import com.pingan.frame.util.InputStreamAt;
import com.pingan.frame.util.MultipartEntity;
import com.pingan.frame.util.Tools;

public class HttpUploadRequest extends HttpRequest implements HttpWriteParamCallback{

	/**
	 * TAG
	 */
	private final String TAG = "HttpUploadRequest";
	/**
	 * 上传文件的路径
	 */
	private String mUploadFilePath ;
	
	/**
	 * 上传的字节数组
	 */
	private byte[] mUploadBytes ;
	
	/**
	 * 上传得bitmap
	 */
	private Bitmap mUploadBitmap ;
	
	/**
	 * 压缩最大高度
	 */
	private int mMaxMeasureHeight = 960;
	/**
	 * 压缩最大宽度
	 */
	private int mMaxMeasureWidth = 960;
	/**
	 * token获取callback
	 */
	private TokenCallback mTokenCallback;
	/**
	 * 文件类型
	 * {@link #FILE_TYPE_OTHER} 等等
	 */
	private final String mFileType;
	/**
	 * 资源类型
	 * {@link #BIZ_TYPE_COMM} 等等
	 */
	private final String mBizType;
	/**
	 * {@link #com.pingan.frame.http.listener.TokenCallback.KEY_TOKEN}
	 */
	private String mHostUrl;;
	/**
	 * 临时保存目录
	 */ 
	private String mTempFilePath;
	/**
	 * 缩略图保存文件路径,终端要用的这个
	 */
	private String mSmallBitmapSaveFolderPath;
	/**
	 * 其他
	 */
	public static final String FILE_TYPE_OTHER = "0";
	/**
	 * 视频 
	 */
	public static final String FILE_TYPE_VIDEO = "3";
	/**
	 * 图片
	 */
	public static final String FILE_TYPE_IMAGE = "1";
	/**
	 * 音频文件
	 */
	public static final String FILE_TYPE_SOUND = "2";
	/**
	 * 公共资源  如头像等等
	 */
	public static final String BIZ_TYPE_COMM = "1";
	/**
	 * 私有资源
	 */
	public static final String BIZ_TYPE_PRIVATE = "0";
	
	private MultipartEntity mMultipartEntity = null;
	
	private Context mContext;
	
	/**
	 * 上传文件的构造函数
	 * @param filePath		要上传得文件文件路径
	 * @param fileType		{@link #FILE_TYPE_IMAGE} 等等
	 * @param bizType		{@link #BIZ_TYPE_COMM} 等等
	 * @param tokenCallback	tokenCallback
	 */
	public HttpUploadRequest(Context context,String filePath,String fileType,String bizType,TokenCallback tokenCallback) {
		super(null, HttpRequest.REQUEST_METHOD_POST);
		this.mUploadFilePath = filePath==null?"":filePath;
		this.mFileType = fileType==null?"":fileType;
		this.mBizType = bizType==null?"":bizType;
		this.mTokenCallback = tokenCallback;
		this.mContext = context;
	}
	
	/**
	 * 图片专用上传通道
	 * @param saveFolderPath	缩略图保存路径
	 * @param filePath			文件路径
	 * @param fileType			上传文件类型
	 * @param bizType			文件共有私有
	 * @param tokenCallback		token
	 */
	public HttpUploadRequest(Context context,String saveFolderPath,String filePath,String fileType,String bizType,TokenCallback tokenCallback) {
		super(null, HttpRequest.REQUEST_METHOD_POST);
		this.mUploadFilePath = filePath==null?"":filePath;
		this.mSmallBitmapSaveFolderPath = saveFolderPath;
		this.mFileType = fileType==null?"":fileType;
		this.mBizType = bizType==null?"":bizType;
		this.mTokenCallback = tokenCallback;
		this.mContext = context;
	}
	
	/**
	 * 上传图片
	 * @param saveFolderPath	缩略图保存路径
	 * @param uploadBitmap		文件路径
	 * @param fileType			上传文件类型
	 * @param bizType			文件共有私有
	 * @param tokenCallback		token
	 */
	public HttpUploadRequest(Context context,String saveFolderPath,Bitmap uploadBitmap,String fileType,String bizType,TokenCallback tokenCallback) {
		super(null, HttpRequest.REQUEST_METHOD_POST);
		this.mSmallBitmapSaveFolderPath = saveFolderPath;
		this.mUploadBitmap = uploadBitmap;
		this.mFileType = fileType==null?"":fileType;
		this.mBizType = bizType==null?"":bizType;
		this.mTokenCallback = tokenCallback;
		this.mContext = context;
	}
	
	/**
	 * 图片上传专用
	 * @param uploadBytes	图片字节数
	 * @param fileType		文件类型
	 * @param bizType		文件共有私有
	 * @param tokenCallback	token
	 */
	public HttpUploadRequest(Context context,String saveFolderPath,byte[] uploadBytes,String fileType,String bizType,TokenCallback tokenCallback) {
		super(null, HttpRequest.REQUEST_METHOD_POST);
		this.mSmallBitmapSaveFolderPath = saveFolderPath;
		this.mUploadBytes = uploadBytes;
		this.mFileType = fileType==null?"":fileType;
		this.mBizType = bizType==null?"":bizType;
		this.mTokenCallback = tokenCallback;
		this.mContext = context;
	}
	
	@Override
	public int getFirstLevel() {
		return mFirstLevel;
	}

	@Override
	public long getReadTimeout() {
		return 60000;
	}

	@Override
	public long getConnectTimeout() {
		return 60000;
	}

	@Override
	public Object getParamData() throws Exception {
		return this;
	}

	@Override
	public HttpResponse onPreExecute() throws Exception {
		//如果是图片
		if(mFileType == FILE_TYPE_IMAGE){
			onPreExecuteBitmap();
		}else{
			String extension = FileUtil.getExtensionName(mUploadFilePath);
			initMultipartEntity(mUploadFilePath,extension);
		}
		
		return null;
	}
	
	private void onPreExecuteBitmap() throws Exception{
		Bitmap updateBitmap = null;
		if(mUploadBytes != null){
			updateBitmap = BitmapUtils.getBitmap(mUploadBytes, mMaxMeasureWidth, mMaxMeasureHeight);
		}
		
		if(mUploadBitmap!=null){
			updateBitmap = BitmapUtils.changeBitmapSize(mMaxMeasureWidth, mMaxMeasureHeight, mUploadBitmap);
		}
		
		if(mUploadFilePath!=null){
			updateBitmap = BitmapUtils.getBitmap(mUploadFilePath, mMaxMeasureWidth,mMaxMeasureHeight,null);
			//如果 是 通过照相被旋转了
			int angle = BitmapUtils.readPictureDegree(mUploadFilePath);
			if(angle>0){
				updateBitmap = BitmapUtils.rotaingImageView(angle, updateBitmap);
			}
		}
		
		String extension = getExtensionName(updateBitmap);
		
		mTempFilePath = getCompressImagePath(updateBitmap, extension);
		
		if(mTempFilePath==null){
			mUploadBytes = FileUtil.compressImageByte(updateBitmap,extension);
			initMultipartEntity(mUploadBytes, extension);
		}else{
			initMultipartEntity(mTempFilePath,extension);
		}
	}
	
	private String getExtensionName(Bitmap bitmap){
		if(mUploadFilePath!=null){
			return FileUtil.getExtensionName(mUploadFilePath);
		}else{
			if(bitmap.getConfig() == Config.RGB_565){
				return BitmapUtils.EXTENSION_IMG_JPEG;
			}else{
				return BitmapUtils.EXTENSION_IMG_PNG;
			}
		}
		
	}
	
	private String getCompressImagePath(Bitmap bm,String extension){
		try {
			String savePath = null;
			long length = FileUtil.getBitmapLength(bm);
			if(mSmallBitmapSaveFolderPath != null&&
					FileUtil.getSDFreeSize(mSmallBitmapSaveFolderPath)>length){
				savePath = mSmallBitmapSaveFolderPath;
			}
			else if(FileUtil.getSDFreeSize(FileUtil.ROOT_PATH)>length){
				savePath = FileUtil.ROOT_PATH;
			}
			
			if(savePath!=null){
				return FileUtil.compressImagePath(bm, savePath, extension);
			}
			
		} catch (IOException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		return null;
	}
	
	private void initMultipartEntity(String uploadFilePath,String extension) throws Exception{
		String fileName = initMultipartEntity(extension);
		mMultipartEntity.addFile("file", "application/octet-stream", fileName, InputStreamAt.fromFile(new File(uploadFilePath),false));
	}
	
	private void initMultipartEntity(byte[] uploadBytes,String extension) throws Exception{
		String fileName = initMultipartEntity(extension);
		mMultipartEntity.addFile("file", "application/octet-stream", fileName, InputStreamAt.fromString(uploadBytes));
	}
	
	private String initMultipartEntity(String extension) throws Exception{
		HashMap<String, String> tokenHashMap = getToken();
		String jid = tokenHashMap.get(TokenCallback.KEY_JID);
		String fileName = "txt_"+jid+FileUtil.getTimeMillisFileName()+"."+extension;
		String bucketName = AuthPolicy.getBucketName(tokenHashMap.get(TokenCallback.KEY_ETC));
		
		//如果是公有文件
		if(!mBizType.equals("0")){
			mHostUrl = tokenHashMap.get(TokenCallback.KEY_ETC);
		}
		
		setUrl(HttpUploadUtil.UP_HOST_URL);
		
		mMultipartEntity = new MultipartEntity(this);
		initHeader(mMultipartEntity.getContentType());
		mMultipartEntity.addField("auth", tokenHashMap.get(TokenCallback.KEY_TOKEN));
		
		String entryUri = HttpUploadUtil.urlsafeEncodeString((bucketName + ":" + fileName).getBytes());
		String mimeType = HttpUploadUtil.urlsafeEncodeString("application/octet-stream".getBytes());
		
		mMultipartEntity.addField("action", "/rs-put/"+entryUri+"/mimeType/"+mimeType);
		
		return fileName;
	}
	
	
	private void initHeader(String contentType){
		HashMap<String, Object> headerMap = new HashMap<String, Object>();
		if(contentType!=null){
			headerMap.put("Content-Type", contentType);
		}
		
		setHeaderMap(headerMap);
	}

	@Override
	public HttpResponse onDoingExecute(HttpURLConnection connection)
			throws Exception {
		InputStream is = connection.getInputStream();
		String jsonStr = Tools.inputStream2String(is);
		JSONObject jsonObject = new JSONObject(jsonStr);
		String hash = jsonObject.getString(HttpUploadResponse.KEY_HASH);
		if(hash!=null&&hash.length()>0){
			HttpUploadResponse response = null; 
			//如果是图片上传
			if(mFileType == FILE_TYPE_IMAGE){
				String smallImagePath = null;
				if(mSmallBitmapSaveFolderPath !=null){
					smallImagePath = mTempFilePath;
				}
				
				response = new HttpUploadResponse(HttpUploadResponse.STATE_SUCCESS, this, 
						smallImagePath, 
						jsonObject, mHostUrl);
				
				if(FileUtil.isFile(mTempFilePath)){
					PAImageFetcher.getInstance().addCacheToStorage(response.getDownloadUrl(), new File(mTempFilePath));
				}
				
			}else{
				response = new HttpUploadResponse(HttpUploadResponse.STATE_SUCCESS, 
						this, jsonObject,mHostUrl);
				
				DownloadFileSaveUtil.saveFile(mContext, response.getDownloadUrl(), 
						new DownloadFileSave(mUploadFilePath));
				
			}
			
			AppFluxLog.i("AppFluxLog", "上行:"+mMultipartEntity.getContentLength()+"   下行:"+connection.getContentLength()+"   时间:"+AppFluxLog.LOG_TIME_FORMAT.format(new Date(System.currentTimeMillis()))+"   url:"+mUrl);
			
			return response;
			
		}else{
			return new HttpUploadResponse(HttpUploadResponse.STATE_FAILED, this);
		}
		
	}

	@Override
	public void onPostExecute(boolean isSuccess) {
		mUploadBytes = null;
		if(mSmallBitmapSaveFolderPath ==null){
			FileUtil.deleteFile(mTempFilePath,true);
		}
	}

	@Override
	public HttpResponse createErrorResponse(int state, HttpRequest httpTask) {
		return new HttpUploadResponse(state,httpTask);
	}

	@Override
	public int getRequestType() {
		return REQUEST_TYPE_UPLOAD;
	}
	
	/**
	 * 是否是完整的可用的任务
	 * @return	如果返回true代表是完整的可执行的任务
	 */
	public boolean isComplete(){
		if(!Tools.isEmpty(mUploadFilePath)){
			if(new File(mUploadFilePath).isFile()){
				return true;
			}else{
				AppLog.e(TAG, "httpFrame isComplete 上传的文件不存在");
				return false;
			}
		}
		else if(mUploadBytes!=null){
			if(mUploadBytes.length>0){
				return true;
			}
			else{
				AppLog.e(TAG, "httpFrame isComplete 上传的字节数组不能为空");
				return false;
			}
		}
		else if(mUploadBitmap!=null){
			if(!mUploadBitmap.isRecycled()){
				return true;
			}
			else{
				AppLog.e(TAG, "httpFrame isComplete 上传的图片已经被释放 无法上传");
				return false;
			}
		}
		AppLog.e(TAG, "httpFrame isComplete 没有可上传的东西");
		return false;
	}

	
	private HashMap<String, String> getToken() throws Exception{
		if(mTokenCallback==null){
			return AuthPolicy.getDefaultToken();
		}else{
			HashMap<String, String> tokenMap = mTokenCallback.getToken(this);
			if(tokenMap==null){
				AppLog.e(TAG, "httpFrame  threadName:"+Thread.currentThread().getName()+" getToken  token 获取失败");
				throw new Exception("httpFrame getToken  token 获取失败");
			}
			return tokenMap;
		}
	}
	
	private String getTokenStr() throws Exception{
		HashMap<String, String> tokenMap = getToken();
		
		if(tokenMap!=null){
			return tokenMap.get(TokenCallback.KEY_TOKEN);
		}
		AppLog.e(TAG, "httpFrame  threadName:"+Thread.currentThread().getName()+" getTokenStr  token 获取失败");
		return null;
		
	}
	
	public void setImgMeasureSize(int maxMeasureWidth,int maxMeasureHeight){
		mMaxMeasureWidth = maxMeasureWidth;
		mMaxMeasureHeight = maxMeasureHeight;
	}
	

	@Override
	public void writeParam(HttpURLConnection connection) throws Exception{
		connection.setFixedLengthStreamingMode((int) mMultipartEntity.getContentLength());
		connection.setUseCaches(false);
		connection.setAllowUserInteraction(false);
		connection.setRequestProperty("Connection", "close");  
		System.setProperty("http.keepAlive", "false");  
		
		mMultipartEntity.writeTo(connection.getOutputStream());
	}
	
	/**
	 * GET
	 * @return {@link #FILE_TYPE_IMAGE} 等等
	 */
	public String getFileType() {
		return mFileType;
	}
	/**
	 * GET
	 * @return	{@link #BIZ_TYPE_COMM} 等等
	 */
	public String getBizType() {
		return mBizType;
	}
	
	public String getFilePath(){
		return mUploadFilePath;
	}
	
	@Override
	public boolean equals(Object o) {
		if(o instanceof HttpUploadRequest){
			HttpUploadRequest tempRequest = (HttpUploadRequest)o;
			if(
				(
					(mUploadFilePath!=null&&Tools.equals(mUploadFilePath, tempRequest.getFilePath()))
					||
					(mUploadBytes!=null&&Tools.equals(mUploadBytes, tempRequest.mUploadBytes))
					||
					(mUploadBitmap!=null&&Tools.equals(mUploadBitmap, tempRequest.mUploadBitmap))
				)&&
					Tools.equals(mFileType, tempRequest.getFileType())&&
					Tools.equals(mBizType, tempRequest.getBizType())){
				return true;
			}
		}
		return false;
	}

	@Override
	public void onPreExecute(HttpURLConnection connection)
			throws Exception {
	}
	
}
