package com.pingan.frame.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Bitmap.CompressFormat;
import android.graphics.Bitmap.Config;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.media.ExifInterface;
import android.text.TextUtils;
import android.util.Base64;

public class BitmapUtils {
	private static final String TAG = "BitmapUtils";
	
	private static final Object OBJECT_LOCKED = new Object();
	
	public static final String EXTENSION_IMG_PNG = "png";
	
	public static final String EXTENSION_IMG_JPEG = "jpg";
	
	
	public static Bitmap changeBitmapSize(int maxWidth,int maxHeight,Bitmap oldBitmap){
		if(oldBitmap==null||oldBitmap.isRecycled()){
			AppLog.e(TAG, "httpFrame 不能改变一个已经被释放的图片的大小");
			return null;
		}
		int[] measureArrary = BitmapUtils.getMeasureSize(oldBitmap.getWidth(), oldBitmap.getHeight(), maxWidth,maxHeight);
		if(measureArrary!=null){
			oldBitmap = BitmapUtils.changeBitmapSizeFixed(measureArrary[0], measureArrary[1], oldBitmap);
		}
		return oldBitmap;
	}
	
	/**
	 * 同步 线程获取图片
	 * @param imgPath	图片本地路径
	 * @param maxWidth	最大宽度
	 * @param maxHeight	最大高度
	 * @return	bitmap
	 */
	public static Bitmap getBitmap(String imgPath,int maxWidth,int maxHeight){
		return getBitmap(imgPath,  maxWidth,maxHeight, null);
	}
	
	/**
	 * 同步 线程获取图片
	 * @param imgPath	图片本地路径
	 * @param maxWidth	最大宽度
	 * @param maxHeight	最大高度
	 * @return	bitmap
	 */
	public static Bitmap getBitmap(byte[] date,int maxWidth,int maxHeight){
		return getBitmap(date, maxWidth,maxHeight , null);
	}
	
	/**
	 * 同步 线程获取图片
	 * @param imgPath	图片本地路径
	 * @param maxWidth	最大宽度
	 * @param maxHeight	最大高度
	 * @return	bitmap
	 */
	public static Bitmap getBitmap(String imgPath,int maxWidth,int maxHeight,BitmapFactory.Options options){
		//避免内存匿出  同步处理
		synchronized (OBJECT_LOCKED) {
			Bitmap bm = BitmapUtils.getBitmapBySampleSize(imgPath, maxWidth,maxHeight,options);
			System.gc();
			return BitmapUtils.changeBitmapSize(maxWidth,maxHeight, bm);
		}
	}
	
	/**
	 * 同步 线程获取图片
	 * @param imgPath	图片本地路径
	 * @param maxWidth	最大宽度
	 * @param maxHeight	最大高度
	 * @return	bitmap
	 */
	public static Bitmap getBitmap(byte[] date,int maxWidth,int maxHeight,BitmapFactory.Options options){
		Bitmap bm = BitmapUtils.getBitmapBySampleSize(date, maxWidth,maxHeight,options);
		return BitmapUtils.changeBitmapSize(maxWidth,maxHeight, bm);
	}
	
	/**
	 * 同步 线程获取图片
	 * @param imgPath	图片本地路径
	 * @param maxWidth	最大宽度
	 * @param maxHeight	最大高度
	 * @return	bitmap
	 */
	public static Bitmap getBitmap(int resource,Context context,int maxWidth,int maxHeight,BitmapFactory.Options options){
		Bitmap bm = BitmapUtils.getBitmapBySampleSize(context, resource, maxWidth, maxHeight, options);
		return BitmapUtils.changeBitmapSize(maxWidth,maxHeight, bm);
	}
	
	/**
	 * 获取图片
	 * @param imgPath	本地路径
	 * @param maxWidth	最大宽度
	 * @param maxHeight	最大高度
	 * @return	bitmap
	 */
	private static Bitmap getBitmapBySampleSize(String imgPath,int maxWidth,int maxHeight,BitmapFactory.Options options){
		BitmapFactory.Options tempOptions = new BitmapFactory.Options();
		tempOptions.inJustDecodeBounds = true;
		
		BitmapFactory.decodeFile(imgPath, tempOptions);
		// Calculate inSampleSize
		int inSampleSize = calculateInSampleSize(tempOptions, maxWidth, maxHeight);
		
		if(options!=null){
			tempOptions = options;
		}
		tempOptions.inSampleSize = inSampleSize;
		// Decode bitmap with inSampleSize set
		tempOptions.inJustDecodeBounds = false;
		
		return BitmapFactory.decodeFile(imgPath, tempOptions);
	}
	
	/**
	 * 获取图片
	 * @param imgPath	本地路径
	 * @param maxWidth	最大宽度
	 * @param maxHeight	最大高度
	 * @return	bitmap
	 */
	private static Bitmap getBitmapBySampleSize(byte[] data,int maxWidth,int maxHeight,BitmapFactory.Options options){
		BitmapFactory.Options tempOptions = new BitmapFactory.Options();
		tempOptions.inJustDecodeBounds = true;
		
		BitmapFactory.decodeByteArray(data, 0, data.length, tempOptions);
		// Calculate inSampleSize
		int inSampleSize = calculateInSampleSize(tempOptions, maxWidth, maxHeight);
		
		if(options!=null){
			tempOptions = options;
		}
		tempOptions.inSampleSize = inSampleSize;
		// Decode bitmap with inSampleSize set
		tempOptions.inJustDecodeBounds = false;

		return BitmapFactory.decodeByteArray(data, 0, data.length, tempOptions);
	}
	
	/**
	 * 获取图片
	 * @param resource	本地路径
	 * @param maxWidth	最大宽度
	 * @param maxHeight	最大高度
	 * @return	bitmap
	 */
	private static Bitmap getBitmapBySampleSize(Context context,int resource,int maxWidth,int maxHeight,BitmapFactory.Options options){
		BitmapFactory.Options tempOptions = new BitmapFactory.Options();
		tempOptions.inJustDecodeBounds = true;
		
		BitmapFactory.decodeResource(context.getResources(), resource, tempOptions);
		// Calculate inSampleSize
		int inSampleSize =  calculateInSampleSize(tempOptions, maxWidth,maxHeight);
		
		if(options!=null){
			tempOptions = options;
		}
		tempOptions.inSampleSize = inSampleSize;
		// Decode bitmap with inSampleSize set
		tempOptions.inJustDecodeBounds = false;

		return BitmapFactory.decodeResource(context.getResources(), resource, tempOptions);
	}
	
	/**
	 * 计算图片的缩放值
	 * 
	 * @param options
	 * @param reqWidth
	 * @param reqHeight
	 * @return
	 */
	private static int calculateInSampleSize(BitmapFactory.Options options,
			int maxWidth,int maxHeight) {
		// Raw height and width of image
		final int height = options.outHeight;
		final int width = options.outWidth;
		
		if(height>=width){
			final int heightRatio = Math.round((float) height
					/ (float) maxHeight);
			return heightRatio;
		}else{
			final int widthRatio = Math.round((float) width 
					/ (float) maxWidth);
			return widthRatio;
		}
	}
	
	
	/**
	 * @param newWidth
	 * @param newHight
	 * @param oldBitmap
	 * @return
	 */
	private static Bitmap changeBitmapSizeFixed(float newWidth,float newHight,Bitmap oldBitmap){
		if(oldBitmap==null||oldBitmap.isRecycled()){
			AppLog.e(TAG, "httpFrame 不能改变一个已经被释放的图片的大小");
			return null;
		}
		float bmpWidth = oldBitmap.getWidth();   
        float bmpHeight = oldBitmap.getHeight();   
        
        if(bmpWidth==newWidth&&newHight==bmpHeight){
        	return oldBitmap;
        }
//        float scaleWidth = newWidth/bmpWidth;   
//        float scaleHeight = newHight/bmpHeight;   
//        
//        Matrix matrix = new Matrix();   
//        matrix.postScale(scaleWidth, scaleHeight);   
//        Bitmap bm=Bitmap.createBitmap(oldBitmap, 0, 0, oldBitmap.getWidth(), oldBitmap.getHeight(),   
//                matrix, true);   
//        Bitmap.createBitmap(source, x, y, width, height)
        Bitmap newBitmap = Bitmap.createScaledBitmap(oldBitmap, (int)newWidth,  (int)newHight, true);
        oldBitmap.recycle();
        System.gc();
        return newBitmap;
	}
	
	public static int[] getMeasureSize(int width,int height,int maxWidth,int maxHeight){
		if(width<=maxWidth||height<=maxHeight){
			return null;
		}
		int newHeight = 0;
		int newWidth = 0;
		if(width>height){
			newHeight = (int) getHeightByNewWidth(maxWidth, width, height);
			newWidth = maxWidth;
		}else{
			newWidth = (int) getWidthByNewHeight(maxHeight, width, height);
			newHeight = maxHeight;
		}
		return new int[]{newWidth,newHeight};
	}
	
	/**
     * 根据新的宽度  获取等比的新的高度
     * @param newWidth   新的宽度
     * @param oldWidth   旧的宽度
     * @param oldHeight  旧的高度
     * @return
     */
    public static float getHeightByNewWidth(float newWidth,float oldWidth,float oldHeight){
    	//640 * 101
		float scale = oldHeight/oldWidth;
		return newWidth*scale;
    }
    
    
    /**
     * 根据新的宽度  获取等比的新的高度
     * @param newWidth   新的宽度
     * @param oldWidth   旧的宽度
     * @param oldHeight  旧的高度
     * @return
     */
    public static float getWidthByNewHeight(float newHeight,float oldWidth,float oldHeight){
    	//640 * 101
		float scale = oldHeight/oldWidth;
		return newHeight/scale;
    }
    
    
    /** 
	 * 读取图片属性：旋转的角度 
	 * @param path 图片绝对路径 
	 * @return degree旋转的角度 
	 */  
	public static int readPictureDegree(String path) {  
		if(Tools.isEmpty(path)||!isFile( path)){
			return 0;
		}
		int degree  = 0;  
	    try {  
	    	ExifInterface exifInterface = new ExifInterface(path);  
	     	int orientation = exifInterface.getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL);  
	     	switch (orientation) {  
	        	case ExifInterface.ORIENTATION_ROTATE_90:  
	            	degree = 90;  
	             	break;  
	        	case ExifInterface.ORIENTATION_ROTATE_180:  
	              	degree = 180;  
	             	break;  
	        	case ExifInterface.ORIENTATION_ROTATE_270:  
	              	degree = 270;  
	             	break;  
	     	}  
		} catch (IOException e) {  
			e.printStackTrace();  
		}
		return degree;  
	}  
	
	public static boolean isFile(String path){
		try{
			if(new File(path).isFile()){
				return true;
			}
		}catch(Exception ex){
			ex.printStackTrace();
		}
		return false;
		
	}
	

	/**
	 * 旋转图片
	 * @param angle
	 * @param bitmap
	 * @return Bitmap
	 */
	public static Bitmap rotaingImageView(int angle , Bitmap bitmap) {
	      //旋转图片 动作
		Matrix matrix = new Matrix();;
	    matrix.postRotate(angle);
	    // 创建新的图片
	    Bitmap resizedBitmap = Bitmap.createBitmap(bitmap, 0, 0,
	      		bitmap.getWidth(), bitmap.getHeight(), matrix, true);
	    bitmap.recycle();
		return resizedBitmap;
	}
	
	/**
	 * 压缩图片异步处理
	 * @param savePath	保存文件夹路径
	 * @param imagePath	被压缩的图片
	 * @param maxWidth	最大宽度
	 * @param maxHeight	最大高度
	 * @param callback	回调
	 */
	public static void compressBitmap(final String savePath,final String imagePath,
			final int maxWidth,final int maxHeight,final CompressBitmapCallback callback){
		compressBitmap(savePath,null, imagePath, maxWidth, maxHeight, callback);
	}
	/**
	 * 压缩图片异步处理
	 * @param savePath	保存文件夹路径
	 * @param bitmap	被压缩的图片
	 * @param maxWidth	最大宽度
	 * @param maxHeight	最大高度
	 * @param callback	回调
	 */
	public static void compressBitmap(final String savePath,final Bitmap bitmap,
			final int maxWidth,final int maxHeight,final CompressBitmapCallback callback){
		compressBitmap(savePath, bitmap,null, maxWidth, maxHeight, callback);
	}
	
	private static void compressBitmap(final String savePath,final Bitmap bitmap,final String imagePath,
			final int maxWidth,final int maxHeight,final CompressBitmapCallback callback){
		new Thread(){
			@Override
            public void run() {
                if (!TextUtils.isEmpty(imagePath)) {
                    Bitmap bm = null;
                    if (imagePath != null) {
                        if (maxWidth > 0 && maxHeight > 0) {
                            bm = BitmapUtils.getBitmap(imagePath, maxWidth, maxHeight, null);
                        } else {
                            bm = BitmapFactory.decodeFile(imagePath);
                        }
                    } else if (bitmap != null && !bitmap.isRecycled()) {
                        bm = BitmapUtils.changeBitmapSize(maxWidth, maxHeight, bitmap);
                    } else {
                        callback.compress(null);
                        return;
                    }

                    if (bm != null && !bm.isRecycled()) {
                        int angle = BitmapUtils.readPictureDegree(imagePath);
                        if (angle > 0) {
                            bm = BitmapUtils.rotaingImageView(angle, bm);
                        }

                        String extensionName = FileUtil.getExtensionName(imagePath);
                        CompressFormat format = CompressFormat.PNG;
                        if (BitmapUtils.EXTENSION_IMG_JPEG.equalsIgnoreCase(extensionName)) {
                            format = CompressFormat.JPEG;
                        } else {
                            format = CompressFormat.PNG;
                        }
                        String localPath = FileUtil.saveBitmap(savePath, bm, format, extensionName);
                        callback.compress(localPath);
                    }
                    if (bm != null) {
                        bm.recycle();
                    }
                }
            }
		}.start();
	}
	
	
	public static void getBitmapBase64(final String imagePath,
			final CompressBase64Callback callback){
		new Thread(){
			@Override
			public void run() {
				final int maxWidth = 600;
				final int maxHeight = 800;
				if(!TextUtils.isEmpty(imagePath)){
					Bitmap bm = null;
					if(imagePath!=null){
						if(maxWidth>0&&maxHeight>0){
							bm = BitmapUtils.getBitmap(imagePath, maxWidth,maxHeight,null);
						}else{
							bm = BitmapFactory.decodeFile(imagePath);
						}
					}
					
					if(bm!=null&&!bm.isRecycled()){
						int angle = BitmapUtils.readPictureDegree(imagePath);
						if(angle>0){
							bm = BitmapUtils.rotaingImageView(angle, bm);
						}
						
					}
					
					if(bm!=null&&!bm.isRecycled()){
						callback.finish(bitmapToBase64(bm));
					}else{
						callback.finish(null);
					}
					
					
				}
			}
		}.start();
	}
	
	/**
	 * bitmap转为base64
	 * @param bitmap
	 * @return
	 */
	public static String bitmapToBase64(Bitmap bitmap) {

		String result = null;
		ByteArrayOutputStream baos = null;
		try {
			if (bitmap != null) {
				baos = new ByteArrayOutputStream();
				bitmap.compress(Bitmap.CompressFormat.JPEG, 100, baos);

				byte[] bitmapBytes = baos.toByteArray();
				result = Base64.encodeToString(bitmapBytes, Base64.DEFAULT);
				
				result = result.replaceAll("\n", "");
			}
		} catch (Exception e) {
			e.printStackTrace();
		} finally {
			try {
				if (baos != null) {
					baos.flush();
					baos.close();
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		return result;
	}

	/**
	 * base64转为bitmap
	 * @param base64Data
	 * @return
	 */
	public static Bitmap base64ToBitmap(String base64Data) {
		byte[] bytes = Base64.decode(base64Data, Base64.DEFAULT);
		return BitmapFactory.decodeByteArray(bytes, 0, bytes.length);
	}
	
	public static interface CompressBitmapCallback{
		public void compress(String imagePath);
	}
	
	public static interface CompressBase64Callback{
		/**
		 * 
		 * @param imagePath
		 */
		public void finish(String base64);
	}
}
