package com.pingan.frame.util;

import java.io.*;
import java.util.Arrays;
import java.util.zip.CRC32;

public class InputStreamAt implements Closeable {
	private RandomAccessFile mFileStream;
	private byte[] mData;

	private File mTmpFile;
	private boolean mClosed;
	private boolean mDelWhenClose = false;
	private long mCrc32 = -1;
	private long mLength = -1;


	public static InputStreamAt fromFile(File f,boolean delWhenClose) throws FileNotFoundException {
		InputStreamAt isa = new InputStreamAt(f);
		return isa;
	}

	public static InputStreamAt fromString(byte[] bytes) {
		return new InputStreamAt(bytes);
	}

	public InputStreamAt(File file) throws FileNotFoundException {
		this(file, false);
	}

	public InputStreamAt(File file, boolean delWhenClose) throws FileNotFoundException {
		mTmpFile = file;
		mDelWhenClose = delWhenClose;
		mFileStream = new RandomAccessFile(mTmpFile, "r");
	}

	public InputStreamAt(byte[] data) {
		mData = data;
	}

	public long getCrc32(long offset, int length) throws IOException {
		CRC32 crc32 = new CRC32();
		byte[] data = read(offset, length);
		crc32.update(data);
		return crc32.getValue();
	}

	public long crc32() throws IOException {
		if (mCrc32 >= 0) return mCrc32;
		CRC32 crc32 = new CRC32();
		long index = 0;
		int blockSize = 128 * 1024;
		long length = length();
		while (index < length) {
			int size = length-index>=blockSize ? blockSize : (int) (length-index);

			byte[] data = read(index, size);
			if (data == null) return -1;
			crc32.update(data);
			index += size;
		}
		mCrc32 = crc32.getValue();
		return mCrc32;
	}

	public long length() {
		if (mLength >= 0) return mLength;
		if (mData != null) {
			mLength = mData.length;
			return mLength;
		}

		if (mFileStream != null) {
			try {
				mLength = mFileStream.length();
				return mLength;
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
		return -1;
	}


	public byte[] read(long offset, int length) throws IOException {
		if (mClosed) return null;
		if (mFileStream != null) {
			return fileStreamRead(offset, length);
		}
		if (mData != null) {
			byte[] ret = new byte[length];
			System.arraycopy(mData, (int) offset, ret, 0, length);
			return ret;
		}

		return null;
	}

	protected byte[] fileStreamRead(long offset, int length) throws IOException {
		if (mFileStream == null) return null;
		byte[] data = new byte[length];

		int read;
		int totalRead = 0;
		synchronized (data) {
			mFileStream.seek(offset);
			do {
				read = mFileStream.read(data, totalRead, length);
				if (read <= 0) break;
				totalRead += read;
			} while (length > totalRead);
		}

		if (totalRead != data.length) {
			data = Arrays.copyOfRange(data, 0, totalRead);
		}
		return data;
	}

	@Override
	public synchronized void close(){
		if (mClosed) return;
		mClosed = true;

		if (mFileStream != null) {
			try {
				mFileStream.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}

		if (mTmpFile != null && mDelWhenClose) {
			mTmpFile.delete();
		}
	}

	public int read(byte[] data) throws IOException {
		return mFileStream.read(data);
	}

}
