package im.zego.zegoexpress;

import android.app.Application;
import android.graphics.*;
import im.zego.zegoexpress.*;
import im.zego.zegoexpress.callback.*;
import im.zego.zegoexpress.constants.*;
import im.zego.zegoexpress.entity.*;
import im.zego.zegoexpress.internal.*;
import java.nio.*;
import java.util.*;
import org.json.*;

/**
 * Zego MediaPlayer.
 *
 * Yon can use ZegoMediaPlayer to play media resource files on the local or remote server, and can mix the sound of the media resource files that are played into the publish stream to achieve the effect of background music.
 */
public abstract class ZegoMediaPlayer {

    /**
     * Set event callback handler for media player.
     *
     * Developers can change the player UI widget according to the related event callback of the media player
     *
     * @param handler Media player event callback object
     */
    public abstract void setEventHandler(IZegoMediaPlayerEventHandler handler);

    /**
     * Set video callback handler.
     *
     * Developers can set this callback to throw the video data of the media resource file played by the media player
     * Note: This function is only available in ZegoExpressVideo SDK!
     *
     * @param handler Video event callback object for media player
     * @param format Video frame format for video data
     */
    public abstract void setVideoHandler(IZegoMediaPlayerVideoHandler handler,
                                         ZegoVideoFrameFormat format);

    /**
     * Set audio callback handler.
     *
     * You can set this callback to throw the audio data of the media resource file played by the media player
     *
     * @param handler Audio event callback object for media player
     */
    public abstract void setAudioHandler(IZegoMediaPlayerAudioHandler handler);

    /**
     * Load media resource.
     *
     * Available: since 1.3.4
     * Description: Load media resources.
     * Use case: Developers can load the absolute path to the local resource or the URL of the network resource incoming. Android developers can pass parameters through Uri, namely：String path = uri.toString();
     * When to call: It can be called after the engine by [createEngine] has been initialized and the media player has been created by [createMediaPlayer].
     * Related APIs: Resources can be loaded through the [loadResourceWithPosition] or [loadResourceFromMediaData] function.
     *
     * @param path The absolute resource path or the URL of the network resource and cannot be null or "".
     * @param callback Notification of resource loading results
     */
    public abstract void loadResource(String path, IZegoMediaPlayerLoadResourceCallback callback);

    /**
     * Load media resource.
     *
     * Available: since 2.14.0
     * Description: Load media resources, and specify the progress, in milliseconds, at which playback begins.
     * Use case: Developers can load the absolute path to the local resource or the URL of the network resource incoming. Android developers can pass parameters through Uri, namely：String path = uri.toString();
     * When to call: It can be called after the engine by [createEngine] has been initialized and the media player has been created by [createMediaPlayer].
     * Related APIs: Resources can be loaded through the [loadResource] or [loadResourceFromMediaData] function.
     * Caution: When [startPosition] exceeds the total playing time, it will start playing from the beginning.
     *
     * @param path The absolute resource path or the URL of the network resource and cannot be null or "".
     * @param startPosition The progress at which the playback started.
     * @param callback Notification of resource loading results
     */
    public abstract void loadResourceWithPosition(String path, long startPosition,
                                                  IZegoMediaPlayerLoadResourceCallback callback);

    /**
     * Load media resource.
     *
     * Available: since 2.10.0
     * Description: Load binary audio data.
     * Use case: Developers do not want to cache the audio data locally, and directly transfer the audio binary data to the media player, directly load and play the audio.
     * When to call: It can be called after the engine by [createEngine] has been initialized and the media player has been created by [createMediaPlayer].
     * Related APIs: Resources can be loaded through the [loadResource] or [loadResourceWithPosition] function.
     * Caution: When [startPosition] exceeds the total playing time, it will start playing from the beginning.
     *
     * @param mediaData Binary audio data.
     * @param startPosition Position of starting playback, in milliseconds.
     * @param callback Notification of resource loading results.
     */
    public abstract void loadResourceFromMediaData(byte[] mediaData, long startPosition,
                                                   IZegoMediaPlayerLoadResourceCallback callback);

    /**
     * Load copyrighted music resource.
     *
     * Available: since 2.14.0
     * Description: Load media resources, and specify the progress, in milliseconds, at which playback begins.
     * Use case: Developers can load the resource ID of copyrighted music.
     * When to call: It can be called after the engine by [createEngine] has been initialized and the media player has been created by [createMediaPlayer].
     * Caution: When [startPosition] exceeds the total playing time, it will start playing from the beginning.
     *
     * @param resourceID The resource ID obtained from the copyrighted music module.
     * @param startPosition The progress at which the playback started.
     * @param callback Notification of resource loading results
     */
    public abstract void
    loadCopyrightedMusicResourceWithPosition(String resourceID, long startPosition,
                                             IZegoMediaPlayerLoadResourceCallback callback);

    /**
     * Start playing.
     *
     * You need to load resources before playing
     */
    public abstract void start();

    /**
     * Stop playing.
     */
    public abstract void stop();

    /**
     * Pause playing.
     */
    public abstract void pause();

    /**
     * resume playing.
     */
    public abstract void resume();

    /**
     * Set the specified playback progress.
     *
     * Unit is millisecond
     *
     * @param millisecond Point in time of specified playback progress
     * @param callback the result notification of set the specified playback progress
     */
    public abstract void seekTo(long millisecond, IZegoMediaPlayerSeekToCallback callback);

    /**
     * Whether to repeat playback.
     *
     * @param enable repeat playback flag. The default is false.
     */
    public abstract void enableRepeat(boolean enable);

    /**
     * Set the speed of play.
     *
     * Available since: 2.12.0
     * Description: Set the playback speed of the player.
     * When to call: You should load resource before invoking this function.
     * Restrictions: None.
     * Related APIs: Resources can be loaded through the [loadResource] function.
     *
     * @param speed The speed of play. The range is 0.5 ~ 2.0. The default is 1.0.
     */
    public abstract void setPlaySpeed(float speed);

    /**
     * Whether to mix the player's sound into the main stream channel being published.
     *
     * @param enable Aux audio flag. The default is false.
     */
    public abstract void enableAux(boolean enable);

    /**
     * Whether to play locally silently.
     *
     * If [enableAux] switch is turned on, there is still sound in the publishing stream. The default is false.
     *
     * @param mute Mute local audio flag, The default is false.
     */
    public abstract void muteLocal(boolean mute);

    /**
     * Set the view of the player playing video.
     * Note: This function is only available in ZegoExpressVideo SDK!
     *
     * @param canvas Video rendered canvas object
     */
    public abstract void setPlayerCanvas(ZegoCanvas canvas);

    /**
     * Set mediaplayer volume. Both the local play volume and the publish volume are set.
     *
     * @param volume The range is 0 ~ 200. The default is 60.
     */
    public abstract void setVolume(int volume);

    /**
     * Set mediaplayer local play volume.
     *
     * @param volume The range is 0 ~ 200. The default is 60.
     */
    public abstract void setPlayVolume(int volume);

    /**
     * Set mediaplayer publish volume.
     *
     * @param volume The range is 0 ~ 200. The default is 60.
     */
    public abstract void setPublishVolume(int volume);

    /**
     * Set playback progress callback interval.
     *
     * This function can control the callback frequency of [onMediaPlayerPlayingProgress]. When the callback interval is set to 0, the callback is stopped. The default callback interval is 1s
     * This callback are not returned exactly at the set callback interval, but rather at the frequency at which the audio or video frames are processed to determine whether the callback is needed to call
     *
     * @param millisecond Interval of playback progress callback in milliseconds
     */
    public abstract void setProgressInterval(long millisecond);

    /**
     * Gets the current local playback volume of the mediaplayer, the range is 0 ~ 200, with the default value of 60.
     */
    public abstract int getPlayVolume();

    /**
     * Gets the current publish volume of the mediaplayer, the range is 0 ~ 200, with the default value of 60.
     */
    public abstract int getPublishVolume();

    /**
     * Get the total progress of your media resources.
     *
     * You should load resource before invoking this function, otherwise the return value is 0
     *
     * @return Unit is millisecond
     */
    public abstract long getTotalDuration();

    /**
     * Get current playing progress.
     *
     * You should load resource before invoking this function, otherwise the return value is 0
     */
    public abstract long getCurrentProgress();

    /**
     * Get the number of audio tracks of the playback file.
     */
    public abstract int getAudioTrackCount();

    /**
     * Set the audio track of the playback file.
     *
     * @param index Audio track index, the number of audio tracks can be obtained through the [getAudioTrackCount] function.
     */
    public abstract void setAudioTrackIndex(int index);

    /**
     * Setting up the specific voice changer parameters.
     *
     * @param audioChannel The audio channel to be voice changed
     * @param param Voice changer parameters
     */
    public abstract void setVoiceChangerParam(ZegoMediaPlayerAudioChannel audioChannel,
                                              ZegoVoiceChangerParam param);

    /**
     * Get the current playback status.
     */
    public abstract ZegoMediaPlayerState getCurrentState();

    /**
     * Get media player index.
     */
    public abstract int getIndex();

    /**
     * Take a screenshot of the current playing screen of the media player.
     *
     * Only in the case of calling [setPlayerCanvas] to set the display controls and the playback state, can the screenshot be taken normally
     * Note: This function is only available in ZegoExpressVideo SDK!
     *
     * @param callback The callback of the screenshot of the media player playing screen
     */
    public abstract void takeSnapshot(IZegoMediaPlayerTakeSnapshotCallback callback);

    /**
     * Enable accurate seek and set relevant attributes.
     *
     * Available since: 2.4.0
     * Description: The timestamp specified by normal seek may not an I frame, and then returns the I frame near the specified timestamp, which is not so accurate. But the accurate seek, when the specified timestamp is not an I frame, it will use the I frame near the specified timestamp to decode the frame of the specified timestamp.
     * Use cases: When user needs to seek to the specified timestamp accurately.
     * When to call: The setting must be called before [loadResource], and it will take effect during the entire life cycle of the media player.
     *
     * @param enable Whether to enable accurate seek
     * @param config The property setting of accurate seek, only valid when enable is true.
     */
    public abstract void enableAccurateSeek(boolean enable, ZegoAccurateSeekConfig config);

    /**
     * Set the maximum cache duration and cache data size of web materials.
     *
     * The setting must be called before loading the resource, and it will take effect during the entire life cycle of the media player.
     * Time and size are not allowed to be 0 at the same time. The SDK internal default time is 5000, and the size is 15*1024*1024 byte.When one of time and size reaches the set value first, the cache will stop.
     *
     * @param time The maximum length of the cache time, in ms, the SDK internal default is 5000; the effective value is greater than or equal to 2000; if you fill in 0, it means no limit.
     * @param size The maximum size of the cache, the unit is byte, the internal default size of the SDK is 15*1024*1024 byte; the effective value is greater than or equal to 5000000, if you fill in 0, it means no limit.
     */
    public abstract void setNetWorkResourceMaxCache(int time, int size);

    /**
     * Get the playable duration and size of the cached data of the current network material cache queue
     *
     * @return Returns the current cached information, including the length of time the data can be played and the size of the cached data.
     */
    public abstract ZegoNetWorkResourceCache getNetWorkResourceCache();

    /**
     * Use this interface to set the cache threshold that the media player needs to resume playback. The SDK default value is 5000ms，The valid value is greater than or equal to 1000ms
     *
     * @param threshold Threshold that needs to be reached to resume playback, unit ms.
     */
    public abstract void setNetWorkBufferThreshold(int threshold);

    /**
     * Whether to enable sound level monitoring.
     *
     * Available since: 2.15.0
     * Description: Whether to enable sound level monitoring.
     * When to call: It can be called after the engine by [createEngine] has been initialized and the media player has been created by [createMediaPlayer].
     * Restrictions: None.
     * Related callbacks: After it is turned on, user can use the [onMediaPlayerSoundLevelUpdate] callback to monitor sound level updates.
     *
     * @param enable Whether to enable monitoring, true is enabled, false is disabled.
     * @param millisecond Monitoring time period of the sound level, in milliseconds, has a value range of [100, 3000].
     */
    public abstract void enableSoundLevelMonitor(boolean enable, int millisecond);

    /**
     * Whether to enable frequency spectrum monitoring.
     *
     * Available since: 2.15.0
     * Description: Whether to enable frequency spectrum monitoring.
     * When to call: It can be called after the engine by [createEngine] has been initialized and the media player has been created by [createMediaPlayer].
     * Restrictions: None.
     * Related APIs: After it is turned on, user can use the [onMediaPlayerFrequencySpectrumUpdate] callback to monitor frequency spectrum updates.
     *
     * @param enable Whether to enable monitoring, true is enabled, false is disabled.
     * @param millisecond Monitoring time period of the frequency spectrum, in milliseconds, has a value range of [100, 3000].
     */
    public abstract void enableFrequencySpectrumMonitor(boolean enable, int millisecond);

    /**
     * Set the playback channel.
     *
     * Available since: 2.20.0
     * Description: Set the playback channel.
     * When to call: It can be called after the engine by [createEngine] has been initialized and the media player has been created by [createMediaPlayer].
     * Restrictions: None.
     *
     * @param audioChannel Playback channel, the default is ZegoMediaPlayerAudioChannelAll.
     */
    public abstract void setActiveAudioChannel(ZegoMediaPlayerAudioChannel audioChannel);

    /**
     * Clears the last frame of the playback control that remains on the control after playback ends.
     *
     * Available since: 2.20.0
     * Description: Clears the last frame of the playback control that remains on the control after playback ends.
     * When to call: It can be called after the engine by [createEngine] has been initialized and the media player has been created by [createMediaPlayer].
     * Restrictions: The interface call takes effect only when the media player ends playing.
     */
    public abstract void clearView();
}
