package im.zego.zegoexpress;

import android.app.Application;
import android.graphics.*;
import im.zego.zegoexpress.*;
import im.zego.zegoexpress.callback.*;
import im.zego.zegoexpress.constants.*;
import im.zego.zegoexpress.entity.*;
import im.zego.zegoexpress.internal.*;
import java.nio.*;
import java.util.*;
import org.json.*;

public abstract class ZegoCopyrightedMusic {
    /**
     * set copyrighted music event handler.
     *
     * Available since: 2.13.0
     * Description: Set the callback function of the copyrighted music module, which can receive callback notifications related to song playback status .
     * When to call: After create the copyrighted music [createCopyrightedMusic].
     *
     * @param handler The object used to receive copyrighted music callbacks.
     */
    public abstract void setEventHandler(IZegoCopyrightedMusicEventHandler handler);

    /**
     * Initialize the copyrighted music module.
     *
     * Available since: 2.13.0
     * Description: Initialize the copyrighted music so that you can use the function of the copyrighted music later.
     * When to call: After initializing the copyrighted music [createCopyrightedMusic].
     * Caution: 1. The real user information must be passed in, otherwise the song resources cannot be obtained for playback. 2. The user ID set when initializing copyrighted music needs to be the same as the user ID set when logging in to the room.
     *
     * @param config the copyrighted music configuration.
     * @param callback init result
     */
    public abstract void initCopyrightedMusic(ZegoCopyrightedMusicConfig config,
                                              IZegoCopyrightedMusicInitCallback callback);

    /**
     * Get cache size.
     *
     * Available since: 2.13.0
     * Description: When using this module, some cache files may be generated, and the size of the cache file can be obtained through this interface.
     * Use case: Used to display the cache size of the App.
     * When to call: After initializing the copyrighted music [createCopyrightedMusic].
     *
     * @return cache file size, in byte.
     */
    public abstract long getCacheSize();

    /**
     * Clear cache.
     *
     * Available since: 2.13.0
     * Description: When using this module, some cache files may be generated, which can be cleared through this interface.
     * Use case: Used to clear the cache of the App.
     * When to call: After initializing the copyrighted music [createCopyrightedMusic].
     */
    public abstract void clearCache();

    /**
     * Send extended feature request.
     *
     * Available since: 2.13.0
     * Description: Initialize the copyrighted music so that you can use the function of the copyrighted music later.
     * Use case: Used to get a list of songs.
     * When to call: After initializing the copyrighted music success [initCopyrightedMusic].
     *
     * @param command request command, details about the commands supported.
     * @param params request parameters, each request command has corresponding request parameters.
     * @param callback send extended feature request result
     */
    public abstract void
    sendExtendedRequest(String command, String params,
                        IZegoCopyrightedMusicSendExtendedRequestCallback callback);

    /**
     * Get lyrics in lrc format.
     *
     * Available since: 2.13.0
     * Description: Get lyrics in lrc format, support parsing lyrics line by line.
     * Use case: Used to display lyrics line by line.
     * When to call: After initializing the copyrighted music success [initCopyrightedMusic].
     *
     * @param songID the ID of the song or accompaniment, the song and accompaniment of a song share the same ID.
     * @param callback get lyrics result
     */
    public abstract void getLrcLyric(String songID,
                                     IZegoCopyrightedMusicGetLrcLyricCallback callback);

    /**
     * Get lyrics in krc format.
     *
     * Available since: 2.13.0
     * Description: Get lyrics in krc format, support parsing lyrics word by word.
     * Use case: Used to display lyrics word by word.
     * When to call: After initializing the copyrighted music success [initCopyrightedMusic].
     *
     * @param krcToken The krcToken obtained by calling requestAccompaniment.
     * @param callback get lyrics result.
     */
    public abstract void
    getKrcLyricByToken(String krcToken, IZegoCopyrightedMusicGetKrcLyricByTokenCallback callback);

    /**
     * Request a song.
     *
     * Available since: 2.13.0
     * Description: By requesting a song, you can not only obtain basic information about a song (such as duration, song name, and artist), but also obtain the resource ID for local playback, share_token for sharing with others, and related authentication information. Support by the time, by the user monthly, by the room monthly subscription three ways.
     * Use case: Get copyrighted songs for local playback and sharing.
     * When to call: After initializing the copyrighted music success [initCopyrightedMusic].
     * Caution: This interface will trigger billing. A song may have three sound qualities: normal, high-definition, and lossless. Each sound quality has a different resource file, and each resource file has a unique resource ID.
     *
     * @param config request configuration.
     * @param callback request a song result
     */
    public abstract void requestSong(ZegoCopyrightedMusicRequestConfig config,
                                     IZegoCopyrightedMusicRequestSongCallback callback);

    /**
     * Request accompaniment.
     *
     * Available since: 2.13.0
     * Description: You can get the accompaniment resources of the song corresponding to the songID, including resource_id, krc_token, share_token, etc. Supports click-by-point accompaniment.
     * Use case: Get copyrighted accompaniment for local playback and sharing.
     * When to call: After initializing the copyrighted music success [initCopyrightedMusic].
     * Caution: This interface will trigger billing.
     *
     * @param config request configuration.
     * @param callback request accompaniment result.
     */
    public abstract void
    requestAccompaniment(ZegoCopyrightedMusicRequestConfig config,
                         IZegoCopyrightedMusicRequestAccompanimentCallback callback);

    /**
     * Request accompaniment clip.
     *
     * Available since: 2.13.0
     * Description: You can get the climax clip resources of the song corresponding to the songID, including resource_id, krc_token, share_token, etc. Supports accompaniment climax clips by pay-per-use.
     * Use case: Get copyrighted accompaniment clip for local playback and sharing.
     * When to call: After initializing the copyrighted music success [initCopyrightedMusic].
     * Caution: This interface will trigger billing.
     *
     * @param config request configuration.
     * @param callback request accompaniment clip result.
     */
    public abstract void
    requestAccompanimentClip(ZegoCopyrightedMusicRequestConfig config,
                             IZegoCopyrightedMusicRequestAccompanimentClipCallback callback);

    /**
     * Get a song or accompaniment.
     *
     * Available since: 2.13.0
     * Description: After the user successfully obtains the song/accompaniment/climax clip resource, he can get the corresponding shareToken, share the shareToken with other users, and other users call this interface to obtain the shared music resources.
     * Use case: In the online KTV scene, after receiving the song or accompaniment token shared by the lead singer, the chorus obtains the corresponding song or accompaniment through this interface, and then plays it on the local end.
     * When to call: After initializing the copyrighted music success [initCopyrightedMusic].
     *
     * @param shareToken access the corresponding authorization token for a song or accompaniment.
     * @param callback get a song or accompaniment result.
     */
    public abstract void getMusicByToken(String shareToken,
                                         IZegoCopyrightedMusicGetMusicByTokenCallback callback);

    /**
     * Download song or accompaniment.
     *
     * Available since: 2.13.0
     * Description: Download a song or accompaniment. It can only be played after downloading successfully.
     * Use case: Get copyrighted accompaniment for local playback and sharing.
     * When to call: After initializing the copyrighted music success [initCopyrightedMusic].
     * Caution: Loading songs or accompaniment resources is affected by the network.
     *
     * @param resourceID the resource ID corresponding to the song or accompaniment.
     * @param callback download song or accompaniment result.
     */
    public abstract void download(String resourceID,
                                  IZegoCopyrightedMusicDownloadCallback callback);

    /**
     * Query the resource's cache is existed or not.
     *
     * Available since: 2.13.0
     * Description: Query the resource is existed or not.
     * Use case: Can be used to check the resource's cache is existed or not
     * When to call: After initializing the copyrighted music success [initCopyrightedMusic].
     *
     * @param songID the ID of the song or accompaniment, the song and accompaniment of a song share the same ID.
     * @param type the song resource type.
     */
    public abstract boolean queryCache(String songID, ZegoCopyrightedMusicType type);

    /**
     * Get the playing time of a song or accompaniment file.
     *
     * Available since: 2.13.0
     * Description: Get the playing time of a song or accompaniment file.
     * Use case: Can be used to display the playing time information of the song or accompaniment on the view.
     * When to call: After initializing the copyrighted music success [initCopyrightedMusic].
     *
     * @param resourceID the resource ID corresponding to the song or accompaniment.
     */
    public abstract long getDuration(String resourceID);

    /**
     * Set the difficulty level of scoring.
     *
     * Available since: 2.21.0
     * Description: Users can set the scoring difficulty level through this function.
     * When to call: After calling [initCopyrightedMusic] to initialize copyrighted music successfully, call [startScore] to start scoring.
     * Default value: When this function is not called, the difficulty level of scoring is 4.
     * Restrictions: This function does not support dynamic settings. After calling this function successfully, the next call to [startScore] will take effect.
     *
     * @param level The difficulty level of scoring. The level ranges from 0 to 4. The scoring difficulty decreases from 0 to 4.
     */
    public abstract void setScoringLevel(int level);

    /**
     * Start scoring.
     *
     * Available since: 2.15.0
     * Description: Start the scoring function.After starting scoring, the scoring result OnCurrentPitchValueUpdate callback will be received according to the set callback time interval.
     * Use case: Can be used to display the singing score on the view.
     * When to call: After obtaining krc verbatim lyrics and playing the accompaniment resources of copyrighted music.
     * Restrictions: Only support use this api after [startPublishingStream].
     *
     * @param resourceID the resource ID corresponding to the song or accompaniment.
     * @param pitchValueInterval the time interval of real-time pitch line callback, in milliseconds, the default is 50 milliseconds.
     */
    public abstract int startScore(String resourceID, int pitchValueInterval);

    /**
     * Pause scoring.
     *
     * Available since: 2.15.0
     * Description: Pause ongoing scoring,will stop the [OnCurrentPitchValueUpdate] callback.
     * Use case: You can call this interface to pause the scoring function while scoring.
     * When to call: It can be called while grading. 
     *
     * @param resourceID the resource ID corresponding to the song or accompaniment.
     */
    public abstract int pauseScore(String resourceID);

    /**
     * Resume scoring.
     *
     * Available since: 2.15.0
     * Description: Resume currently paused scoring.
     * Use case: When there is currently paused scoring, this interface can be called to resume the scoring function.
     * When to call: It can be called when there is currently a paused scoring. 
     *
     * @param resourceID the resource ID corresponding to the song or accompaniment.
     */
    public abstract int resumeScore(String resourceID);

    /**
     * Stop scoring.
     *
     * Available since: 2.15.0
     * Description: End the current rating.The [OnCurrentPitchValueUpdate] callback will be stopped, but the average or total score can still be obtained normally.
     * Use case: You can call this interface to end the scoring while scoring.
     * When to call: It can be called while grading. 
     *
     * @param resourceID the resource ID corresponding to the song or accompaniment.
     */
    public abstract int stopScore(String resourceID);

    /**
     * Reset scoring.
     *
     * Available since: 2.15.0
     * Description: Reset the scores that have already been performed,The [OnCurrentPitchValueUpdate] callback will be stopped and the average or total score will be 0.
     * Use case: Often used in scenes where the same song is re-sung.
     * When to call: It can be called after scoring has been performed. 
     *
     * @param resourceID the resource ID corresponding to the song or accompaniment.
     */
    public abstract int resetScore(String resourceID);

    /**
     * Get the score of the previous sentence.
     *
     * Available since: 2.15.0
     * Description: Get the score of the previous sentence.
     * Use case: Can be used to display the score of each sentence on the view.
     * When to call: After obtaining krc verbatim lyrics and playing the accompaniment resources of copyrighted music.The user gets it once after singing each sentence.
     *
     * @param resourceID the resource ID corresponding to the song or accompaniment.
     */
    public abstract int getPreviousScore(String resourceID);

    /**
     * Get average score.
     *
     * Available since: 2.15.0
     * Description: Get the average score.
     * Use case: Can be used to display the average score on the view.
     * When to call: After obtaining krc verbatim lyrics and playing the accompaniment resources of copyrighted music.
     *
     * @param resourceID the resource ID corresponding to the song or accompaniment.
     */
    public abstract int getAverageScore(String resourceID);

    /**
     * Get total score .
     *
     * Available since: 2.15.0
     * Description: Get the total score.
     * Use case: Can be used to display the total score on the view.
     * When to call: After obtaining krc verbatim lyrics and playing the accompaniment resources of copyrighted music. 
     *
     * @param resourceID the resource ID corresponding to the song or accompaniment.
     */
    public abstract int getTotalScore(String resourceID);

    /**
     * Get standard pitch data.
     *
     * Available since: 2.15.0
     * Description: Get standard pitch data.
     * Use case: Can be used to display standard pitch lines on the view.
     * Cation: Only accompaniment or climactic clip assets have pitch lines.
     *
     * @param resourceID the resource ID corresponding to the song or accompaniment.
     * @param callback get standard pitch data result.
     */
    public abstract void getStandardPitch(String resourceID,
                                          IZegoCopyrightedMusicGetStandardPitchCallback callback);

    /**
     * Get real-time pitch data.
     *
     * Available since: 2.15.0
     * Description: Get real-time pitch data.
     * Use case: Can be used to display real-time pitch lines on the view.
     * When to call: It can be called after playing the copyright accompaniment and starting to score.
     *
     * @param resourceID the resource ID corresponding to the song or accompaniment.
     */
    public abstract int getCurrentPitch(String resourceID);
}
