--[[Copyright (c) 2012-2020 baby-bus.com

http://www.baby-bus.com/LizardMan/

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

]]
--[[!--

游戏角色类，定义游戏角色公用操作方法及逻辑实现。

-   定义游戏角色相关信息。

]]
----------------------
-- 类 擦除
----------------------
local M    = classSpriteTouch("Texture")
M.TAG    = "Texture"

----------------------
-- 公共参数
----------------------
-- [常量]
-- ..
----------------------
-- 构造方法
----------------------
function M:ctor(params)
    -- [超类调用]
    M.super.ctor(self, params)
    self._size           = params.size
    -- 显示的图片
    self._showImgPath    = params.showImgPath
    -- 画笔图片
    self._paint          = params.paint
    -- 擦除点
    self._clearPos       = params.clearPos or {}
    -- 擦除点总数
    self._maxPos         = #self._clearPos
    -- 擦除过的点计数
    self._clearCount     = 0

    self._test           = params.test
end

----------------------
-- 结点渲染 
----------------------
function M:onRender()
    -- [超类调用]
    M.super.onRender(self)
    -- 图片渲染
    self:loadDisplay()
    if self._test then
        self:loadTestNode()
    end
    -- 创建画布
    self:loadRenderTexture()
end

-- 图片渲染
function M:loadDisplay()
    if self._showImgPath then
        self:display(self._showImgPath)
    else
        -- 设置尺寸
        self:setContentSize(self._size)
    end
end

-- 创建画布
function M:loadRenderTexture()
    -- 创建一个画布
    local renderTexture    = CCNodeExtend.extend(cc.RenderTexture:create(self:cw(), self:ch()))
    :p(self:cw()/2, self:ch()/2)

    -- 设置大小
    -- renderTexture:setContentSize(self._size)

    -- 添加画布到裁剪节点上
    renderTexture:to(self, 50)
    self._renderTexture    = renderTexture
end

----------------------
-- 功能函数
----------------------
-- 清除
function M:clearRenderTexture()
    self._renderTexture:beginWithClear(0, 0, 0, 0)
    self._renderTexture:endToLua()
end

-- 画版渐隐
function M:fadeOutEvent()
    local pos = self:convertToNodeSpace(self:worldpoint())
    local index = 2
    self:cycle({
        { "fn", function()
            local renderNode = D.img("common/point.png"):scale(index)
            renderNode:setBlendFunc(GL_ZERO, GL_ONE_MINUS_SRC_ALPHA)
            renderNode:setPosition(pos)
            self._renderTexture:begin()
            renderNode:visit()
            self._renderTexture:endToLua()
            index = index * 1.2
        end },
        {"delay", 0.08},
    }, 30)
end

-- 添加擦除点
function M:loadTestNode(num)
    num = num or #self._clearPos
    Tools:loadTestNode(self, num, self._clearPos)
    self._test = true
end


-- 完成，将自身改变为最后的状态
function M:finish(callback)
    self:clearRenderTexture()
    self:line({
        {"delay", 0.1},
        {"fn", function()
            if self._renderTexture then
                self._renderTexture:remove()
                self._renderTexture = nil
            end
        end},
    })
    
end

function M:reset(imgPath)
    self:display(imgPath)
end

-- 添加图片
function M:addImg(path, pos, scale, opacity)
    scale = scale or 1
    opacity = opacity or 255
    self._renderTexture:begin()
    local node = D.img(path):p(pos):scale(scale):opacity(opacity)
    node:visit()
    self._renderTexture:endToLua()
end


-- 擦除（起点，终点）
function M:erasure(startPos, endPos, paint, opacity, paintSize, paintRadius)
    paint       = ifnil(paint, self._paint)
    paintSize   = ifnil(paintSize, 1)
    paintRadius = ifnil(paintRadius, 1)
    opacity     = opacity or 255
    -- 转换为节点坐标
    startPos = self:convertToNodeSpace(startPos)
    endPos = self:convertToNodeSpace(endPos)
    -- 绘图
    self._renderTexture:begin()
    local size        = 5                                       -- 步长
    local distance    = PT.distance(startPos, endPos)       -- 两点之间距离
    local distanceX = endPos.x - startPos.x             -- x轴距离
    local distanceY = endPos.y - startPos.y             -- y轴距离
    
    for i = 0, distance, size do
        local rate    = i / distance                        -- 比例
        local pos    = cc.p(startPos.x + (distanceX * rate),
        startPos.y + (distanceY * rate))
        local node = D.img(paint):p(pos):scale(paintSize):opacity(opacity)
        node:setBlendFunc(GL_ZERO, GL_ONE_MINUS_SRC_ALPHA)
        node:visit()

        if self._clearPos then
            local radius = node:cw()/2 * paintSize * paintRadius
            for i,v in ipairs(self._clearPos) do
                if PT.distance(v, pos) < radius then
                    if self._testNodeTable then
                        self._testNodeTable[i]:remove()
                        table.remove(self._testNodeTable, i)
                    end
                    table.remove(self._clearPos, i)
                    self._clearCount = self._clearCount + 1
                    break
                end
            end
        end
    end
    self._renderTexture:endToLua()
    return self._clearCount/self._maxPos
end

function M:getRatio()
    return self._clearCount/self._maxPos
end

function M:setClearPos(pos)
    if not istable(pos) then return end
    self._clearPos = pos
    self._clearCount = 0
    self._maxPos = #pos
end

----------------------
-- 结点析构
----------------------
function M:onDestructor()
    -- [超类调用]
    M.super.onDestructor(self)

    -- 删除画笔
    if self._erasePaint then
        self._erasePaint:release()
        self._erasePaint = nil
    end
end

----------------------
-- 验证
----------------------
-- 验证合法性[构造函数参数]
function M:assertParameters(params)
end

return M
