--[[Copyright (c) 2012-2020 baby-bus.com

http://www.baby-bus.com/LizardMan/

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

]]

local ScrollView = class("ScrollUiEx", function(params)
    local rect = params.rect or cc.rect(0, 0, 0, 0)
    local node = display.newClippingRegionNode(rect)

    node:setNodeEventEnabled(true)
    --require("framework.api.EventProtocol").extend(node)
    cc(node):addComponent("components.behavior.EventProtocol"):exportMethods()
    return node
end)

ScrollView.DIRECTION_VERTICAL   = 1
ScrollView.DIRECTION_HORIZONTAL = 2

function ScrollView:ctor(params)
    -- 滚屏的方向
    self.direction = params.dir or ScrollView.DIRECTION_HORIZONTAL
    -- 显示区域
    self.touchRect = params.rect
    -- 第一个对象的位置
    self.firstPoint = params.firstPoint
    -- 每个对象占据的空间大小
    self.itemSize = params.contentSize
    -- 拖动下限
    self.dragThreshold = 10
    -- 回弹下限
    self.bouncThreshold = 50
    self.defaultAnimateTime = 0.4
    self.defaultAnimateEasing = "backOut"

    self.offsetX = 0
    self.offsetY = 0
    self.currentIndex = 0
    self.touchItem = nil
    self.items = {}

    -- 加载容器
    self:loadContainer()

    -- self:bindTouch()
end

-- 加载容器
function ScrollView:loadContainer()
    self.view = display.newNode()
    self:addChild(self.view)
end

function ScrollView:getCurrentItem()
    if self.currentIndex > 0 then
        return self.items[self.currentIndex]
    else
        return nil
    end
end

function ScrollView:getCurrentIndex()
    return self.currentIndex
end

function ScrollView:setCurrentIndex(index)
    self:scrollToItem(index)
end

-- 
function ScrollView:setItems(items)
    self.items = items
end

-- 
function ScrollView:getItemNum()
    return #self.items
end

-- 索引增加
function ScrollView:indexIncrease(num, animated)
    local index = self.currentIndex + num
    self:assertCurrentIndex(index)
    self:scrollToItem(self.currentIndex, animated)
end

-- 设置对象所占尺寸的大小
function ScrollView:setItemSize(size)
    self.itemSize = size
end

function ScrollView:addItem(item)
    self.view:addChild(item)
    self.items[#self.items + 1] = item
    self:reorderAllItems()
end

function ScrollView:insertItemAtIndex(item, index)
    self.view:addChild(item)
    table.insert(self.items, index, item)
    self:reorderAllItems()
end

function ScrollView:removeItemAtIndex(index)
    local item = self.items[index]
    item:removeSelf()
    table.remove(self.items, index)
    self:reorderAllItems()
end

-- 清空滚屏里的所有对象
function ScrollView:removeAllItems()
    for i = #self.items, 1, -1 do
        local item = self.items[i]
        item:removeSelf()
        table.remove(self.items, i)
    end
    self:reorderAllItems()
end

function ScrollView:getView()
    return self.view
end

function ScrollView:getTouchRect()
    return self.touchRect
end

function ScrollView:setTouchRect(rect)
    self.touchRect = rect
end

function ScrollView:getClippingRect()
    return self:getClippingRegion()
end

function ScrollView:setClippingRect(rect)
    self:setClippingRegion(rect)
end

function ScrollView:scrollToItem(index, animated, time, easing)
    local count = #self.items
    if count < 1 then
        self.currentIndex = 0
        return
    end

    index = index > count - 1 and count - 1 or index
    self:assertCurrentIndex(index)

    local offset = 0
    for i = 2, index do
        local item = self.items[i - 1]
        local size = item:getContentSize()
        if self.direction == ScrollView.DIRECTION_HORIZONTAL then
            offset = offset - self.itemSize.width
        else
            offset = offset + self.itemSize.height
        end
    end

    self:setContentOffset(offset, animated, time, easing)
    self:dispatchEvent({
        name = "ScrollView", 
        event = "scrollToItem",
    })
end

function ScrollView:isTouchEnabled()
    return self.view:isTouchEnabled()
end

function ScrollView:setTouchEnabled(enabled)
    self.view:setTouchEnabled(enabled)
end

-- events
function ScrollView:onTouchBegan(x, y, toucches)
    NV.spriteCppCrashPath(D.getRunningScene().name.."-ScrollView-began")
    self.drag = {
        currentOffsetX = self.offsetX,
        currentOffsetY = self.offsetY,
        startX = x,
        startY = y,
        isTap = true,
    }

    -- if self:isFindTouchItem(x, y) then
        -- audio.playSound("effect/play/pickUp.wav")
        -- self.touchItem:setScale(self.touchItem.scaleFactor + 0.1)
    -- end
    
    return true
end

function ScrollView:onTouchMoved(x, y, touches)
    local offset = 0
    if math.abs(x - self.drag.startX) > self.dragThreshold or math.abs(y - self.drag.startY) > self.dragThreshold then
        self.drag.isTap = false
    end
    
    if self.direction == ScrollView.DIRECTION_HORIZONTAL then
        offset = x - self.drag.startX + self.drag.currentOffsetX
    else
        offset = y - self.drag.startY + self.drag.currentOffsetY
    end

    if not self.drag.isTap then
        self:setContentOffset(offset)
    end
end


function ScrollView:onTouchEnded(x, y, touches)
    NV.spriteCppCrashPath(D.getRunningScene().name.."-ScrollView-end")
    if self.drag.isTap then
        self:onTouchEndedWithTap(x, y)
    else
        self:onTouchEndedWithoutTap(x, y)
    end
    
    if self.touchItem then
        self.touchItem:setScale(self.touchItem.scaleFactor)
        self.touchItem = nil
    end

    self.drag = nil
end

function ScrollView:onTouchEndedWithTap(x, y)
    self:dispatchEvent({
        name = "ScrollView",
        event = "tap",
        data = {
            touchP = ccp(x, y),
        },
    })
end

function ScrollView:onTouchEndedWithoutTap(x, y, top)
    local offsetX, offsetY = self.offsetX, self.offsetY
    local index = 0
    local count = #self.items + 1
    if self.direction == ScrollView.DIRECTION_HORIZONTAL then
        offsetX = -offsetX
        local x = self.itemSize.width * 0.5
        local i = 1
        while i <= count do
            if offsetX < x + self.itemSize.width / 2 - 80 then
                index = i
                break
            end
            x = x + self.itemSize.width
            i = i + 1
        end
        if i > count then index = count end
    else
        top = top or 1
        local y = self.itemSize.height * 0.5
        local i = top
        while i <= count do
            if offsetY < y + self.itemSize.height / 2 - 40 then
                index = i
                break
            end
            y = y + self.itemSize.height
            i = i + 1
        end

        if i > count then index = count end
    end

    self:scrollToItem(index, true)
end

function ScrollView:onTouchCancelled(x, y)
    self.drag = nil
end

-- private methods
-- 
function ScrollView:isFindTouchItem(x, y)
    local touchP = self.view:convertToNodeSpace(ccp(x, y))
    for _, item in ipairs(self.items) do
        if item:getBoundingBox():containsPoint(touchP) then
            self.touchItem = item
            return true
        end
    end
    return false
end

function ScrollView:reorderAllItems()
    local count = #self.items
    local x, y = self.firstPoint.x, self.firstPoint.y
    -- self.itemSize.width * 0.5, self.itemSize.height * 0.5
    for i = 1, count do
        local item = self.items[i]
        item:setPosition(x, y)
        if self.direction == ScrollView.DIRECTION_HORIZONTAL then
            x = x + self.itemSize.width
        else
            y = y - self.itemSize.height
        end
    end

    -- 重新设置currentIndex
    self:resetCurrentIndex()
    -- 重新设置容器的大小
    self:resetContentSize(x, y)
end

-- 重新设置currentIndex
function ScrollView:resetCurrentIndex()
    local count = #self.items
    if count > 0 then
        if self.currentIndex < 1 then
            self.currentIndex = 1
        elseif self.currentIndex > count then
            self.currentIndex = count
        end
    else
        self.currentIndex = 0
    end
end

-- 重新设置容器的大小
function ScrollView:resetContentSize(x, y)
    local size
    if self.direction == ScrollView.DIRECTION_HORIZONTAL then
        size = CCSize(x, self.itemSize.height)
    else
        size = CCSize(self.itemSize.width, math.abs(y))
    end
    self.view:setContentSize(size)
end

function ScrollView:setContentOffset(offset, animated, time, easing)
    local ox, oy = self.offsetX, self.offsetY
    local x, y = ox, oy
    local viewSize = self.view:getContentSize()
    if self.direction == ScrollView.DIRECTION_HORIZONTAL then
        x = offset

        local maxX = self.bouncThreshold
        local minX = -viewSize.width - self.bouncThreshold + self.touchRect.width
        if x > maxX then
            x = maxX
        elseif x < minX then
            x = minX
        end
        self.offsetX = x
    else
        y = offset

        -- local maxY = self.bouncThreshold
        -- local minY = -viewSize.height - self.bouncThreshold + self.touchRect.size.height

        local maxY = self.firstPoint.y + viewSize.height + self.bouncThreshold - self.touchRect.height + 10
        local minY = -20

        if y > maxY then
            y = maxY
        elseif y < minY then
            y = minY
        end
        self.offsetY = y 
    end

    if animated then
        transition.stopTarget(self.view)
        transition.moveTo(self.view, {
            x = x,
            y = y,
            time = time or self.defaultAnimateTime,
            easing = easing or self.defaultAnimateEasing,
        })
    else
        self.view:setPosition(x, y)
    end
end

-- 验证currentIndex的有效性
function ScrollView:assertCurrentIndex(index)
    if index < 1 then
        index = 1
    elseif index > #self.items then
        index = #self.items
    end
    self.currentIndex = index
end

function ScrollView:onEnter()

end

function ScrollView:onExit()
    self:removeAllEventListeners()
end



return ScrollView
