--[[Copyright (c) 2012-2020 baby-bus.com

http://www.baby-bus.com/LizardMan/

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.

]]

--[[!--

游戏角色类，定义游戏角色公用操作方法及逻辑实现。

-   定义游戏角色相关信息。

]]

----------------------
-- 类
----------------------

local M     = classSpriteTouch("ScrollView")
M.TAG       = "ScrollView"

----------------------
-- 公共参数
----------------------

-- [常量]
-- ..

-- [操作变量]
-- ..

----------------------
-- 构造方法
----------------------

--[[--

构造方法，定义结点实例初始化逻辑

### Parameters:
-   table **params**    参数集合

### Return: 
-   object              对象实例

]]

function M:ctor(params)
    -- [超类调用]
	M.super.ctor(self, params)
	-- 主场景
    self._layer 		    = params.layer
    -- ui栏
	self._uiBar      	    = params.uiBar
	-- 坐标
	self._pos         	    = ifnil(params.pos, cc.p(0, 0))
	-- 缩放系数
	self._scaleNum  	    = ifnil(params.scaleNum, 1)
	-- 在父类上的层级
    self._zOrder      	    = ifnil(params.zOrder, 0)
    -- 滚动栏中点坐标
    self._midPos            = ifnil(params.midPos, ccp(self._uiBar:cw()/2, self._uiBar:ch()/2)) 
    -- 滚动栏内显示个数
    self._showNumber        = ifnil(params.showNumber, 3)
    -- 是否是水平方向
    self._isHorizontal      = ifnil(params.isHorizontal, false)
    -- 是否允许滚动
    self._arrowScroll       = ifnil(params.arrowScroll, true)
    -- 滚动方向
    self._cycleDirection    = 1
    -- 滚动速度
    self._cycleSpeed        = 50     
    -- 项集合
    self._itemTable         = {}
    -- 触控锁
    self._canTouch          = true
    --实际显示区域占UI栏图片的比例
    self._showRatio         = ifnil(params.showRatio, 1)
end

----------------------
-- 结点渲染 
----------------------
--[[--

实体渲染，处理实体结点加载、事件绑定等相关操作

]]

function M:onRender()
    -- [超类调用]
    M.super.onRender(self)
    -- 设置间距
    self:setScrollViewSpacing()
    -- 开始滚动
    if self._arrowScroll then 
        self:startRoll()
    end
end

----------------------
-- 加载
----------------------

-- 设置间距
function M:setScrollViewSpacing()
    if self._isHorizontal then
        local width  = self._uiBar:cw() * self._showRatio
        local spacing = width / self._showNumber
        self._spacing = spacing
    else
        local height  = self._uiBar:ch() * self._showRatio
        local spacing = height / self._showNumber 
        self._spacing = spacing
    end
end

-- 在滚动栏中添加初始项
function M:addItemToScrollView(itemTab)
    if self._isHorizontal then
        --UI栏最左的X坐标为0， 再减去非显示区域 cw() * (1 - self._showRatio) / 2，即为实际显示区域的左顶点X坐标，最后减去每个选项间隙的一半，即为从左往右的第一个选项X坐标
        local startPosX = 0 + self._uiBar:cw() * (1 - self._showRatio) / 2 + self._spacing / 2
        local startPosY = self._uiBar:ch() * 0.5

        for i, item in ipairs(itemTab) do
            local posX = startPosX + self._spacing * (i - 1)
            local posY = startPosY
            item:to(self):p(posX, posY)
            table.insert(self._itemTable, item)
        end
    else
        local startPosX = self._uiBar:cw() * 0.5
        --UI栏最上的Y坐标为ch() 再减去非显示区域 ch() * (1 - self._showRatio) / 2，即为实际显示区域的上顶点Y坐标，最后减去每个选项间隙的一半，即为从上往下的第一个选项Y坐标
        local startPosY = self._uiBar:ch() - self._uiBar:ch() * (1 - self._showRatio)/2 - self._spacing / 2
        for i, item in ipairs(itemTab) do
            local posX = startPosX
            local posY = startPosY - self._spacing * (i - 1)
            item:to(self):p(posX, posY)
            table.insert(self._itemTable, item)
        end
    end
end

-- 滚动所有项
function M:scrollAllItems(deltaPos)
    -- 滚动的总距离
    local totalDis = self._spacing * #self._itemTable
    -- 是否水平方向滚动
    if self._isHorizontal then
        for i, item in ipairs(self._itemTable) do
            local newPosX = item:px() + deltaPos.x
            if newPosX < - item._itemSize * 0.5 then
                newPosX = newPosX + totalDis
            elseif newPosX > totalDis - item._itemSize * 0.5 then
                newPosX = newPosX - totalDis
            end
            item:px(newPosX)
        end
    else
        for i, item in ipairs(self._itemTable) do
            local newPosY = item:py() + deltaPos.y
            if newPosY > totalDis - item._itemSize * 0.5 then
                newPosY = newPosY - totalDis
            elseif newPosY < - item._itemSize * 0.5 then
                newPosY = newPosY + totalDis
            end
            item:py(newPosY)
        end
    end
end

-- 开始滚动
function M:startRoll()
    self:scheduleUpdateWithPriorityLua(function(dt)
        local dis = dt * self._cycleDirection * self._cycleSpeed
        self:scrollAllItems(ccp(dis, dis))
    end, 0);
end

-- 停止滚动
function M:stopRoll()
    self:unscheduleUpdate()
end

-- 移除滚动栏中的项
function M:removeItemFromScrollView(item)
    -- 移除滚动栏中项的时候，禁止触控
    self._canTouch = false
    local pos = nil
    for i, v in ipairs(self._itemTable) do
        if v == item then
            pos = v:point()
            table.remove(self._itemTable, i)
            break
        end
    end
    -- 调整或重新排列坐标
    self:revisionOrReorganizeItem(pos)
end

-- 调整或重新排列坐标
function M:revisionOrReorganizeItem(pos)
    if #self._itemTable > self._showNumber then
        -- 超出范围的元素移动到后面
        self:moveToBackItem()
        -- 重新排列坐标
        self:reorganizeItemPosition(pos)
    else
        -- 调整坐标
        self:revisionItemPosition()
    end
end

-- 超出范围的元素移动到后面
function M:moveToBackItem()
    if self._isHorizontal then
        local posX = self._uiBar:cw() + self._spacing * 0.5
        for i, item in ipairs(self._itemTable) do
            if item:px() >= posX then
                item:px(item:px() - self._spacing * (#self._itemTable + 1))
            end
        end
    else
        local posY = self._uiBar:ch() + self._spacing * 0.5
        for i, item in ipairs(self._itemTable) do
            if item:py() >= posY then
                item:py(item:py() - self._spacing * (#self._itemTable + 1))
            end
        end
    end
end

-- 重新排列坐标
function M:reorganizeItemPosition(pos)
    local aTable = {}
    if self._isHorizontal then
        for i, item in ipairs(self._itemTable) do
            if item:px() <= pos.x then
                table.insert(aTable, item)
            end
        end
        for i, item in ipairs(aTable) do
            item:line({
                {"moveBy", 0.5, ccp(self._spacing, 0)},
                {"fn", function()
                    -- 开始滚动
                    self:startRoll()
                    -- 触控打开
                    self._canTouch = true
                end},
            })
        end
    else
        for i, item in ipairs(self._itemTable) do
            if item:py() <= pos.y then
                table.insert(aTable, item)
            end
        end
        for i, item in ipairs(aTable) do
            item:line({
                {"moveBy", 0.5, ccp(0, self._spacing)},
                {"fn", function()
                    -- 开始滚动
                    self:startRoll()
                    -- 触控打开
                    self._canTouch = true
                end},
            })
        end
    end    
end

-- 调整坐标
function M:revisionItemPosition()
    local aTable               = nil
    local startPosX, startPosY = nil, nil

    if self._isHorizontal then
        aTable = self:arrangementItemWithPositionX()
        startPosX = 0 + self._uiBar:cw() * (1 - self._showRatio) / 2 + self._spacing / 2--self._midPos.x - self._spacing
        startPosY = self:ch() * 0.5
    else
        aTable = self:arrangementItemWithPositionY()
        startPosX = self:cw() * 0.5
        startPosY = self._uiBar:ch() - self._uiBar:ch() * (1 - self._showRatio) / 2 - self._spacing / 2--self._midPos.y + self._spacing
    end

    for i, item in ipairs(aTable) do
        local pos = nil
        if self._isHorizontal then
            local posX  = startPosX + self._spacing * (i - 1)
            local posY  = self._midPos.y
            pos = ccp(posX, posY)
        else
            local posX  = self._midPos.x
            local posY  = startPosY - self._spacing * (i - 1)
            pos = ccp(posX, posY)
        end

        local dis   = ccpDistance(pos, item:point())
        local time  = 0.3
        item:line({
            {"moveTo", time, pos},
            {"fn", function()
                -- 是否允许滚动
                self._arrowScroll = false
                -- 开启触控
                self._canTouch = true
            end},
        })
    end
end

-- 余下的项，根据Y坐标从高到低排列
function M:arrangementItemWithPositionY()
    local aTable = {}
    for i, item in ipairs(self._itemTable) do
        local hasInsert = false
        for j, v in ipairs(aTable) do
            if item:py() > v:py() then
                hasInsert = true
                table.insert(aTable, j, item)
                break
            end
        end
        if not hasInsert then
            table.insert(aTable, item)
        end
    end
    return aTable
end

-- 余下的项，根据X坐标从大到小排列
function M:arrangementItemWithPositionX()
    local aTable = {}
    for i, item in ipairs(self._itemTable) do
        local hasInsert = false
        for j, v in ipairs(aTable) do
            if item:px() < v:px() then
                hasInsert = true
                table.insert(aTable, j, item)
                break
            end
        end
        if not hasInsert then
            table.insert(aTable, item)
        end
    end
    return aTable
end

-- 是否点击在UI栏内
function M:isTouchAtUiBar(x, y)
    local box = self._uiBar:getBoundingBox()
    -- 获取的是ui栏的包围盒，要获取ui栏内的区域，需要减去一定像素并调整坐标
    box.width = box.width - 30
    box.height = box.height - 30
    box.x = box.x + 15
    box.y = box.y + 15
    local pos = self._uiBar:getParent():convertToNodeSpace(ccp(x, y))
    return cc.rectContainsPoint(box, pos)
end

----------------------
-- 触控
----------------------

function M:onTouchBegan(x, y)
    -- 是否点击在UI栏内
    if not self:isTouchAtUiBar(x, y) then return end
    if not self._canTouch then return end
    if #self._itemTable <= self._showNumber then return end
    self._startPos = ccp(x, y)
    -- 停止滚动
    self:stopRoll()
    return true
end

function M:onTouchMoved(x, y)
    if not self._canTouch then return end
    if not self._arrowScroll then return end
    local deltaPos = ccp(x - self._startPos.x, y - self._startPos.y)
    -- 拖着滚动全部项
    self:scrollAllItems(deltaPos)
    self._startPos = ccp(x, y)
end

function M:onTouchEnded(x, y)
    if not self._canTouch then return end
    if not self._arrowScroll then return end
    if #self._itemTable <= self._showNumber then return end
    -- 开始滚动
    self:startRoll()
end

----------------------
-- 结点析构
----------------------

function M:onDestructor()
  	-- [超类调用]
  	M.super.onDestructor(self)
end

----------------------
-- 过度动画开始之前的调用
----------------------

--[[--

1.资源释放，退出场景前的操作准备
2.处理视图结点卸载、事件解除绑定等相关操作

]]

function M:onExitTransitionStart()

end

----------------------
-- 点击
----------------------
--[[--

判断是否点击到内容区域

### Example:

### Parameters:
-   string **event**    事件名称
-   number **x**        点击位置x(相对于屏幕)
-   number **y**        点击位置y(相对于屏幕)

]]

function M:onTouchHandled(event, x, y)

end

----------------------
-- 验证
----------------------

-- 验证合法性[构造函数参数]
function M:assertParameters(params)
    
end

return M
