/**
 * 自定义lb-search-input
 * 搜索输入框组件，支持搜索、清除、聚焦等功能
 * 
 * 使用示例：
 * 
 * 1. 基本用法：
 * <lb-search-input
 *   search-key="vm.searchKeyword"
 *   on-search="vm.handleSearch(searchKey)"
 *   on-clear="vm.handleClear()">
 * </lb-search-input>
 * 
 * 2. 带自定义占位符：
 * <lb-search-input
 *   search-key="vm.searchKeyword"
 *   placeholder="请输入客户名称或手机号"
 *   on-search="vm.handleSearch(searchKey)"
 *   on-clear="vm.handleClear()">
 * </lb-search-input>
 * 
 * 3. 禁用状态：
 * <lb-search-input
 *   search-key="vm.searchKeyword"
 *   disabled="vm.isDisabled"
 *   on-search="vm.handleSearch(searchKey)"
 *   on-clear="vm.handleClear()">
 * </lb-search-input>
 * 
 * 4. 在控制器中的处理函数：
 * vm.searchKeyword = '';
 * vm.isDisabled = false;
 * 
 * vm.handleSearch = function(searchKey) {
 *   console.log('搜索关键词:', searchKey);
 *   // 执行搜索逻辑
 *   vm.performSearch(searchKey);
 * };
 * 
 * vm.handleClear = function() {
 *   console.log('清除搜索');
 *   // 执行清除后的逻辑
 *   vm.resetSearch();
 * };
 */
export default function registerLbSearchInput(module) {
  module.directive("lbSearchInput", ["$timeout", lbSearchInput]);

  function lbSearchInput($timeout) {
    return {
      restrict: "E",
      replace: true,
      templateUrl: "directives/lbSearchInput/lbSearchInput.html",
      scope: {
        searchKey: "=",
        onSearch: "&", // 搜索回调函数
        onClear: "&",  // 清除回调函数
        placeholder: "@", // 占位符文本
        disabled: "=", // 是否禁用
      },
      link: function (scope, element, attrs) {
        // 初始化搜索关键词
        if (!scope.searchKey) {
          scope.searchKey = "";
        }

        // 初始化聚焦状态
        scope.focused = "";

        // 聚焦功能
        scope.focus = function() {
          if (scope.disabled) return;
          
          // 设置聚焦状态
          scope.focused = "text-left";
          
          // 聚焦到输入框
          var input = element.find("input");
          if (input.length > 0) {
            input[0].focus();
            // 将光标移动到文字尾部
            var length = input[0].value.length;
            input[0].setSelectionRange(length, length);
          }
        };

        // 清除搜索内容
        scope.clearSearch = function($event) {
          if ($event) {
            $event.stopPropagation();
          }
          
          scope.searchKey = "";
          scope.focused = "";
          
          // 触发清除回调
          if (scope.onClear) {
            scope.onClear();
          }
          
          // 重新聚焦到输入框
          scope.focus();
        };

        // 搜索功能
        scope.searchByKeyword = function() {
          if (scope.disabled) return;
          
          // 触发搜索回调
          if (scope.onSearch) {
            scope.onSearch({ searchKey: scope.searchKey });
          }
        };

        // 输入框失焦处理
        scope.blur = function() {
          // 延迟设置失焦状态，避免清除按钮点击时立即失焦
          setTimeout(function() {
            scope.focused = "";
            scope.$apply();
          }, 150);
        };

        // 监听搜索关键词变化
        scope.$watch("searchKey", function(newVal, oldVal) {
          if (newVal !== oldVal) {
            // 可以在这里添加实时搜索逻辑
            // scope.searchClientByKeywordList();
          }
        });

        // 监听禁用状态
        scope.$watch("disabled", function(newVal) {
          if (newVal) {
            scope.focused = "";
          }
        });

        // 组件销毁时清理
        scope.$on("$destroy", function () {
          // 清理工作
        });
      },
    };
  }
}
