/**
 * 表格组件
 * 功能：支持数据展示、排序、搜索、滚动加载、行合并、表头固定、列固定、列宽配置等功能的移动端表格组件
 * 
 * @usage
 * <lb-table 
 *   data="tableData" 
 *   columns="tableColumns"
 *   page-size="20"
 *   show-search="true"
 *   merge-cells="mergeCellsConfig"
 *   fixed-header="true"
 *   fixed-columns="fixedColumnsConfig"
 *   on-row-click="onRowClick(row, index)"
 *   on-load-more="onLoadMore()">
 * </lb-table>
 * 
 * @param {Array} data - 表格数据
 * @param {Array} columns - 列配置 [{title: '姓名', field: 'name', width: '100px', sortable: true}]
 * @param {Number} pageSize - 每次加载条数
 * @param {Boolean} showSearch - 是否显示搜索框
 * @param {Array} mergeCells - 合并单元格配置 [{row: 0, col: 0, rowspan: 2, colspan: 1}]
 * @param {Boolean} fixedHeader - 是否固定表头
 * @param {Object} fixedColumns - 列固定配置 {left: ['field1'], right: ['field2']}
 * @param {Function} onRowClick - 行点击回调
 * @param {Function} onLoadMore - 加载更多回调
 */
export default function registerLbTable(module) {
  module.directive('lbTable', ["$filter", "$timeout", "$ionicScrollDelegate", "$sce", "$compile", lbTable]);
  
  function lbTable($filter, $timeout, $ionicScrollDelegate, $sce, $compile) {
    return {
      restrict: "E",
      replace: true,
      scope: {
        data: "=",
        columns: "=",
        pageSize: "=",
        showSearch: "=",
        mergeCells: "=", // 只保留mergeCells参数
        onRowClick: "&",
        onHeaderClick: "&",
        onSort: "&",
        onLoadMore: "&",
        hasMore: "=",
        fixedHeader: "=",
        fixedColumnsConfig: "=",
        fixedColumns: "=",
        height: "="
      },
      templateUrl: "directives/lbTable/lbTable.html",
      link: function (scope, element) {
        // 初始化默认值
        scope.pageSize = scope.pageSize || 30;
        scope.showSearch = scope.showSearch !== false;
        scope.hasMore = scope.hasMore !== false;
        scope.currentPage = 1;
        scope.searchText = "";
        scope.sortColumn = "";
        scope.sortDirection = "asc";
        scope.isLoading = false;
        scope.displayData = [];
        scope.scrollThreshold = 50;
        scope.fixedHeader = scope.fixedHeader !== false;
        
        function initMergeCache() {
          scope.mergeCache = {
            rowspan: {},
            colspan: {},
            shouldShow: {}
          };
        }
        initMergeCache()
        // 合并监听 data 和 mergeCells
        scope.$watchGroup(['data', 'mergeCells'], function(newVals, oldVals) {
          var newData = newVals[0];
          var newMergeCells = newVals[1];
          // var oldData = oldVals[0];
          // var oldMergeCells = oldVals[1];

          initMergeCache();

          // 处理数据
          if (newData) {
            scope.filteredData = angular.copy(newData);
            // 为所有数据添加行索引
            scope.filteredData.forEach(function(row, index) {
              row.rowIndex = index;
            });
            scope.resetDisplayData();
            
            // 如果mergeCells为空，自动生成合并配置
            if (!scope.mergeCells || scope.mergeCells.length === 0) {
              scope.generateDefaultMergeConfig();
            }
            
            // 计算合并信息
            scope.computeMergeInfo();
          }

          // 处理合并配置
          
          // 重新计算合并信息
          scope.computeMergeInfo(newMergeCells);
          
          // 使用$applyAsync确保在下一个digest循环中执行
          scope.$applyAsync(function() {});
        });

        // 监听固定列配置变化
        scope.$watch('fixedColumns', function(newConfig) {
          initMergeCache()
          scope.fixedColumnsConfig = newConfig || { left: [], right: [] };
        });
        
        // 计算合并信息（缓存版本）
        scope.computeMergeInfo = function(newMergeCells) {
          
          // 初始化合并缓存
          initMergeCache()
          
          // 如果没有合并配置，直接返回
          if (!newMergeCells || newMergeCells.length === 0) {
            return;
          }
          
          // 验证合并配置，检查是否有冲突
          var conflicts = [];
          for (var i = 0; i < newMergeCells.length; i++) {
            for (var j = i + 1; j < newMergeCells.length; j++) {
              var cell1 = newMergeCells[i];
              var cell2 = newMergeCells[j];
              
              // 检查是否有重叠
              if (cell1.row < cell2.row + (cell2.rowspan || 1) &&
                  cell1.row + (cell1.rowspan || 1) > cell2.row &&
                  cell1.col < cell2.col + (cell2.colspan || 1) &&
                  cell1.col + (cell1.colspan || 1) > cell2.col) {
                conflicts.push({
                  cell1: cell1,
                  cell2: cell2,
                  message: `合并配置冲突: (${cell1.row},${cell1.col}) 与 (${cell2.row},${cell2.col}) 重叠`
                });
              }
            }
          }
          
          if (conflicts.length > 0) {
            return;
          }
          
          // 按行和列排序合并配置，确保处理顺序正确
          var sortedMergeCells = newMergeCells.slice().sort(function(a, b) {
            if (a.row !== b.row) {
              return a.row - b.row;
            }
            return a.col - b.col;
          });
          
          
          // 处理每个合并配置
          sortedMergeCells.forEach(function(mergeCell, index) {
            var startRow = mergeCell.row;
            var startCol = mergeCell.col;
            var rowspan = mergeCell.rowspan || 1;
            var colspan = mergeCell.colspan || 1;
            
            
            // 检查是否已经被其他合并配置覆盖
            var startKey = startRow + '_' + startCol;
            if (scope.mergeCache.shouldShow[startKey] === false) {
              return;
            }
            
            // 设置起始单元格的合并信息
            scope.mergeCache.rowspan[startKey] = rowspan;
            scope.mergeCache.colspan[startKey] = colspan;
            scope.mergeCache.shouldShow[startKey] = true;
            
            
            // 标记被合并的单元格为不显示
            for (var r = startRow; r < startRow + rowspan; r++) {
              for (var c = startCol; c < startCol + colspan; c++) {
                var key = r + '_' + c;
                
                // 跳过起始单元格
                if (r === startRow && c === startCol) {
                  continue;
                }
                
                // 检查是否已经被标记为不显示
                if (scope.mergeCache.shouldShow[key] === false) {
                  continue;
                }
                
                scope.mergeCache.shouldShow[key] = false;
              }
            }
          });
          
        };

        // 生成默认合并配置
        scope.generateDefaultMergeConfig = function() {
          if (!scope.filteredData || !scope.columns || scope.filteredData.length === 0) {
            return;
          }
          
          
          var mergeConfig = [];
          
          // 找到colorName和sizeName的列索引
          var colorColIndex = -1;
          var sizeColIndex = -1;
          
          scope.columns.forEach(function(column, index) {
            if (column.field === 'colorName') {
              colorColIndex = index;
            } else if (column.field === 'sizeName') {
              sizeColIndex = index;
            }
          });
          
          if (colorColIndex >= 0) {
            // 处理colorName列合并
            var currentColor = null;
            var startRow = 0;
            var currentSpan = 0;
            
            for (var i = 0; i < scope.filteredData.length; i++) {
              var color = scope.filteredData[i].colorName;
              
              if (color === currentColor) {
                currentSpan++;
              } else {
                if (currentSpan > 1) {
                  mergeConfig.push({
                    row: startRow,
                    col: colorColIndex,
                    rowspan: currentSpan,
                    colspan: 1
                  });
                }
                currentColor = color;
                currentSpan = 1;
                startRow = i;
              }
            }
            
            // 处理最后一组
            if (currentSpan > 1) {
              mergeConfig.push({
                row: startRow,
                col: colorColIndex,
                rowspan: currentSpan,
                colspan: 1
              });
            }
          }
          
          if (sizeColIndex >= 0) {
            // 处理sizeName列合并
            var currentSize = null;
            var startRow = 0;
            var currentSpan = 0;
            
            for (var i = 0; i < scope.filteredData.length; i++) {
              var size = scope.filteredData[i].sizeName;
              
              if (size === currentSize) {
                currentSpan++;
              } else {
                if (currentSpan > 1) {
                  mergeConfig.push({
                    row: startRow,
                    col: sizeColIndex,
                    rowspan: currentSpan,
                    colspan: 1
                  });
                }
                currentSize = size;
                currentSpan = 1;
                startRow = i;
              }
            }
            
            // 处理最后一组
            if (currentSpan > 1) {
              mergeConfig.push({
                row: startRow,
                col: sizeColIndex,
                rowspan: currentSpan,
                colspan: 1
              });
            }
          }
          
        };

        // 重置显示数据
        scope.resetDisplayData = function() {
          scope.currentPage = 1;
          scope.displayData = [];
          scope.hasMore = true;
          scope.loadMoreData();
        };
        
        // 加载更多数据
        scope.loadMoreData = function() {
          if (scope.isLoading || !scope.hasMore) return;
          
          scope.isLoading = true;
          
          if (!scope.filteredData) {
            scope.isLoading = false;
            return;
          }
          
          var start = (scope.currentPage - 1) * scope.pageSize;
          var end = start + scope.pageSize;
          var newData = scope.filteredData.slice(start, end);
          
          if (newData.length > 0) {
            // 为每行数据添加行索引
            newData.forEach(function(row, index) {
              row.rowIndex = start + index;
            });
            
            scope.displayData = scope.displayData.concat(newData);
            scope.currentPage++;
            scope.hasMore = newData.length === scope.pageSize && end < scope.filteredData.length;
          } else {
            scope.hasMore = false;
          }
          
          scope.isLoading = false;
          
          // 触发加载更多回调
          if (scope.onLoadMore && newData.length > 0) {
            scope.onLoadMore();
          }
        };
        
        // 搜索功能
        scope.search = function() {
          if (!scope.data) return;
          
          if (!scope.searchText) {
            scope.filteredData = angular.copy(scope.data);
          } else {
            scope.filteredData = $filter('filter')(scope.data, scope.searchText);
          }
          
          // 重新为过滤后的数据添加行索引
          scope.filteredData.forEach(function(row, index) {
            row.rowIndex = index;
          });
          
          scope.resetDisplayData();
          
          // 重新生成合并配置
          scope.generateDefaultMergeConfig();
        };
        
        // 表头点击事件
        scope.headerClick = function(column, $event) {
          if (scope.onHeaderClick) {
            scope.onHeaderClick({column: column, $event: $event});
          }
          if (column && column.sortable) {
            scope.sort();
          }
        };
        // 排序功能
        scope.sort = function(column) {
          if (scope.sortColumn === column) {
            scope.sortDirection = scope.sortDirection === 'asc' ? 'desc' : 'asc';
          } else {
            scope.sortColumn = column;
            scope.sortDirection = 'asc';
          }
          
          scope.filteredData = $filter('orderBy')(scope.filteredData, scope.sortColumn, scope.sortDirection === 'desc');
          
          // 重新为排序后的数据添加行索引
          scope.filteredData.forEach(function(row, index) {
            row.rowIndex = index;
          });
          
          scope.resetDisplayData();
          
          // 重新生成合并配置
          scope.generateDefaultMergeConfig();
          
          // 触发排序回调
          if (scope.onSort) {
            scope.onSort({column: column, direction: scope.sortDirection});
          }
        };
        
        // 行点击事件
        scope.handleRowClick = function(row, index) {
          if (scope.onRowClick) {
            scope.onRowClick({row: row, index: index});
          }
        };
        
        // 获取排序图标类名
        scope.getSortIconClass = function(column) {
          if (scope.sortColumn !== column) {
            return 'ion-arrow-up-b';
          }
          return scope.sortDirection === 'asc' ? 'ion-arrow-up-b' : 'ion-arrow-down-b';
        };
        
        // 信任HTML内容，允许内联样式
        // scope.trustHtml = function(html) {
        //   if (!html && parseInt(html) !== 0) return '';
        //   return $sce.trustAsHtml(html);
        // };

        scope.trustHtml = function(html) {
          if (!html && parseInt(html) !== 0) {
            return '';
          }
          if (typeof html === "number") {
            return $sce.trustAsHtml(html.toString());
          }
          
          // 如果 HTML 包含指令，需要编译
          if (html && html.indexOf('ng-') !== -1) {
            try {
              var compiledElement = $compile(html)(scope);
              var compiledHtml = compiledElement[0].outerHTML;
              return $sce.trustAsHtml(compiledHtml);
            } catch (e) {
              console.warn('编译 HTML 失败:', e);
              return $sce.trustAsHtml(html);
            }
          }
          
          return $sce.trustAsHtml(html);
        };
        
        
        // 获取行合并数 - 使用缓存
        scope.getRowSpan = function(rowIndex, columnIndex, field) {
          var key = rowIndex + '_' + columnIndex;
          var rowspan = scope.mergeCache.rowspan[key];
          
          if (rowspan !== undefined) {
            return rowspan;
          }
          
          return 1;
        };

        // 获取列合并数 - 使用缓存
        scope.getColSpan = function(rowIndex, columnIndex, field) {
          var key = rowIndex + '_' + columnIndex;
          var colspan = scope.mergeCache.colspan[key];
          
          if (colspan !== undefined) {
            return colspan;
          }
          
          return 1;
        };

        // 检查单元格是否应该显示 - 使用缓存
        scope.shouldShowCell = function(rowIndex, columnIndex, field) {
          var key = rowIndex + '_' + columnIndex;
          var shouldShow = scope.mergeCache.shouldShow[key];
          
          if (shouldShow !== undefined) {
            return shouldShow;
          }
          
          return true;
        };

        // 检查是否需要加载更多
        scope.checkLoadMore = function() {
          var wrapper = element[0].querySelector('.lb-table-wrapper');
          if (!wrapper) return;
          
          var scrollTop = wrapper.scrollTop;
          var scrollHeight = wrapper.scrollHeight;
          var clientHeight = wrapper.clientHeight;
          
          if (scrollHeight - scrollTop - clientHeight < scope.scrollThreshold && 
              !scope.isLoading && 
              scope.hasMore) {
            scope.loadMoreData();
          }
        };
        
        // 滚动事件处理
        function handleScroll() {
          if (!scope.isLoading && scope.hasMore) {
            scope.$applyAsync(function() {
              scope.checkLoadMore();
            });
          }
        }

        // 计算固定列偏移并更新列样式
        var fixedLeftOffsets = {};
        
        function computeFixedOffsets() {
          try {
            fixedLeftOffsets = {};
            if (!scope.columns || !scope.columns.length) return;

            var headerCells = element[0].querySelectorAll('.lb-table-header-cell');
            var cumLeft = 0;

            // 计算偏移量
            for (var i = 0; i < scope.columns.length; i++) {
              var col = scope.columns[i];
              if (!col) continue;

              if (scope.isColumnFixed(col.field, 'left')) {
                var widthPx = 0;

                if (col.width) {
                  if (typeof col.width === 'number') {
                    widthPx = col.width;
                  } else if (typeof col.width === 'string') {
                    var parsed = parseFloat(col.width);
                    widthPx = isNaN(parsed) ? 0 : parsed;
                  }
                }

                if (!widthPx && headerCells && headerCells[i]) {
                  widthPx = headerCells[i].getBoundingClientRect().width || 0;
                }

                fixedLeftOffsets[col.field] = Math.max(0, Math.round(cumLeft));
                cumLeft += Math.max(0, Math.round(widthPx));
              }
            }

            // 更新每列的样式对象（避免重复计算）
            scope.columns.forEach(function(col) {
              if (!col.cellStyle) {
                col.cellStyle = {};
              }
              
              // 设置宽度
              if (col.width) {
                col.cellStyle.width = col.width;
                col.cellStyle['min-width'] = col.width;
                col.cellStyle['max-width'] = col.width;
              }
              
              // 设置固定列偏移
              if (scope.isColumnFixed(col.field, 'left')) {
                col.cellStyle.left = (fixedLeftOffsets[col.field] || 0) + 'px';
              } else {
                // 移除非固定列的 left 属性
                delete col.cellStyle.left;
              }
            });

          } catch (e) {
            console.warn('计算固定列偏移失败:', e);
          }
        }

        // 监听列或固定列配置变化，重新计算偏移
        scope.$watchGroup(['columns', 'fixedColumns', 'fixedColumnsConfig.left', 'fixedColumnsConfig.right'], function() {
          $timeout(function() { 
            computeFixedOffsets(); 
          }, 0);
        });

        // 检查列是否固定
        scope.isColumnFixed = function(field, position) {
          if (!scope.fixedColumnsConfig[position]) return false;
          return scope.fixedColumnsConfig[position].indexOf(field) !== -1;
        };
        
        // 获取列固定样式类
        scope.getColumnFixedClass = function(field) {
          var classes = [];
          if (scope.isColumnFixed(field, 'left')) {
            classes.push('fixed-left');
          }
          if (scope.isColumnFixed(field, 'right')) {
            classes.push('fixed-right');
          }
          return classes.join(' ');
        };
        
        // 渲染插槽内容
        scope.renderSlot = function(row, column, rowIndex, columnIndex) {
          if (column.template) {
            // 如果有自定义模板，返回模板内容
            return column.template;
          }
          // 默认返回字段值
          return row[column.field];
        };
        
        // 初始化
        $timeout(function() {
          scope.loadMoreData();
          
          var wrapper = element[0].querySelector('.lb-table-wrapper');
          if (wrapper) {
            wrapper.addEventListener('scroll', handleScroll, { passive: true });
          }
          
          var resizeHandler = function() {
            scope.$applyAsync(function() {
              scope.checkLoadMore();
              computeFixedOffsets();
            });
          };
          window.addEventListener('resize', resizeHandler);

          // 首次渲染后计算一次
          $timeout(function() { computeFixedOffsets(); }, 0);
          
          scope.$on('$destroy', function() {
            if (wrapper) {
              wrapper.removeEventListener('scroll', handleScroll);
            }
            window.removeEventListener('resize', resizeHandler);
          });
        });
      }
    };
  }
}