
/**
 * 滚动监听指令
 * 用法示例：
 * <div 
 *   scroll-listener 
 *   on-scroll="yourScrollHandler(top)" 
 *   scroll-target="#scrollableDiv" 
 *   throttle="200">
 *   <!-- 内容 -->
 * </div>
 * 
 * 参数说明：
 * - on-scroll：滚动时触发的回调函数（必填），会传入 top 参数
 * - scroll-target：可选，指定需要监听滚动的容器选择器（如 #id 或 .class），不填则监听自身
 * - throttle：可选，滚动事件节流时间，单位毫秒，默认100ms
 * 
 * 示例：
 * <div 
 *   id="scrollableDiv" 
 *   style="height:300px;overflow:auto;">
 *   <div style="height:1000px;">内容很长</div>
 * </div>
 * <div 
 *   scroll-listener 
 *   scroll-target="#scrollableDiv" 
 *   on-scroll="onDivScroll(top)" 
 *   throttle="150">
 * </div>
 * 
 * 在控制器中定义：
 * $scope.onDivScroll = function(top) {
 *   // 处理滚动事件，top 是滚动位置的 top 值
 *   console.log('滚动了！当前 top 值：', top);
 * };
 */

export default function registerScrollListener(module) {
  module.directive('scrollListener', [
    '$timeout',
    '$ionicScrollDelegate',
    function($timeout, $ionicScrollDelegate) {
      return {
        restrict: 'A',
        // 移除 scope 配置，使用父作用域避免与 ion-content 冲突
        link: function(scope, element, attrs) {
          var scrollElement;
          var throttleTime = parseInt(attrs.throttle) || 300;
          var scrollHandler;
          var isScrolling = false;

          // 初始化滚动监听
          function initScrollListener() {
            // 确定滚动容器
            if (attrs.scrollTarget) {
              scrollElement = document.querySelector(attrs.scrollTarget);
            } else {
              // 默认使用当前元素
              scrollElement = element[0];
            }

            if (!scrollElement) {
              console.warn('ScrollListener: 未找到滚动容器');
              return;
            }

            // 创建节流函数
            scrollHandler = function(event) {
              if (!isScrolling) {
                isScrolling = true;
                
                $timeout(function() {
                  // 获取滚动位置的 top 值
                  var scrollTop = scrollElement.scrollTop || 0;
                  
                  // 直接调用父作用域中的函数，传入 top 值
                  if (attrs.onScroll && scope.$eval) {
                    scope.$eval(attrs.onScroll, { $event: event, top: scrollTop });
                  }
                  isScrolling = false;
                }, throttleTime);
              }
            };

            // 绑定滚动事件
            angular.element(scrollElement).on('scroll', scrollHandler);
          }

          // 监听器清理函数
          function cleanup() {
            if (scrollElement && scrollHandler) {
              angular.element(scrollElement).off('scroll', scrollHandler);
            }
          }

          // 初始化
          $timeout(function() {
            initScrollListener();
          }, 100);

          // 监听元素变化
          scope.$watch(function() {
            return element[0].offsetHeight;
          }, function() {
            // 元素尺寸变化时重新初始化
            cleanup();
            $timeout(function() {
              initScrollListener();
            }, 50);
          });

          // 清理
          scope.$on('$destroy', cleanup);
        }
      };
    }
  ]);
}