package com.alibaba.tesseract.page.view;

import android.content.Context;
import android.graphics.drawable.Drawable;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.View;
import android.view.animation.AlphaAnimation;
import android.view.animation.Animation;
import android.view.animation.LinearInterpolator;
import android.view.animation.ScaleAnimation;
import android.widget.FrameLayout;
import android.widget.ImageView;

import com.alibaba.android.tesseract.core.utils.RouterUtils;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.gov.android.api.picLoader.IPictureLoaderService;
import com.alibaba.gov.android.api.picLoader.OnPictureLoaderCallback;
import com.alibaba.gov.android.servicebus.manager.ServiceManager;

import java.util.ArrayList;
import java.util.List;


public class SlideScaleView extends FrameLayout implements Animation.AnimationListener {
    private Context mContext;
    private ImageView topImageView, bottomImageView;
    private Animation topImageToLarge, topImageToSmall, bottomImageToLarge, bottomImageToSmall;
    private Animation topImageAlphaShow, topImageAlphaHide, bottomImageAlphaShow, bottomAlphaHide;
    private int totalImageCount = 0;
    private int showImageIndex = -1;
    private List<Animation> largeSmallAnimList = new ArrayList<>();
    private List<Animation> scaleAnimList = new ArrayList<>();
    private List<ImageData> imageDataList = new ArrayList();
    private List<Drawable> drawables = new ArrayList<>();
    private Handler uiHandler = new Handler(Looper.getMainLooper());

    public SlideScaleView(Context context) {
        super(context);
        this.mContext = context;
        addView();
    }

    public SlideScaleView(Context context, AttributeSet attrs) {
        super(context, attrs);
        this.mContext = context;
        addView();
    }

    public SlideScaleView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        this.mContext = context;
        addView();
    }

    private void addView() {
        topImageView = new ImageView(mContext);
        bottomImageView = new ImageView(mContext);
        topImageView.setScaleType(ImageView.ScaleType.FIT_XY);
        bottomImageView.setScaleType(ImageView.ScaleType.FIT_XY);
        this.addView(bottomImageView);
        this.addView(topImageView);
    }

    public void addImageList(JSONArray array) {
        if (array == null || array.size() <= 0) return;
        reset();
        for (int i = 0; i < array.size(); i++) {
            ImageData data = new ImageData();
            data.parse(array.getJSONObject(i));
            imageDataList.add(data);
        }
        totalImageCount = imageDataList.size();
        IPictureLoaderService loaderService = ServiceManager.getInstance().getService(IPictureLoaderService.class.getName());
        if (loaderService != null) {
            for (ImageData data : imageDataList) {
                loaderService.load(getContext(), data.bgImage).onLoadCallback(new OnPictureLoaderCallback() {
                    @Override
                    public void onSuccess(Drawable drawable) {
                        if (drawables.isEmpty()) {
                            topImageView.setImageDrawable(drawable);
                        }
                        drawables.add(drawable);
                        refreshDrawables();
                    }

                    @Override
                    public void onFail(Exception e) {}
                }).into(new ImageView(getContext()));
            }
        }
    }

    private void refreshDrawables() {
        if (drawables.size() <= 1) return; //只有一张图片，不轮播
        if (drawables.size() < totalImageCount) return; //图片没下载完，不轮播
        initAnim();
        startAnim();
    }

    private void initAnim() {
        largeSmallAnimList.clear();
        scaleAnimList.clear();

        topImageToLarge = new ScaleAnimation(1.0f, 1.12f, 1.0f, 1.12f, Animation.RELATIVE_TO_SELF, 0.5f, Animation.RELATIVE_TO_SELF, 0.5f);
        topImageToSmall = new ScaleAnimation(1.12f, 1.0f, 1.12f, 1.0f, Animation.RELATIVE_TO_SELF, 0.5f, Animation.RELATIVE_TO_SELF, 0.5f);
        bottomImageToLarge = new ScaleAnimation(1.0f, 1.12f, 1.0f, 1.12f, Animation.RELATIVE_TO_SELF, 0.5f, Animation.RELATIVE_TO_SELF, 0.5f);
        bottomImageToSmall = new ScaleAnimation(1.12f, 1.0f, 1.12f, 1.0f, Animation.RELATIVE_TO_SELF, 0.5f, Animation.RELATIVE_TO_SELF, 0.5f);

        topImageAlphaShow = new AlphaAnimation(0f, 1f);
        topImageAlphaHide = new AlphaAnimation(1f, 0f);
        bottomImageAlphaShow = new AlphaAnimation(0f, 1f);
        bottomAlphaHide = new AlphaAnimation(1f, 0f);

        largeSmallAnimList.add(topImageToLarge);
        largeSmallAnimList.add(topImageToSmall);
        largeSmallAnimList.add(bottomImageToLarge);
        largeSmallAnimList.add(bottomImageToSmall);

        scaleAnimList.add(topImageAlphaShow);
        scaleAnimList.add(topImageAlphaHide);
        scaleAnimList.add(bottomImageAlphaShow);
        scaleAnimList.add(bottomAlphaHide);

        for (Animation animation : largeSmallAnimList) {
            animation.setInterpolator(new LinearInterpolator());
            animation.setDuration(5000);
            animation.setFillAfter(true);
            animation.setAnimationListener(this);
        }
        for (Animation animation : scaleAnimList) {
            animation.setInterpolator(new LinearInterpolator());
            animation.setFillAfter(true);
            animation.setDuration(2000);
            animation.setAnimationListener(this);
        }
    }

    private void startAnim() {
        topImageView.clearAnimation();
        bottomImageView.clearAnimation();
        bindImageInfo(topImageView, imageDataList.get(0), drawables.get(0));
        bindImageInfo(bottomImageView, imageDataList.get(1), drawables.get(1));
        showImageIndex = 0;
        topImageView.startAnimation(topImageToLarge);
        bottomImageView.startAnimation(topImageToLarge);
    }

    private void bindImageInfo(final ImageView imageView, final ImageData data, Drawable drawable) {
        imageView.setImageDrawable(drawable);
        imageView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                RouterUtils.goToUri(getContext(), data.link);
            }
        });
        if (!TextUtils.isEmpty(data.accessibilityText)) {
            imageView.setImportantForAccessibility(View.IMPORTANT_FOR_ACCESSIBILITY_YES);
            imageView.setContentDescription(data.accessibilityText);
        }
    }

    @Override
    public void onAnimationStart(Animation animation) {}

    @Override
    public void onAnimationEnd(Animation animation) {
        try {
            animationEnd(animation);
        } catch (Throwable e) {
            e.printStackTrace();
        }
    }

    private void animationEnd(Animation animation) {
        if (animation == topImageToLarge) {
            ++showImageIndex;
            bindImageInfo(bottomImageView, imageDataList.get(showImageIndex % totalImageCount), drawables.get(showImageIndex % totalImageCount));
            bottomImageView.setAnimation(bottomImageAlphaShow);
            topImageView.startAnimation(topImageAlphaHide);
        }
        if (animation == topImageAlphaHide) {
            bottomImageView.bringToFront();
            uiHandler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    bottomImageView.startAnimation(bottomImageToSmall);
                    topImageView.startAnimation(topImageToSmall);
                }
            }, 600);
        }
        if (animation == bottomImageToSmall) {
            ++showImageIndex;
            bindImageInfo(topImageView, imageDataList.get(showImageIndex % totalImageCount), drawables.get(showImageIndex % totalImageCount));
            bottomImageView.startAnimation(bottomAlphaHide);
            topImageView.startAnimation(topImageAlphaShow);
        }
        if (animation == bottomAlphaHide) {
            topImageView.bringToFront();
            uiHandler.postDelayed(new Runnable() {
                @Override
                public void run() {
                    topImageView.startAnimation(topImageToLarge);
                    bottomImageView.startAnimation(topImageToLarge);
                }
            }, 600);
        }
    }

    @Override
    public void onAnimationRepeat(Animation animation) {}

    private void reset() {
        imageDataList.clear();
        drawables.clear();
    }

    private final class ImageData {
        public String bgImage;
        public String link;
        public String accessibilityText;

        public void parse(JSONObject json) {
            this.bgImage = json.getString("bgImage");
            this.link = json.getString("link");
            this.accessibilityText = json.getString("accessibilityText");
        }
    }
}