/**
 * 获取用户媒体设备
 * @param {object} constraints - 选项
 * @param {function} callback - 处理回调
 */
function getUserMedia(constraints, callback) {
	// 老的浏览器可能根本没有实现 mediaDevices，所以我们可以先设置一个空的对象
	if (navigator.mediaDevices === undefined) {
		navigator.mediaDevices = {};
	}

	// 一些浏览器部分支持 mediaDevices。我们不能直接给对象设置 getUserMedia
	// 因为这样可能会覆盖已有的属性。这里我们只会在没有getUserMedia属性的时候添加它。
	if (navigator.mediaDevices.getUserMedia === undefined) {
		navigator.mediaDevices.getUserMedia = function(constraints) {

			// 首先，如果有getUserMedia的话，就获得它
			var getUserMedia = navigator.getUserMedia || navigator.webkitGetUserMedia || navigator
				.mozGetUserMedia || navigator.msGetUserMedia || navigator.oGetUserMedia;

			// 一些浏览器根本没实现它 - 那么就返回一个error到promise的reject来保持一个统一的接口
			if (!getUserMedia) {
				return Promise.reject(new Error('getUserMedia is not implemented in this browser'));
			}

			// 否则，为老的navigator.getUserMedia方法包裹一个Promise
			return new Promise(function(resolve, reject) {
				getUserMedia.call(navigator, constraints, resolve, reject);
			});
		}
	}

	navigator.mediaDevices.getUserMedia(constraints).then(function(stream) {
		callback(false, stream);
	}).catch(function(err) {
		callback(err);
	});
}

/**
 * 获取压缩后的图片Base64数据
 * @param {Image} img - Image对象
 */
function getBase64DataAfterCompress(img) {
	var canvas = document.createElement('canvas');
	var context = canvas.getContext('2d');
	// 原始尺寸
	var originWidth = img.width;
	var originHeight = img.height;
	// 最大尺寸限制
	var maxWidth = 800;
	var maxHeight = 800;
	// 目标尺寸
	var targetWidth = originWidth,
		targetHeight = originHeight;
	// 图片尺寸超过最大值的限制
	if (originWidth > maxWidth || originHeight > maxHeight) {
		if (originWidth / originHeight > maxWidth / maxHeight) {
			targetWidth = maxWidth;
			targetHeight = Math.round(maxWidth * (originHeight / originWidth));
		} else {
			targetHeight = maxHeight;
			targetWidth = Math.round(maxHeight * (originWidth / originHeight));
		}
	}
	// canvas对图片进行缩放
	canvas.width = targetWidth;
	canvas.height = targetHeight;
	// 清除画布
	context.clearRect(0, 0, targetWidth, targetHeight);
	// 图片压缩
	context.drawImage(img, 0, 0, targetWidth, targetHeight);
	return canvas.toDataURL('image/jpeg', 0.92);
}

/**
 * Blob转Base64
 * @param {blob} blob - Blob数据
 * @param {function} callback - 处理回调
 */
function blobToBase64(blob, callback) {
	var fileReader = new FileReader();
	fileReader.onload = function(e) {
		callback(e.target.result);
	}
	fileReader.readAsDataURL(blob);
}

/**
 * 验证是身份证号码
 * @param {string} idCardNumber - 身份证号码
 */
function isIdCardNumber(idCardNumber) {
	// 身份证号码为15位或者18位
	// 15位时全为数字
	// 18位时前17位为数字，最后一位是校验位，可能为数字或字符X
	var reg = /(^\d{15}$)|(^\d{18}$)|(^\d{17}(\d|X|x)$)/;
	return reg.test(idCardNumber);
}

function download(filename, text) {
	var pom = document.createElement('a');
	pom.setAttribute('href', 'data:text/plain;charset=utf-8,' + encodeURIComponent(text));
	pom.setAttribute('download', filename);
	if (document.createEvent) {
		var event = document.createEvent('MouseEvents');
		event.initEvent('click', true, true);
		pom.dispatchEvent(event);
	} else {
		pom.click();
	}
}


function getParams(name) {
	var result = {} // 定义一个全局的对象
	var str = window.location.search;
	if (str.startsWith('?')) { // 判断str以？开头的
		var strParams = str.split('?')[1];
		var arrParams = strParams.split('&');
		//然后进行for循环，这里直接用了forEach
		arrParams.forEach((item) => {
			var temKey = item.split('=')[0];
			var temVal = item.split('=')[1];
			result[temKey] = temVal
		})
	}

	return result[name]
}