(function () {
  var eventList = []
  /**
   * 定义DOM操作
   * @param {*} selector 选择器
   */
  function DomElement (selector) {
    if (!selector) {
      return
    }
    if (selector instanceof DomElement) {
      return selector
    }
    this.selector = selector

    var nodeType = selector.nodeType
    var selectorResult = []
    if (nodeType === 9 || nodeType === 1) {
      selectorResult = [selector]
    } else if (this.isDomList(selector) || selector instanceof Array) {
      selectorResult = selector
    } else if (typeof selector === 'string') {
      selector = selector.replace('/\n/mg', '').trim()
      if (selector.indexOf('<') === 0) {
        selectorResult = this.createElemByHTML(selector)
      } else {
        selectorResult = this.querySelectorAll(selector)
      }
    }
    var length = selectorResult.length
    if (!length) {
      return this
    }
    for (var i = 0; i < length; i++) {
      this[i] = selectorResult[i]
    }
    this.length = length
  }

  DomElement.prototype = {
    constructor: DomElement,
    querySelectorAll: function (selector) {
      var result = document.querySelectorAll(selector)
      if (this.isDomList(result)) {
        return result
      } else {
        return [result]
      }
    },
    createElemByHTML: function (html) {
      var div = document.createElement('div')
      div.innerHTML = html
      return div.children
    },
    isDomList: function (selector) {
      if (!selector) {
        return false
      }
      if (selector instanceof HTMLCollection || selector instanceof NodeList) {
        return true
      }
      return false
    },
    forEach: function (fn) {
      var i = void 0
      for (i = 0; i < this.length; i++) {
        var elem = this[i]
        var result = fn.call(elem, elem, i)
        if (result === false) {
          break
        }
      }
      return this
    },
    attr: function (key, val) {
      if (val == null) {
        return this[0].getAttribute(key)
      } else {
        return this.forEach(function (elem) {
          elem.setAttribute(key, val)
        })
      }
    },
    hasClass: function (className) {
      if (!this[0].className) {
        return
      }
      return this[0].className.indexOf(className) >= 0
    },
    addClass: function (className) {
      if (!className) {
        return this
      }
      return this.forEach(function (elem) {
        var arr = void 0
        if (elem.className) {
          arr = elem.className.split(/\s/)
          arr = arr.filter(function (item) {
            return !!item.trim()
          })
          if (arr.indexOf(className) < 0) {
            arr.push(className)
          }
          elem.className = arr.join(' ')
        } else {
          elem.className = className
        }
      })
    },
    removeClass: function (className) {
      if (!className) {
        return this
      }
      return this.forEach(function (elem) {
        var arr = void 0
        if (elem.className) {
          arr = elem.className.split(/\s/)
          arr = arr.filter(function (item) {
            item = item.trim()
            if (!item || item === className) {
              return false
            }
            return true
          })
          elem.className = arr.join(' ')
        }
      })
    },
    append: function ($children) {
      return this.forEach(function (elem) {
        $children.forEach(function (child) {
          elem.appendChild(child)
        })
      })
    },
    html: function (value) {
      var elem = this[0]
      if ((!value && value != '') || value == null) {
        return elem.innerHTML
      } else {
        elem.innerHTML = value
        return this
      }
    },
    text: function (val) {
      if (!val) {
        var elem = this[0]
        return elem.innerHTML.replace(/<.*?>/g, function () {
          return ''
        })
      } else {
        return this.forEach(function (elem) {
          elem.innerHTML = val
        })
      }
    },
    on: function (type, selector, fn) {
      if (!fn) {
        fn = selector
        selector = null
      }
      var types = []
      types = type.split(/\s+/)
      return this.forEach(function (elem) {
        types.forEach(function (type) {
          if (!type) {
            return
          }
          eventList.push({
            elem: elem,
            type: type,
            fn: fn
          })
          if (!selector) {
            elem.addEventListener(type, fn)
            return
          }
          elem.addEventListener(type, function (e) {
            var target = e.target
            if (target.matches(selector)) {
              fn.call(target, e)
            }
          })
        })
      })
    },
    get: function (index) {
      var length = this.length;
      if (index >= length) {
        index = index % length;
      }
      return this[index];
    },
    equal: function (elem) {
      if (elem.nodeType === 1) {
        return this[0] === elem;
      } else {
        return this[0] === elem[0];
      }
    },
    parentUntil: function (selector, _currentElem) {
      var results = document.querySelectorAll(selector);
      var length = results.length;
      if (!length) {
        return null;
      }
      var elem = _currentElem || this[0];
      if (elem.nodeName === 'BODY') {
        return null;
      }
      var parent = elem.parentNode || elem.parentElement;
      var i = void 0;
      for (i = 0; i < length; i++) {
        if (parent === results[i]) {
          return $(parent);
        }
      }
      return this.parentUntil(selector, parent)
    },
    parent: function () {
      var parent = this[0].parentNode || this[0].parentElement
      return $(parent);
    },
    prev: function () {
      return $(this[0].previousSibling || this[0].previousElementSibling)
    },
    next: function () {
      return $(this[0].nextSibling || this[0].nextElementSibling)
    },
    getLastChild: function () {
      return $(this[0].lastElementChild)
    },
    insertBefore: function (selector) {
      var $referenceNode = $(selector);
      var referenceNode = $referenceNode[0];
      if (!referenceNode) {
        return this;
      }
      return this.forEach(function (elem) {
        var parent = referenceNode.parentNode;
        parent.insertBefore(elem, referenceNode);
      })
    },
    insertAfter: function (selector) {
      var $referenceNode = $(selector);
      var referenceNode = $referenceNode[0];
      if (!referenceNode) {
        return this;
      }
      return this.forEach(function (elem) {
        var parent = referenceNode.parentNode;
        if (parent.lastChild === referenceNode) {
          parent.appendChild(elem);
        } else {
          parent.insertBefore(elem, referenceNode.nextSibling);
        }
      })
    },
    remove: function () {
      return this.forEach(function (elem) {
        if (elem.remove) {
          elem.remove()
        } else {
          var parent = elem.parentNode || elem.parentElement
          parent && parent.removeChild(elem)
        }
      })
    },
    getNodeName: function () {
      if (!this[0]) {
        return ''
      }
      return this[0].nodeName
    }
  }

  function $ (selector) {
    return new DomElement(selector)
  }

  function Command () {
    this._currentRange = null
    this._currentText = null
    this._currentIndex = -1
    this._networkImageId = 0
    this._clientHeight = 0
    this._clientWidth = (document.body.clientWidth || document.documentElement.clientWidth || window.screen.width) - 24
    this.textEl = $('.editor-editable')
    this.editorEl = $('#editor')
    this.lastUpdateTimestamp = new Date().getTime();
    this._init()
  }
  Command.prototype = {
    constructor: Command,
    setBold: function () {
      if (document.queryCommandState('bold')) {
        document.execCommand('bold', false, null)
        var selection = window.getSelection()
        if(selection.toString().length == 0){
          this.insertHTML('&zwnj;')
        }
      } else {
        document.execCommand('bold', false, null)
      }
      this._updateMenu()
    },
    setHead: function () {
      var formatBlock = document.queryCommandValue('formatBlock')
      if (formatBlock == 'h2') {
        document.execCommand('formatBlock', false, '<p>')
      } else {
        document.execCommand('formatBlock', false, '<h2>')
      }
      this._updateMenu()
    },
    setBlockquote: function () {
      var isBlock = false
      var block = this._getCommonAncestorContainer()
      while(!block.parent().equal(this.textEl) || block.getNodeName() == 'BLOCKQUOTE') {
        if (block.getNodeName() == 'BLOCKQUOTE') {
          isBlock = true
          break
        }
        block = block.parent()
      }
      if (isBlock) {
        var childs = block[0].children
        for (var i=0; i<childs.length; i++) {
          var p = $('<p></p>')
          p.html(childs[i].innerHTML)
          p.insertBefore(block)
        }
        var prev = block.prev()
        block.remove()
        this._createRangeByElem(prev, false, true)
        this._restoreSelection()
      } else {
        var dom = this._getCommonAncestorContainer()
        while (!dom.parent().equal(this.textEl)) {
          dom = dom.parent()
        }
        var block = $('<blockquote></blockquote>')
        var p = $('<p></p>')
        if (dom.getNodeName() == 'P') {
          p.html(dom.html())
          block.append(p)
        } else if (dom.getNodeName() == 'UL' || dom.getNodeName() == 'OL') {
          var lis = dom[0].querySelectorAll('li')
          for (var i=0; i<lis.length; i++) {
            p = $('<p></p>')
            p.html(lis[i].innerText)
            block.append(p)
          }
        } else {
          p.html('<br>')
          block.append(p)
        }
        block.insertAfter(dom)
        dom.remove()
        this._createRangeByElem(p, false, true)
        this._restoreSelection()
      }
      this._updateMenu()
    },
    setUnorderedlist: function () {
      this._orderListHandler('ul')
    },
    setOrderedlist: function () {
      this._orderListHandler('ol')
    },
    _orderListHandler: function (value) {
      if (this._queryState('InsertOrderedList') || this._queryState('InsertUnorderedList')) {
        var reset = false
        if (this._queryState('InsertOrderedList') && value == 'ul' || this._queryState('InsertUnorderedList') && value == 'ol') {
          reset = true
        }
        this._saveRange()
        var li = this._getCommonAncestorContainer()
        if (li.getNodeName() == 'UL' || li.getNodeName() == 'OL') {
          li = li.getLastChild()
        }
        while (li.getNodeName() != 'LI' || li.equal(this.textEl)) {
          li = li.parent()
        }
        var list = li.parent()
        var next = list.next()
        if (!next[0]) {
          return
        }
        if (next.html() != '<br>' && next.html() != '') {
          var p = $('<p></p>')
          p.html('<br>')
          p.insertAfter(list)
          next = p
        }
        if (li.html() != '') {
          next.html(li.html())
        }
        li.remove();
        if (list.html() == '') {
          list.remove()
        }
        this._createRangeByElem(next, false, true)
        this._restoreSelection()
        if (reset) {
          if (value == 'ol') {
            this.setOrderedlist()
          } else {
            this.setUnorderedlist()
          }
        }
      } else {
        this._saveRange()
        var dom = this._getCommonAncestorContainer()
        while (!dom.parent().equal(this.textEl)) {
          dom = dom.parent()
        }
        var list = null
        if (value == 'ol') {
          list = $('<ol></ol>')
        } else {
          list = $('<ul></ul>')
        }
        var li = $('<li></li>')
        if (dom.getNodeName() == 'BLOCKQUOTE' && dom.html() != '<p><br></p>') {
          var paragraphs = dom[0].querySelectorAll('p')
          for (var i=0; i<paragraphs.length; i++) {
            li.html(paragraphs[i].innerHTML)
            if (i < paragraphs.length-1) {
              list.append(li)
              li = $('<li></li>')
            }
          }
        } else if (dom.html() != '<br>' && dom.html() != '') {
          li.html(dom.text())
        }
        list.append(li)
        list.insertAfter(dom)
        dom.remove()
        this._createRangeByElem(li, false, false)
        this._restoreSelection()
      }
      this._updateMenu()
    },
    imageEditHandler: function (dom) {
      // var el = $('<div class="image-editor-box"><div class="btn-delete"></div><div class="btn-add-desc">图注</div></div>')
      var el = $('<div class="image-editor-box"><div class="tooltip"><div class="btn-add-desc">图注</div><div class="btn-delete">删除</div></div></div>')
      el.attr('contenteditable', 'false')
      dom.append(el)
    },
    insertImageDesc: function (id, text) {
      var dom = document.getElementById(id)
      var desc = dom.querySelector('.img-desc')
      if (desc) {
        if (text) {
          desc.innerHTML = text
        } else {
          desc.remove()
        }
      } else {
        desc = document.createElement('h4')
        desc.className = 'img-desc'
        desc.innerHTML = text
        dom.appendChild(desc)
      }
      var nextEl = $(dom).next();
      this._createRangeByElem(nextEl, true, false)
      this._restoreSelection()
    },
    insertImage: function (imgs) {
      if (typeof imgs === 'string') {
        imgs = JSON.parse(imgs)
      }
      var _this = this
      var node = this._getCommonAncestorContainer()
      
      if (node.equal(this.textEl)) {
        var lastChild = this.textEl.getLastChild()
        this._insertUnAbleEditHandler(lastChild)
      } else {
        while (!node.parent().equal(this.textEl)) {
          node = node.parent()
        }
        this._insertUnAbleEditHandler(node)
      }
      this.focusEditor()
      this._hidePlaceHolder()
      this._executeTimer = setTimeout(function () {
        var html = ''
        imgs.forEach(function (img, index) {
          var rect = _this._calculateImagesRect(img)
          if (img.url) {
            html += '<p class="user-image" style="width:' + rect.width + ';height:' + rect.height + ';" id ="' + img.id + '" data-width="' + img.width + '" data-height="' + img.height + '"><img src="' + img.url + '"></p>'
          } else {
            html += '<div class="image-box" style="width:' + rect.width + ';height:' + rect.height + ';" id="' + img.id + '" data-width="' + img.width + '" data-height="' + img.height + '">图片上传中</div>'
          }
          if (index < imgs.length-1) {
            html += '<p><br></p>'
          }
        })
        _this.insertHTML(html)
        imgs.forEach(function (img) {
          $(document.getElementById(img.id)).attr('contenteditable', 'false')
        })
        var imgbox = $(document.getElementById(imgs[imgs.length-1].id))
        var nextEl = imgbox.next()
        if (nextEl[0] && nextEl.html() == '<br>') {
          nextEl = imgbox.next()
        } else {
          var p = _this._createEmptyP()
          p.insertAfter(imgbox)
          nextEl = p
        }
        _this._executeTimer2 = setTimeout(function () {
          _this._createRangeByElem(nextEl, true, false)
          _this.focusEditor()
          _this._clearEmptyP()
          clearTimeout(_this._executeTimer2)
          clearTimeout(_this._executeTimer)
          _this._updateMenu()
        }, 0)
      }, 0)
    },
    _calculateImagesRect: function (img) {
      var rect = {
        width: 'auto',
        height: 'auto'
      }
      if (img.url && img.url.indexOf('cdn.max-c.com/stickers') > 0) {
        rect.width = '80px'
        rect.height = '80px'
      } else if (img.width && img.height && img.width != 'null' && img.height != 'null') {
        var dpi = img.height / img.width
        if (img.width <= 0.5 * this._clientWidth) {
          rect.width = ~~(this._clientWidth * 0.5) + 'px'
          rect.height = ~~(this._clientWidth * dpi * 0.5) + 'px'
        } else if (img.width < this._clientWidth && img.width > 0.5 * this._clientWidth) {
          rect.width = img.width + 'px'
          rect.height = img.height + 'px'
        } else {
          rect.width = ~~this._clientWidth + 'px'
          rect.height = ~~(this._clientWidth * dpi) + 'px'
        }
      }
      return rect
    },
    _clearEmptyP: function () {
      var childs = this.textEl[0].children
      for (var i=0; i<childs.length; i++) {
        if (childs[i].innerHTML == '') {
          $(childs[i]).remove()
        }
      }
    },
    _insertUnAbleEditHandler: function (node) {
      // 如果没有前一个元素，则在前面添加一个空行
      if (((!node.prev() || !node.prev()[0]) && (node.html() == '<br>' || node.html() == ''))) {
        this._createEmptyP().insertBefore(node)
      }
      if (node.html() == '<br>' || node.html() == '') {
        if (node.prev()[0] && node.prev().attr('contenteditable') != 'false') {
          this._createRangeByElem(node, true, false)
        } else {
          var next = node.next()
          if (next && next[0] && (next.html() == '<br>' || next.html() == '')) {
            this._createRangeByElem(next, true, false)
          } else {
            var p = this._createEmptyP()
            p.insertAfter(node)
            this._createRangeByElem(p, true, false)
          }
        }
      } else {
        var p = this._createEmptyP()
        p.insertAfter(node)
        this._createRangeByElem(p, true, false)
      }
    },
    uploadSuccessImage: function (imgs) {
      if (typeof imgs === 'string') {
        imgs = JSON.parse(imgs)
      }
      var _this = this
      imgs.forEach(function (img) {
        _this.insertURLImage([{
          url: img.url, 
          id: img.id
        }])
      })
    },
    uploadFailedImage: function (imgs) {
      if (typeof imgs === 'string') {
        imgs = JSON.parse(imgs)
      }
      imgs.forEach(function (img) {
        var imgbox = $('#' + img.id)
        imgbox.attr('data-status', 'failed')
        imgbox.html('上传失败，点击重新上传')
      })
    },
    insertURLImage: function (imgs) {
      if (typeof imgs === 'string') {
        imgs = JSON.parse(imgs)
      }
      var _this = this
      imgs.forEach(function (img) {
        var imgEl = document.createElement('img')
        imgEl.onload = function () {
          if (img.id && typeof img.id == 'string') {
            var imgbox = $(document.getElementById(img.id))
            var box = $('<div class="box"></div>')
            box.attr('contenteditable', 'false')
            box.attr('style', imgbox.attr('style'))
            box.html('<img src="' + img.url + '">')
            imgbox.html('')
            imgbox.append(box)
            imgbox.attr('style', '')
            imgbox.attr('data-status', 'success')
            imgbox.removeClass('image-box')
            imgbox.addClass('user-image')
            _this._updateMenu()
          } else {
            _this.insertImage([{
              id: 'network-img-' + _this._networkImageId++,
              url: img.url
            }])
          }
        }
        imgEl.onerror = function (e) {
          if (img.id && typeof img.id == 'string') {
            var imgbox = $(document.getElementById(img.id))
            imgbox.html('上传失败，点击重新上传')
            imgbox.attr('data-status', 'failed')
          }
        }
        imgEl.src = img.url
      })
      this._hidePlaceHolder()
    },
    insertLink: function (url, title) {
      this._restoreSelection()
      this.insertHTML('<a href="' + url + '" target="_blank">' + title + '</a>')
      this._hidePlaceHolder()
    },
    updateLink: function (url, title) {
      this.insertLink(url, title)
    },
    removeLink: function () {
      var node = this._getCommonAncestorContainer()
      while ((node.nodeType == 3 && node.nodeName != 'A') || node.nodeName == 'BODY') {
        node = node.parentNode || node.parentElement
      }
      $(node).remove()
    },
    setTextToSubject: function (text) {
      var start = this._currentText.data.indexOf('#')
      var data = this._currentText.data.slice(0, start) + this._currentText.data.slice(this._currentIndex)
      if (!data || data.length == 0) {
        data = '_'
        start = 1
      }
      this._currentText.data = data
      this._createRangeByText(this._currentText, start)
      this._restoreSelection()
      if (data == '_') {
        document.execCommand('delete')
      }
      this.insertHTML('&nbsp;<a href="heybox://open_subject" target="_blank">' + text + '</a>&nbsp;')
    },
    insertSubject: function (text) {
      this._restoreSelection()
      var sel = window.getSelection()
      var char = sel.focusNode
      if (char.nodeType == 3 && char.data[sel.focusOffset-1] == '#') {
        this._createRangeByText(sel.focusNode, sel.focusOffset-1, sel.focusOffset)
        this._restoreSelection()
        document.execCommand('delete')
      }
      this.insertHTML('&nbsp;<a href="heybox://open_subject" target="_blank">' + text + '</a>&nbsp;')
      this._hidePlaceHolder()
    },
    insertFriend: function (text, user_id) {
      this._restoreSelection()
      var sel = window.getSelection()
      var char = sel.focusNode
      if (char.nodeType == 3 && char.data[sel.focusOffset-1] == '@') {
        this._createRangeByText(sel.focusNode, sel.focusOffset-1, sel.focusOffset)
        this._restoreSelection()
        document.execCommand('delete')
      }
      this.insertHTML('&nbsp;<a data-user-id="' + user_id + '" href="heybox://" target="_blank">' + text + '</a>&nbsp;')
      this._hidePlaceHolder()
      // var userat = document.querySelectorAll('a[data-user-id]')
      // for (var i=0; i<userat.length; i++) {
      //   userat[i].setAttribute('contenteditable', 'false')
      // }
    },
    undo: function () {
      document.execCommand('undo', false, undefined)
    },
    redo: function () {
      document.execCommand('redo', false, undefined)
    },
    insertGame: function (game) {
      if (typeof game == 'string') {
        game = JSON.parse(game)
      }
      var _game = {
        steam_appid: game.steam_appid,
        game_type: game.game_type,
        name: game.name,
        image: game.image
      }
      if (game.appid) {
        _game.appid = game.appid
      }
      var _this = this
      var node = this._getCommonAncestorContainer()

      this.focusEditor()

      if (node.equal(this.textEl)) {
        var lastChild = this.textEl.getLastChild()
        this._insertUnAbleEditHandler(lastChild)
      } else {
        while (!node.parent().equal(this.textEl)) {
          node = node.parent()
        }
        this._insertUnAbleEditHandler(node)
      }

      this._executeTimer = setTimeout(function () {
        var html = _this._getGameHTML(game)
        _this.insertHTML(html)
        var gamebox = $(document.getElementById('game-' + (game.steam_appid || game.appid)))
        gamebox.attr('contenteditable', 'false')
        var nextEl = gamebox.next()
        if (nextEl[0] && nextEl.html() == '<br>') {
          nextEl = gamebox.next()
        } else {
          var p = _this._createEmptyP()
          p.insertAfter(gamebox)
          nextEl = p
        }
        _this._executeTimer2 = setTimeout(function () {
          _this._createRangeByElem(nextEl, true, false)
          _this.focusEditor()
          _this._clearEmptyP()
          clearTimeout(_this._executeTimer2)
          clearTimeout(_this._executeTimer)
        }, 0)
      }, 0)
      this._hidePlaceHolder()
    },
    _getGameHTML: function (game) {
      var html = '<div class="mention-game__box" id="game-' + (game.steam_appid || game.appid) + '" data-gameid="' + (game.steam_appid || game.appid) + '" data-game-type="' + game.game_type + '">' +
        '<div class="mention-game"><div class="img-box">' + 
        '<div class="inner-box"><img src="' + game.image + '"></div>' +
        '</div><div class="info-box unfollowing">' +
        '<div class="name-box"><p class="name">' + game.name + '</p></div>' +
        '</div></div></div>'
      return html
    },
    insertText: function (text) {
      this._restoreSelection()
      document.execCommand('insertText', false, text)
    },
    insertHTML: function (html) {
      this._restoreSelection()
      document.execCommand('insertHTML', false, html)
    },
    backspace: function () {
      var current = this._currentText.nodeType == 3 ? $(this._currentText.parentNode) : $(this._currentText)
      while (!current.parent().equal(this.textEl)) {
        current = current.parent()
      }
      if (this._currentText.nodeType == 3) {
        if (this._currentIndex == 0) {
          var prev = this._currentText.previousSibling || this._currentText.previousElementSibling
          if (prev) {
            if (prev.nodeType == 3) {
              this._currentText = prev
              this._currentIndex = (prev.data || prev.textContent).length
            } else {
              var p_prev = prev.previousSibling || prev.previousElementSibling
              if (p_prev.nodeType == 3) {
                this._currentText = p_prev
                this._currentIndex = (p_prev.data || p_prev.textContent).length
              }
              $(prev).remove()
            }
          }
        } else {
          this._currentText.deleteData(--this._currentIndex, 1)
        }
      } else {
        $(this._currentText).remove()
      }
      if (current.html() == '' && current.getNodeName() == 'P') {
        current.html('<br>')
      }
    },
    insertEmoji: function (name) {
      var node = this._getCommonAncestorContainer()
      var str = this._currentText.data || this._currentText.textContent
      // var fragment = document.createDocumentFragment()
      if (node[0].childNodes.length == 1 && node[0].childNodes[0].nodeName == 'BR') {
        // 只有一个br子节点
        var text = document.createTextNode(name)
        var t = node[0].childNodes[0]
        if (t.replaceWith) {
          t.replaceWith(text)
        } else {
          t.parentNode.replaceChild(text, t)
        }
        this._createRangeByText(text, name.length)
      } else {
        // text节点长度为0或光标位置等于text长度
        if ((str.length == 0 || str.length == this._currentIndex)) {
          var text = document.createTextNode(name)
          // node.append($(emoji))
          node[0].appendChild(text)
          this._createRangeByText(text, name.length)
        } else if (this._currentIndex == 0) {
          // 光标在text前面
          // $(emoji).insertBefore($(this._currentText))
          var text = document.createTextNode(name)
          if (this._currentText.nodeName == 'P') {
            node[0].insertBefore(text, this._currentText.childNodes[0])
          } else {
            node[0].insertBefore(text, this._currentText)
          }
          this._currentText.normalize()
          this._createRangeByText(this._currentText.childNodes[0], name.length)
        } else {
          // 光标在text中间
          var text = document.createTextNode(str.substr(0, this._currentIndex) + name + str.slice(this._currentIndex))
          if (this._currentText.replaceWith) {
            this._currentText.replaceWith(text)
          } else {
            this._currentText.parentNode.replaceChild(text, this._currentText)
          }
          this._createRangeByText(text, this._currentIndex + name.length)
        }
      }
      this._hidePlaceHolder()
    },
    _createRangeByText: function (text, start, end) {
      if (!end) {
        end = start
      }
      var range = document.createRange();
      range.selectNode(text);
      range.setStart(text, start)
      range.setEnd(text, end)
      this._currentText = text
      this._currentIndex = start || 0
      this._saveRange(range);
    },
    focusEditor: function () {
      if (!this._currentRange) {
        this.textEl[0].focus()
      }
      this._restoreSelection()
    },
    blurEditor: function () {
      this._saveSelectionInfo();
      this._saveRange();
      this.textEl[0].blur();
    },
    _saveSelectionInfo: function () {
      var selection = window.getSelection();
      this._currentText = selection.focusNode
      this._currentIndex = selection.focusOffset
    },
    _createEmptyP: function () {
      var p = document.createElement('p')
      p.innerHTML = '<br>'
      return $(p)
    },
    _init: function () {
      document.execCommand('defaultParagraphSeparator', false, 'p')
      document.execCommand('styleWithCSS', null, false)
    },
    _restoreSelection: function () {
      if (this._currentRange) {
        var selection = window.getSelection()
        selection.removeAllRanges()
        selection.addRange(this._currentRange)
      }
    },
    _createRangeByElem: function ($elem, toStart, isContent) {
      if (!$elem.length) {
        return;
      }
      var elem = $elem[0];
      var range = document.createRange();
      if (isContent) {
        range.selectNodeContents(elem.childNodes[0]);
      } else {
        range.selectNode(elem);
      }
      if (typeof toStart === 'boolean') {
        range.collapse(toStart);
      } else {
        range.setStart(elem.childNodes[0], 0)
      }
      this._saveRange(range);
    },
    _initialRange: function () {
      if (!this._currentRange) {
        var lastChild = this.textEl.getLastChild()
        if (lastChild.html() == '<br>' || lastChild.html() == '<br/>') {
          this._createRangeByElem(lastChild, true, false);
        } else {
          lastChild = $('<p><br></p>')
          this.textEl.append(lastChild)
          this._createRangeByElem(lastChild, true, false);
        }
      }
    },
    _saveRange: function (_range) {
      if (_range) {
        this._currentRange = _range
      } else {
        var selection = window.getSelection()
        if (selection.rangeCount == 0 || selection.type == 'None') {
          return
        }
        var range = selection.getRangeAt(0)
        var commonAncestorDom = this._getCommonAncestorContainer(range)
        if (!commonAncestorDom) {
          return
        }
        if (commonAncestorDom.parentUntil('.editor-editable')) {
          this._currentRange = range
        }
      }
      this._contentEndTagHandler()
    },
    _contentEndTagHandler: function () {
      var lastChild = this.textEl.getLastChild()
      if (lastChild[0]) {
        var html = lastChild.html()
        if (html != '<br>' && html != '') {
          this.textEl.append(this._createEmptyP())
        }
      }
    },
    _getCommonAncestorContainer: function (range) {
      range = range || this._currentRange
      var elem = null
		  if (range) {
        elem = range.commonAncestorContainer
        elem = elem.nodeType === 1 ? elem : elem.parentNode
        return $(elem)
      } else {
        return this.textEl.getLastChild()
      }
    },
    _queryState: function (command) {
      return document.queryCommandState(command)
    },
    _hidePlaceHolder: function () {
      this.textEl.removeClass('show-placeholder')
    },
    _showPlaceHolder: function () {
      this.textEl.addClass('show-placeholder')
    },
    _updateMenu: function () {
      var now = new Date().getTime();
      if (now - this.lastUpdateTimestamp < 300) {
        return
      }
      this.lastUpdateTimestamp = now
      var _this = this
      if (this.textEl.html() == '<p><br></p>' || this.textEl.html() == '') {
        this._showPlaceHolder()
      } else {
        this._hidePlaceHolder()
      }
      this._timer = setTimeout(function () {
        _this._saveRange()
        var menus = []
        var forbiddens = []
        var formatBlock = document.queryCommandValue('formatBlock')
        if (formatBlock.length > 0) {
          menus.push(formatBlock)
        }
        if (_this._queryState('bold')) {
          if (formatBlock.indexOf('h') != 0) {
            menus.push('bold')
          } else {
            forbiddens.push('bold')
          }
        }
        if (_this._queryState('InsertOrderedList')) {
          menus.push('orderedList')
          forbiddens.push('h2')
        }
        if (_this._queryState('InsertUnorderedList')) {
          menus.push('unorderedList')
          forbiddens.push('h2')
        }
        
        var dom = _this._getCommonAncestorContainer()
        if (!dom[0] || dom.equal(_this.textEl)) {
          return
        }
        while(!dom.parent().equal(_this.textEl) || dom.getNodeName() == 'BLOCKQUOTE') {
          if (dom.getNodeName() == 'BLOCKQUOTE') {
            menus.push('blockquote')
            break
          }
          dom = dom.parent()
        }
        if (menus.length > 0) {
          var url = "hbEditor://" + encodeURI(JSON.stringify({
            "protocol_type": "callback_selection_style", 
            "items": menus,
            "forbiddens": forbiddens
          }))
          window.location.href = url
        }
        clearTimeout(_this._timer)
        _this.timer = null
      }, 50)
    },
  }

  function Editor () {
    this.exec = null
    this.selection = null
    this._clientWidth = (document.body.clientWidth || document.documentElement.clientWidth) - 24
    this.ppppp = {}
    this._init()
  }
  
  Editor.prototype = {
    constructor: Editor,
    _init: function () {
      this._initDom()
      this.execute = new Command()
      this._initDomEvent()
    },
    _initDom: function () {
      this.editorEl = $('#editor')
      this.textEl = $('<div></div>')
      this.textEl.attr('contenteditable', 'true').attr('spellcheck', 'false').attr('translate', 'no').addClass('editor-editable')
      this.textEl.append($('<p><br></p>'))
      this.editorEl.append(this.textEl)
      if (window.MutationObserver) {
        this.ObserverEditor()
      }
    },
    _totalHeight: function (childs) {
      var height = 0
      for (var i=0; i<childs.length; i++) {
        height += (childs[i].clientHeight || childs[i].offsetHeight)
      }
      return height
    },
    _initDomEvent: function () {
      var _this = this
      this.touchPos = {
        start: null,
        end: null
      }
      this.touchTime = {
        start: 0,
        end: 0
      }
      this.textEl.on('input', function (e) {
        // if (e.data == '@') {
        //   window.location.href = 'hbEditor://' + encodeURI(JSON.stringify({
        //     protocol_type: "callback_open_friend_list",
        //   }))
        // } else 
        if (e.data == '#') {
          _this.execute._saveSelectionInfo()
          var matches = _this.execute._currentText.data.match(/(#.*?#)/gi)
          if (matches && matches[0]) {
            if (matches[0].length <= 32) {
              _this.execute.setTextToSubject(matches[0])
            }
          } else {
            window.location.href = 'hbEditor://' + encodeURI(JSON.stringify({
              protocol_type: "callback_open_subject_list",
            }))
          }
        }
      })
      this.textEl.on('keydown', function (e) {
        if (e.keyCode === 8) {
          var txtHtml = _this.textEl.html().toLowerCase().trim()
          if (txtHtml === '' || txtHtml === '<p></p>') {
            _this.textEl.html('<p><br></p>')
            e.preventDefault()
            return
          }
          if (txtHtml === '<p><br></p>') {
            e.preventDefault()
            return
          }
          _this._deleteEventHandler(e)
          _this.execute._updateMenu()
        } else if (e.keyCode === 13) {
          _this._enterEventHandler(e)
          _this.execute._updateMenu()
        } else {
          _this.execute._hidePlaceHolder()
        }
      })
      this.textEl.on('paste', function (e) {
        e.preventDefault();
        _this._pasteEventHandler(e)
        _this.execute._hidePlaceHolder()
      })
      this.textEl.on('click', function (e) {
        _this.execute._updateMenu()
      }, false)
      this.textEl.on('focus', function (e) {
        _this.execute._initialRange()
      })
      this.textEl.on('selectionchange', function (e) {
        _this.execute._updateMenu()
      })
      this.textEl.on('touchstart', function (e) {
        var pos = {
          x: e.touches[0].clientX || e.touches[0].pageX,
          y: e.touches[0].clientY || e.touches[0].pageY
        }
        _this.touchPos.start = pos
        _this.touchTime.start = new Date().getTime()
      })
      this.textEl.on('touchmove', function (e) {
        var pos = {
          x: e.touches[0].clientX || e.touches[0].pageX,
          y: e.touches[0].clientY || e.touches[0].pageY
        }
        _this.touchPos.end = pos
        _this.touchTime.end = new Date().getTime()
      })
      this.textEl.on('touchend', function (e) {
        var isTap = _this.checkTap()
        _this.touchPos.start = null
        _this.touchPos.end = null
        _this.touchTime.start = 0
        _this.touchTime.end = 0
        if (isTap) {
          _this._touchEndEventHandler(e)
          if (_this.isUnableEdit) {
            return
          }
        }
      })
    },
    checkTap: function () {
      if (!this.touchPos.end) {
        return true
      }
      var x = Math.abs(~~this.touchPos.end.x) - Math.abs(~~this.touchPos.start.x)
      var y = Math.abs(~~this.touchPos.end.y) - Math.abs(~~this.touchPos.start.y)
      var distance = Math.sqrt((x * x) + (y * y))
      var time = this.touchTime.end - this.touchTime.start
      return (distance <= 10 && time <= 300)
    },
    _pasteEventHandler: function (e) {
      var _this = this
      var clipboardData = e.clipboardData || e.originalEvent && e.originalEvent.clipboardData
      var pasteText = void 0
      if (clipboardData == null) {
        pasteText = window.clipboardData && window.clipboardData.getData('text')
      } else {
        pasteText = clipboardData.getData('text/plain')
      }
      pasteText = pasteText.replace(/((\n)|(\r)|(\r\n))+/ig, '\n')
      pasteArr = pasteText.split(/(\n)/gi).filter(function (text) {
        return text && text != '\n'
      })
      this.execute._saveRange()
      this.execute._saveSelectionInfo()
      var current = this.execute._getCommonAncestorContainer()
      if (!current.parent().equal(this.textEl)) {
        current = current.parent()
      }
      pasteArr.forEach(function (text, index) {
        var matches = text.match(/(#.*?#)/gi)
        if (matches && matches.length > 0) {
          matches.forEach((subject) => {
            text = text.replace(subject, '&nbsp;<a href="heybox://open_subject" target="_blank">' + subject + '</a>&nbsp;')
          })
        }
        text = text.replace(/(\s)+/gi, ' ')
        if (index == 0) {
          _this.execute.insertHTML(text)
        } else {
          if (index == 1) {
            document.execCommand('insertParagraph')
          }
          var p = $('<p></p>')
          p.html(text)
          p.insertAfter(current)
          current = p
        }
      })
    },
    _replaceHtmlSymbol: function (html) {
      if (html == null) {
        return '';
      }
      return html.replace(/</gm, '&lt;').replace(/>/gm, '&gt;').replace(/"/gm, '&quot;').replace(/(\r\n|\r|\n)/g, '<br/>');
    },
    _touchEndEventHandler: function (e) {
      var dom = $(e.target)
      this.isUnableEdit = false
      if (dom.getNodeName() == 'A') {
        e.preventDefault();
        var url = ''
        // if (dom.attr('data-user-id')) {
        //   url = 'heybox://' + encodeURI(JSON.stringify({
        //     protocol_type: 'openUser',
        //     user_id: dom.attr('data-user-id')
        //   }))
        // } else 
        if (dom.attr('href') == 'heybox://open_subject') {
          url = 'heybox://' + encodeURI(JSON.stringify({
            protocol_type: 'openBBSTag',
            tag: dom.text().replace(/#/gi, '').slice(0, 30)
          }))
        } else {
          $('.image-editor-box').remove()
          this.execute._createRangeByElem(dom, 0, true)
          this.execute._restoreSelection()
          url = 'hbEditor://' + encodeURI(JSON.stringify({
            "protocol_type": "callback_link_tap", 
            "url": dom.attr('href'),
            "title": dom.text()
          }))
        }
        window.location.href = url
      } else if (dom.hasClass('image-box')) {
        e.preventDefault()
        $('.image-editor-box').remove()
        if (dom.attr('id') && dom.attr('data-status') === 'failed') {
          dom.html('图片上传中')
          dom.attr('data-status', 'uploading')
          var url = 'hbEditor://' + encodeURI(JSON.stringify({
            "protocol_type": "callback_img_upload"
          }))
          window.location.href = url
        }
        this.isUnableEdit = true
      } else if (dom.getNodeName() == 'IMG') {
        e.preventDefault()
        var parent = dom.parent()
        if (parent.hasClass('user-image')) {
          var box = $('<div class="box"></div>')
          box.attr('style', parent.attr('style'))
          box.attr('contenteditable', 'false')
          parent.attr('style', '')
          box.html(parent.html())
          parent.html('')
          parent.append(box)
          this.execute.imageEditHandler(box)
        } else if (parent.hasClass('box')) {
          this.execute.imageEditHandler(parent)
        }
        this.isUnableEdit = true
      } else if (dom.hasClass('btn-delete')) {
        e.preventDefault()
        var imgbox = dom.parent()
        while (!imgbox.hasClass('user-image')) {
          imgbox = imgbox.parent()
        }
        var prev = imgbox.prev()
        if (!prev || !prev[0]) {
          prev = imgbox.next()
        }
        imgbox.remove()
        this.execute._createRangeByElem(prev, false, false)
        this.execute._restoreSelection()
        this.isUnableEdit = true
      } else if (dom.hasClass('btn-add-desc')) {
        e.preventDefault()
        this.execute.blurEditor()
        var imgbox = dom.parent()
        while (!imgbox.hasClass('user-image')) {
          imgbox = imgbox.parent()
        }
        var id = imgbox.attr('id')
        if (!id) {
          id = 'network-img-' + this.execute._networkImageId++
          imgbox.attr('id', id)
        }
        var h4 = imgbox[0].querySelector('.img-desc')
        var desc = ''
        if (h4)  {
          desc = h4.innerHTML
        }
        this.isUnableEdit = true
        window.location.href = 'hbEditor://' + encodeURI(JSON.stringify({
          protocol_type: "callback_imageDesc_tap",
          id: id,
          text: desc
        }))
      } else if (dom.hasClass('img-desc')) {
        e.preventDefault()
        this.execute.blurEditor()
        var imgbox = dom.parent()
        while (!imgbox.hasClass('user-image')) {
          imgbox = imgbox.parent()
        }
        var id = imgbox.attr('id')
        if (!id) {
          id = 'network-img-' + this.execute._networkImageId++
          imgbox.attr('id', id)
        }
        this.isUnableEdit = true
        window.location.href = 'hbEditor://' + encodeURI(JSON.stringify({
          protocol_type: "callback_imageDesc_tap",
          id: id,
          text: dom.text()
        }))
      } else if (dom.hasClass('image-editor-box')) {
        this.isUnableEdit = true
      } else {
        $('.image-editor-box').remove()
      }
    },
    _isImageBox: function (dom) {
      return dom.hasClass('image-box') || dom.parent().hasClass('image-box')
    },
    _enterEventHandler: function (e) {
      var dom = this.execute._getCommonAncestorContainer()
      var isBlock = false
      if (dom.equal(this.textEl)) {
        this.execute._saveRange()
        dom = this.execute._getCommonAncestorContainer()
      }
      while(!dom.parent().equal(this.textEl) || dom.getNodeName() == 'BLOCKQUOTE') {
        if (dom.getNodeName() == 'BLOCKQUOTE') {
          isBlock = true
          break
        }
        dom = dom.parent()
      }
      if (isBlock) {
        var last = dom.getLastChild()
        if (last.html() == '<br>' || last.html() == '') {
          e.preventDefault()
          var next = dom.next()
          if (!next[0] || next.getNodeName() != 'P' || (next.html() != '<br>' && next.html == '')) {
            var next = $('<p><p>')
            next.html('<br>')
            next.insertAfter(dom)
          }
          last.remove()
          last = dom.getLastChild()
          this.execute._createRangeByElem(next, false, false)
          this.execute._restoreSelection()
        }
      }
    },
    _deleteEventHandler: function (e) {
      var _this = this
      var current = this.execute._getCommonAncestorContainer()
      if (current.equal(this.textEl)) {
        this.execute._saveRange()
        current = this.execute._getCommonAncestorContainer()
      }
      while(!current.parent().equal(this.textEl)) {
        current = current.parent()
      }
      var prev = current.prev()
      var sel = window.getSelection()
      var focusnode = sel.focusNode
      if (focusnode.nodeType == 3) {
        focusnode = focusnode.parentNode
      }
      if (prev && prev[0] && prev.attr('contenteditable') == 'false' && (sel.anchorOffset==0 && (!focusnode.previousSibling || $(focusnode.previousSibling).attr('contenteditable') == 'false')) && current.attr('contenteditable') != 'false') {
        e.preventDefault()
        var range = document.createRange();
        var mprev = prev.prev()
        if (mprev && mprev[0]) {
          range.setStart(mprev[0], mprev[0].childNodes.length)
        } else {
          range.setStart(prev[0], 0)
        }
        range.setEnd(current[0], 0)
        if (prev.hasClass('user-image')) {
          this.execute.imageEditHandler($(prev[0].querySelector('.box')))
        }
        this.execute._saveRange(range);
        this.execute._restoreSelection()
      } else if (current && current[0] && current.hasClass('editor-editable')) {
        document.execCommand('forwardDelete')
        e.preventDefault()
      } else if (focusnode.getAttribute('href') == 'heybox://open_subject') {
        setTimeout(function () {
          if (!/(#.*?#)/gi.test(focusnode.innerText)) {
            _this._deleteFuncHandler(focusnode)
          }
        }, 30)
      } else if (focusnode.getAttribute('data-user-id')) {
        setTimeout(function () {
          _this._deleteFuncHandler(focusnode)
        }, 30)
      }
      this._timer = setTimeout(function () {
        var html = _this.textEl.html()
        html = html.replace(/<p><\/p>/gi, '')
        if (html == '') {
          _this.textEl.html('<p><br></p>')
        }
        html = html.replace(/<p><br><\/p>/gi, '')
        if (html == '') {
          _this.execute._showPlaceHolder()
        }
      }, 0)
    },
    _deleteFuncHandler: function (focusnode) {
      var text = focusnode.innerText
      var _parent = focusnode.parentElement || focusnode.parentNode
      this.execute._createRangeByText(focusnode.childNodes[0], 0, text.length)
      this.execute._restoreSelection()
      document.execCommand('delete')
      document.execCommand('delete')
      this.execute.insertText(text)
      _parent.normalize()
    },
    updateArticleInfo: function (data, games) {
      if (data.status == 'ok') {
        if (window._isLink) {
          _titleDom.value = data.result.link.title
        }
        var text = eval(data.result.link.text)
        if (text[0].type == 'html') {
          var dom = document.createElement('div')
          dom.innerHTML = text[0].text
          this._draftImageHandler(dom)
          if (games && games.length > 0) {
            this._draftGameHandler(dom, games)
          }
          var html = dom.innerHTML
          var reg = /<p class="news-source">.*?<\/p>/ig
          var source = html.match(reg)
          if (source && source[0]) {
            html = html.replace(source[0], '')
          }
          this.textEl.html(html)
          // if (dom.children[0] && dom.children[0].getAttribute('contenteditable') == 'false') {
          //   html = '<p><br></p>' + html
          // }
          // var last = dom.lastElementChild || dom.lastChild
          // if (last && last.getAttribute('contenteditable') == 'false') {
          //   html += '<p><br></p>'
          // }
          // this.textEl.html(html)
          this.execute._hidePlaceHolder()
        }
      }
    },
    setArticleInfo: function (params, games) {
      if (typeof params == 'string') {
        params = params.replace(/\\\"/, '\\"')
        params = JSON.parse(params)
      }
      var dom = document.createElement('div')
      dom.innerHTML = params.content
      this._draftImageHandler(dom)
      if (games && games.length > 0) {
        this._draftGameHandler(dom, games)
      }
      var html = dom.innerHTML
      var reg = /<p class="news-source">.*?<\/p>/ig
      var source = html.match(reg)
      if (source && source[0]) {
        html = html.replace(source[0], '')
      }
      if (dom.children[0] && dom.children[0].getAttribute('contenteditable') == 'false') {
        html = '<p><br></p>' + html
      }
      var last = dom.lastElementChild || dom.lastChild
      if (last && last.getAttribute('contenteditable') == 'false') {
        html += '<p><br></p>'
      }
      this.textEl.html(html)
      if (window._isLink) {
        _titleDom.value = params.title || ''
      }
      this.execute._hidePlaceHolder()
    },
    _draftGameHandler: function (dom, games) {
      var imgs = dom.querySelectorAll('img[data-gameid]')
      if (imgs.length == 0) {
        return
      }
      for (var i=0;i<imgs.length; i++) {
        var gameid = imgs[i].getAttribute('data-gameid')
        var game = this._findGameFromGames(games, gameid)
        if (game) {
          var html = this.execute._getGameHTML(game)
          var div = document.createElement('div')
          div.innerHTML = html
          var gm = div.querySelector('.mention-game__box')
          gm.setAttribute('contenteditable', 'false')
          var parent = imgs[i].parentNode
          if (parent.nodeName == 'P') {
            // parent.replaceWith(gm)
            parent.parentNode.replaceChild(gm, parent)
          } else {
            // imgs[i].replaceWith(gm)
            imgs[i].parentNode.replaceChild(gm, imgs[i])
          }
        }
      }
    },
    _findGameFromGames: function (games, id) {
      if (id &&games && games.length > 0) {
        var game = false
        for (var i=0;i<games.length;i++) {
          if (games[i].steam_appid == id || games[i].appid == id) {
            game = games[i];
            break
          }
        }
        if (game) {
          return game
        }
        return false
      } else {
        return false
      }
    },
    _draftImageHandler: function (dom) {
      var imgs = dom.querySelectorAll('img:not([data-gameid]):not([data-wikiid])')
      for (var i=0;i<imgs.length;i++) {
        var img = {
          url: imgs[i].getAttribute('data-original') || imgs[i].src,
          width: imgs[i].getAttribute('data-width'),
          height: imgs[i].getAttribute('data-height')
        }

        var rect = hb_editor.execute._calculateImagesRect(img)
        var div = document.createElement('div')
        div.className = 'user-image'
        div.setAttribute('contenteditable', 'false')
        div.setAttribute('style', 'width:' + rect.width + ';height:' + rect.height)
        div.setAttribute('data-width', img.width)
        div.setAttribute('data-height', img.height)
        var content = '<img src="' + img.url + '">'

        var desc = ''
        var $imgNext = $(imgs[i]).next()
        if ($imgNext[0] && $imgNext.hasClass('img-desc')) {
          desc = $imgNext[0].innerText
        } else {
          $imgNext = $(imgs[i]).parent().next()
          if ($imgNext[0] && $imgNext.hasClass('img-desc')) {
            desc = $imgNext[0].innerText
          }
        }
        if (desc) {
          content += '<h4 class="img-desc">' + desc + '</h4>'
          $imgNext.remove()
        }
        div.innerHTML = content

        var parent = imgs[i].parentNode
        var nextEl = $(parent).next();
        if (parent.nodeName == 'P') {
          parent.parentNode.replaceChild(div, parent)
        } else {
          imgs[i].parentNode.replaceChild(div, imgs[i])
        }
        if (!nextEl || !nextEl[0] || nextEl.html() != '<br>') {
          $('<p><br></p>').insertBefore(nextEl)
        }
      }
    },
    getArticleInfo: function () {
      var doms = this.textEl[0].querySelectorAll('.user-image')
      var imgs = []
      var img = null
      for(var i=0; i<doms.length; i++) {
        img = doms[i].querySelector('img')
        var params = {
          type: 'img',
          url: img.src
        }
        var w = doms[i].getAttribute('data-width')
        var h = doms[i].getAttribute('data-height')
        if (w) {
          params.width = w
        }
        if (h) {
          params.height = h
        }
        imgs.push(params)
      }
      var content = this.articleContentHandler()
      var user_id = []
      var subject_text = []

      var subject_dom = document.querySelectorAll('a[href]')
      for (var subject_index = 0; subject_index<subject_dom.length; subject_index++) {
        if (subject_dom[subject_index].href == 'heybox://open_subject') {
          subject_text.push(subject_dom[subject_index].innerText.replace(/#/gi, '').slice(0, 30))
        } else if (subject_dom[subject_index].href == 'heybox://open_user') {
          user_id.push(user_id_dom[user_index].getAttribute('data-user-id'))
        }
      }

      var result = {
        content: content.content, // 帖子内容
        desc: content.desc, // 描述
        imgs: imgs, // 图片
        text: content.text, // 纯文本
        user_ids: user_id.join(','),
        subject_ids: JSON.stringify(subject_text)
      }
      if (window._isLink) {
        result.title = window._titleDom.value // 标题
      }
      return result
    },
    articleContentHandler: function () {
      var html = this.textEl.html()
      var fragment = document.createElement('div')
      fragment.innerHTML = html
      this._postContentImageHandler(fragment)
      this._postContentGameHandler(fragment)
      var html = fragment.innerHTML
      html = html.replace(/<script.*?>.*?<\/script>/ig, '')
      html = html.replace(/<iframe.*?>.*?<\/iframe>/ig, '')
      html = html.replace(/<strong>/gi, '<b>').replace(/<\/strong>/gi, '</b>')
      html = html.replace(/(<span.*?>|<\/span>)/ig, '')
      html = html.replace(/<p><\/p>/ig, '<p><br></p>')
      html = html.replace(/(<p><br><\/p>)+/ig, '<p><br></p>')
      html = html.replace(/on(click|error|load)="?(.*?)"?/gi, '').replace(/alert\(.*?\)/gi, '').replace(/javascript:/gi, '')
      if (html.slice(0, 11) == '<p><br></p>') {
        html = html.slice(11)
      }
      if (html.slice(-11) == '<p><br></p>') {
        html = html.slice(0, -11)
      }
      var ret = {
        content: html,
        text: fragment.innerText,
        desc: fragment.innerText.replace(/请输入图片描述/g, '').substr(0, 100),
      }
      return ret
    },
    _postContentImageHandler: function (dom) {
      var failedImages = dom.querySelectorAll('.image-box')
      var succImages = dom.querySelectorAll('.user-image')
      for (var i=0; i<failedImages.length; i++) {
        $(failedImages[i]).remove()
      }
      for (var i=0; i<succImages.length; i++) {
        var p = document.createElement('p')
        var img = document.createElement('img')
        img.setAttribute('data-original', succImages[i].querySelector('img').src)
        img.setAttribute('data-width', succImages[i].getAttribute('data-width') || succImages[i].naturalWidth)
        img.setAttribute('data-height', succImages[i].getAttribute('data-height') || succImages[i].naturalHeight)
        img.className = 'lazy'
        p.appendChild(img)
        
        var desc = succImages[i].querySelector('.img-desc')
        console.log(desc)
        if (desc && desc.innerText) {
          var h4 = document.createElement('h4')
          h4.className = 'img-desc'
          h4.innerText = desc.innerText
          p.appendChild(h4)
        }

        succImages[i].parentNode.replaceChild(p, succImages[i])
      }
    },
    _postContentGameHandler: function (dom) {
      var games = dom.querySelectorAll('.mention-game__box')
      for (var i=0; i<games.length; i++) {
        var p = document.createElement('p')
        var img = document.createElement('img')
        img.setAttribute('data-gameid', games[i].getAttribute('data-gameid'))
        img.src = games[i].querySelector('.inner-box img').src
        p.appendChild(img)
        // games[i].replaceWith(p)
        games[i].parentNode.replaceChild(p, games[i])
      }
    },
    setPlaceHolder: function (text) {
      if (text) {
        this.textEl.attr('placeholder', text)
        this._setPlaceHolderHandler()
      }
    },
    _setPlaceHolderHandler: function () {
      if (this.textEl.html() == '<p><br></p>' || this.textEl.html() == '') {
        this.execute._showPlaceHolder()
      } else {
        this.execute._hidePlaceHolder()
      }
    },
    ObserverEditor: function () {
      var _this = this
      var observer = new MutationObserver(function (mutationsList) {
        if (_this.textEl[0].childNodes && _this.textEl[0].childNodes[0] && (_this.textEl[0].childNodes[0].nodeType == 3 || _this.textEl[0].childNodes[0].nodeName == 'SPAN')) {
          _this.textEl.html('<p><br></p>')
        }
        // _this.execute._saveRange()
      })
      observer.observe(this.textEl['0'], {
        attributes: true,
        childList: true,
        subtree: true
      });
    }
  }
  if (window.VConsole && typeof VConsole == 'function') {
    window.vConsole = new VConsole();
  }
  
  if (document.querySelector('.title')) {
    window._isLink = true
    window._titleDom = document.querySelector('#title')
    window._titleDom.addEventListener('focus', function (e) {
      window.location.href = 'hbEditor://' + encodeURI(JSON.stringify({
        protocol_type: "callback_title_focus",
      }))
    })
    window._titleDom.addEventListener('blur', function (e) {
      window.location.href = 'hbEditor://' + encodeURI(JSON.stringify({
        protocol_type: "callback_title_blur",
      }))
    })
  }
  window.showLog = function (log) {
    window.location.href = 'hbEditor://' + encodeURI(JSON.stringify({
      protocol_type: "callback_log",
      log: log
    }))
  }

  window.$ = $
  window.hb_editor = new Editor()
})()