/* eslint-disable prettier/prettier */

/**
 * 自定义字段：
 * const customField = [{ label: "xxx", value: "xxx_xxx" }];
 */
class FrontendMonitor {
  constructor (options = {customField: []}) {
    this.config = {
      dsn: '', // 数据上报地址
      appId: '', // 应用标识
      enablePerformance: false, // 是否开启性能监控
      enableAction: true, // 是否开启用户跟踪
      sampleRate: 1, // 采样率 (0-1)
      maxActions: 1000, // 最大记录用户行为条数
      ...options
    };

    // 用户行为跟踪
    this.actions = [];

    this.errorCache = new Map(); // 错误去重缓存
    this.init();
  }

  // 初始化监控
  init() {
    this.setupErrorListener();
    this.setupUnhandledRejection();
    this.setupResourceError();

    if (this.config.enablePerformance) {
      this.setupPerformanceObserver();
    }

    if (this.config.enableAction) {
      this.setupUserAction()
    }
  }

  // 全局错误监听
  setupErrorListener () {
    window.addEventListener('error', (event) => {
      const target = event.target;
      const { message, filename, lineno, colno, error } = event;
      if (!(target instanceof HTMLElement)) {
        this.captureException({
          type: 'js_error',
          message,
          filename,
          lineno,
          colno,
          stack: error?.stack
        });
      }
    }, true);
  }

  // Promise未处理拒绝
  setupUnhandledRejection () {
    window.addEventListener('unhandledrejection', (event) => {
      const reason = event.reason;
      this.captureException({
        type: 'unhandled_rejection',
        message: reason?.message || String(reason),
        stack: reason?.stack
      });
    });
  }

  // 资源加载失败
  setupResourceError () {
    window.addEventListener('error', (event) => {
      const target = event.target;
      if (target instanceof HTMLElement && ['IMG', 'SCRIPT', 'LINK'].includes(target.tagName)) {
        this.captureException({
          type: 'resource_error',
          tagName: target.tagName,
          src: target.src || target.href
        });
      }
    }, true);
  }

  // 性能监控
  setupPerformanceObserver () {
    const observer = new PerformanceObserver((list) => {
      const entries = list.getEntries();
      entries.forEach(entry => {
        this.report({
          type: 'performance',
          name: entry.name,
          entryType: entry.entryType,
          startTime: entry.startTime,
          duration: entry.duration
          // 可添加更多性能指标
        });
      });
    });
    observer.observe({ entryTypes: ['navigation', 'resource', 'paint'] });
  }

  // 错误捕获核心方法
  captureException (errorData) {
    if (Math.random() > this.config.sampleRate) return;

    const errorFingerprint = this.generateErrorFingerprint(errorData);
    if (this.errorCache.has(errorFingerprint)) return;

    this.errorCache.set(errorFingerprint, true);

    const reportData = {
      ...errorData,
      appId: this.config.appId,
      timestamp: Date.now(),
      url: window.location.href,
      userAgent: navigator.userAgent,
      // 用户行为跟踪
      actions: this.actions
      // 可添加更多上下文信息
    };
    // 上报时携带版本号和 SourceMap 地址
    reportData.sourceMap = {
      // version: '1.0.0',
      // mapUrl: 'https://your-sourcemap.com/main.js.map'
    }

    this.actions.push(Object.assign(errorData, {actionType: 'jsError', timestamp: Date.now() }));
    reportData.actions = this.actions;

    this.report(reportData);
  }

  // 生成错误唯一标识（简单示例）
  generateErrorFingerprint (error) {
    return `${error.type}-${error.message}-${error.filename}-${error.lineno}-${error.colno}`;
  }

  // 记录用户行为轨迹
  setupUserAction () {
    this.trackClicks()
    this.trackConsole()
    this.trackNavigation()
    this.trackAJAX()
  }
  // ==================== 通用方法 ====================
  saveRecord (data) {
    this.actions.push(data);
    if (this.actions.length >= this.config.maxActions) {
      this.actions
      this.report();
    }
  }
  // ==================== 点击事件监控 ====================
  trackClicks () {
    document.addEventListener('click', (e) => {
      const target = e.target;
      const data = {
        type: 'click',
        actionType: 'click',
        timestamp: Date.now(),
        selector: this.getElementSelector(target),
        position: { x: e.clientX, y: e.clientY },
        text: target.textContent?.trim().slice(0, 50), // 截取前50字符
        page: window.location.href
      };
      this.saveRecord(data);
    }, true); // 使用捕获阶段确保监听到所有点击
  }
  // 获取元素唯一选择器
  getElementSelector (el) {
    if (!el || el === document) return '';
    const path = [];
    while (el && el.nodeType === Node.ELEMENT_NODE) {
      let selector = el.nodeName.toLowerCase();
      if (el.id) {
        selector += `#${el.id}`;
        path.unshift(selector);
        break;
      } else {
        let sibling = el; let nth = 1;
        while (sibling.previousElementSibling) {
          sibling = sibling.previousElementSibling;
          nth++;
        }
        if (nth !== 1) selector += `:nth-of-type(${nth})`;
      }
      path.unshift(selector);
      el = el.parentNode;
    }
    return path.join(' > ');
  }
  // ==================== 控制台监控 ====================
  trackConsole () {
    if (!this.config.trackConsole) return;

    const consoleTypes = ['error', 'warn', 'log'];
    consoleTypes.forEach(type => {
      const original = console[type];
      console[type] = (...args) => {
        // 记录控制台输出
        this.saveRecord({
          type: 'console',
          actionType: 'console',
          level: type,
          messages: args.map(arg =>
            typeof arg === 'object' ? JSON.stringify(arg) : arg
          ),
          timestamp: Date.now()
        });
        // 调用原始方法
        original.apply(console, args);
      };
    });
  }
  // ==================== 页面跳转监控 ====================
  trackNavigation () {
    const _this = this;
    // 监听SPA路由变化
    ['pushState', 'replaceState'].forEach(method => {
      const original = history[method];
      history[method] = function () {
        const urlBefore = window.location.href;
        original.apply(this, arguments);
        const urlAfter = window.location.href;
        if (urlBefore !== urlAfter) {
          _this._logNavigation('spa', urlAfter);
        }
      };
    });

    // 监听hashchange
    window.addEventListener('hashchange', () => {
      this._logNavigation('hash', window.location.href);
    });
  }

  _logNavigation (type, url) {
    this.saveRecord({
      type: 'navigation',
      actionType: 'navigation',
      navType: type,
      from: document.referrer,
      to: url,
      timestamp: Date.now()
    });
  }

  // ==================== AJAX监控 ====================
  trackAJAX () {
    // 拦截原生XMLHttpRequest
    const originalXHR = window.XMLHttpRequest;
    const _this = this;

    window.XMLHttpRequest = class extends originalXHR {
      constructor () {
        super();
        this._startTime = Date.now();
        this._method = '';
        this._url = '';
      }

      open (method, url) {
        this._method = method;
        this._url = url;
        super.open(method, url);
      }
      send (body) {
        this.addEventListener('readystatechange', function () {
          if (this.readyState === 4) {
            _this.saveRecord({
              type: 'ajax',
              actionType: 'http',
              method: this._method,
              url: this._url,
              status: this.status,
              duration: Date.now() - this._startTime,
              // requestSize: body ? body.toString().length : 0,
              // responseSize: this.responseText?.length || 0,
              timestamp: Date.now()
            });
          }
        });
        super.send(body);
      }
    };
    // 拦截Fetch API
    const originalFetch = window.fetch;
    window.fetch = async (...args) => {
      const start = Date.now();
      const [input, init] = args;
      const url = typeof input === 'string' ? input : input.url;
      const method = (init?.method || 'GET').toUpperCase();

      try {
        const response = await originalFetch(...args);
        _this.saveRecord({
          type: 'fetch',
          actionType: 'http',
          method,
          url,
          status: response.status,
          duration: Date.now() - start,
          timestamp: Date.now()
        });
        return response;
      } catch (error) {
        _this.saveRecord({
          type: 'fetch',
          method,
          url,
          status: 0,
          duration: Date.now() - start,
          error: error.message,
          timestamp: Date.now()
        });
        throw error;
      }
    };
  }



  // 数据上报
  report(data) {
    // 自定义字段
    data.customField = this.config.customField
  
    // 必填字段
    const requestData = {
      appId: this.config.appId,
      version: this.config.version,
      errorDetail: data,
      errorType: data.type,
      occurrenceTime: data.timestamp,
      status: 'open'
    }

    const blob = new Blob([JSON.stringify({ requestObject: requestData })], { type: 'application/json' });

    // 优先使用 sendBeacon 防止页面卸载丢失数据
    if (navigator.sendBeacon) {
      navigator.sendBeacon(this.config.dsn, blob);
    } else {
      fetch(this.config.dsn, {
        method: 'POST',
        body: blob,
        keepalive: true // 确保在页面卸载时发送
      }).catch(err => {
        console.error('Report failed:', err);
      });
    }
  }

  // 手动上报自定义错误
  trackError (error, extra = {}) {
    this.captureException({
      type: 'custom_error',
      message: error.message,
      stack: error.stack,
      ...extra
    });
  }
  // 销毁监听（可选）
  destroy () {
    // 移除所有事件监听器...
  }

  // 添加用户信息
  setUser (user) {
    this.user = user;
  }

  // 根据错误特征自动分类
  classifyError (error) {
    if (/Failed to load resource/.test(error.message)) {
      return 'NETWORK_ERROR';
    }
    // 其他分类规则...
  }

  // 监控 Core Web Vitals
  // const cls = new PerformanceObserver((list) => {
  //   for (const entry of list.getEntries()) {
  //     this.report({ type: 'CLS', value: entry.value });
  //   }
  // });
  // cls.observe({ type: 'layout-shift', buffered: true });
}

export default FrontendMonitor

