// This file is always executed before the App's index.js. It sets up most of
// Elves's functionality. It initializes some global properties, such as window
// dimensions and the userAgent, provides global functions such as setInterval()
// and setTimeout() and the console object, emulates a tiny bit of DOM
// functionality and sets up handlers for Touch events and others.


// Make 'window' the global scope
self = window = this;
window.top = window.parent = window;

(function(window) {

// The 'elv' object provides some basic info and utility functions
var elv = window.elves = new GlobalUtils();

// Set up the screen properties and useragent
window.devicePixelRatio = elv.devicePixelRatio;
window.innerWidth = elv.screenWidth;
window.innerHeight = elv.screenHeight;

Object.defineProperty(window, 'orientation', {
	get: function() {return elv.orientation; }
});

window.screen = {
	availWidth: window.innerWidth,
	availHeight: window.innerHeight
};

var geolocation = null;
window.navigator = {
	language: elv.language,
	userAgent: elv.userAgent,
	appVersion: elv.appVersion,
	platform: elv.platform,
	get onLine() { return elv.onLine; }, // re-evaluate on each get
	get geolocation(){ // Lazily create geolocation instance
		geolocation = geolocation || new Geolocation();
		return geolocation;
	}
};

// Create the default screen canvas
window.canvas = new Canvas();
window.canvas.type = 'canvas';

// The console object
window.console = {

	// We try to be sensible of how much data we want to log here. Only the first
	// level of an Object or Array is parsed. All subsequent levels are ommited.
	// Arrays are shortened to the first 32 entries.
	// To log an Object and traverse all levels, use console.logJSON()
	_arrayMaxLength: 32,
	
	_toString: function(obj, deep) {
		if( deep ) {
			return JSON.stringify(obj);
		}
		else if( obj instanceof Array || ArrayBuffer.isView(obj) ) {
			var s = '',
				length = Math.min(obj.length, window.console._arrayMaxLength),
				omitted = obj.length - length;
			for( var i = 0; i < length; i++ ) {
				s += (i === 0 ? '' : ', ') + window.console._toStringFlat(obj[i]);
			}
			return '[' + s + (omitted ? ', ...'+omitted+' more]' : ']');
		}
		else {
			var s = '',
				first = true;
			for( var i in obj ) {
				s += (first ? '' : ', ') + i + ': ' + window.console._toStringFlat(obj[i]);
				first = false;
			}
			return '{'+s+'}';
		}
	},
	
	_toStringFlat: function(obj) {
		if( typeof(obj) === 'function' ) {
			return '[Function]';
		}
		else if( obj instanceof Array || ArrayBuffer.isView(obj) ) {
			return '[Array '+obj.length+']';
		}
		else {
			return obj;
		}
	},
	
	_log: function(level, args, deep) {
		var s = level + ':';
		for (var i = 0; i < args.length; i++) {
			var arg = args[i];
			s += ' ' + (!arg || typeof(arg) !== 'object'
				? arg
				: window.console._toString(arg, deep));
		}
		elv.log( s );
	},
	
	assert: function() {
		var args = Array.prototype.slice.call(arguments);
		var assertion = args.shift();
		if( !assertion ) {
			elv.log( 'Assertion failed: ' + args.join(', ') );
		}
	}
};
window.console.debug = function () { window.console._log('DEBUG', arguments); };
window.console.info = function () { window.console._log('INFO', arguments); };
window.console.warn = function () { window.console._log('WARN', arguments); };
window.console.error = function () { window.console._log('ERROR', arguments); };
window.console.log = function () { window.console._log('LOG', arguments); };
window.console.logJSON = function () { window.console._log('JSON', arguments, true); };

var consoleTimers = {};
console.time = function(name) {
	consoleTimers[name] = elv.performanceNow();
};

console.timeEnd = function(name) {
	var timeStart = consoleTimers[name];
	if( !timeStart ) {
		return;
	}

	var timeElapsed = elv.performanceNow() - timeStart;
	console.log(name + ": " + timeElapsed + "ms");
	delete consoleTimers[name];
};


// CommonJS style require()
var loadedModules = {};
window.require = function( name ) {
	var id = name.replace(/\.js$/,'');
	if( !loadedModules[id] ) {
		var exports = {};
		var module = { id: id, uri: id + '.js', exports: exports };
		window.elves.requireModule( id, module, exports );
		// Some modules override module.exports, so use the module.exports
		// reference only after loading the module
		loadedModules[id] = module.exports;
	}
	
	return loadedModules[id];
};

// Timers
window.performance = {now: function() {return elv.performanceNow();} };
window.setTimeout = function(cb, t){ return elv.setTimeout(cb, t||0); };
window.setInterval = function(cb, t){ return elv.setInterval(cb, t||0); };
window.clearTimeout = function(id){ return elv.clearTimeout(id); };
window.clearInterval = function(id){ return elv.clearInterval(id); };
window.requestAnimationFrame = function(cb, element){
	return elv.setTimeout(function(){ cb(elv.performanceNow()); }, 16);
};
window.cancelAnimationFrame = function (id) {
	return elv.clearTimeout(id);
};


// The native Image, Audio, HttpRequest and LocalStorage class mimic the real elements
window.Image = Image;
window.Audio = Audio;
window.Video = Video;
window.XMLHttpRequest = FAHttpRequest;
window.localStorage = new LocalStorage();
window.WebSocket = WebSocket;

window.gameControl = new GameControl();
window.pagView = new PagView();
window.monitor = new Monitor();

window.Event = function (type) {
	this.type = type;
	this.cancelBubble = false;
	this.cancelable = false;
	this.target = null;
	this.timestamp = elv.performanceNow();
	
	this.initEvent = function (type, bubbles, cancelable) {
		this.type = type;
		this.cancelBubble = bubbles;
		this.cancelable = cancelable;
		this.timestamp = elv.performanceNow();
	};

	this.preventDefault = function () {};
	this.stopPropagation = function () {};
};

window.location = { href: 'index.js' };
window.location.reload = function() {
	elves.load('index.js');
}

window.open = function(url) { elv.openURL(url); };


// Set up a "fake" HTMLElement
HTMLElement = function( tagName ){
	this.tagName = tagName.toUpperCase();
	this.children = [];
	this.style = {};
};

HTMLElement.prototype.appendChild = function( element ) {
	this.children.push( element );

	elves.log("HTMLElement.prototype.appendChild element:" + element);

	// If the child is a script element, begin to load it or execute it
	if( element.tagName && element.tagName.toLowerCase() == 'script' ) {
		if ( element.src ) {
			elv.setTimeout( function(){
			    elv.log("HTMLElement.prototype.appendChild inner element.src:" + element.src);
				elv.include( element.src );
				 elv.log("HTMLElement.prototype.appendChild inner element.src:" + element.src + "  end");
				if( element.onload ) {
					element.onload({
						type: 'load',
						currentTarget: element
					});
				}
			}, 1);
		} else if( element.text ) {
			window.eval( element.text );
		}
	}
};

HTMLElement.prototype.insertBefore = function( newElement, existingElement ) {
	// Just append; we don't care about order here
	this.appendChild( newElement );
};

HTMLElement.prototype.removeChild = function( node ) {
	for( var i = this.children.length; i--; ) {
		if( this.children[i] === node ) {
			this.children.splice(i, 1);
		}
	}
};

HTMLElement.prototype.getBoundingClientRect = function() {
	return {top: 0, left: 0, width: window.innerWidth, height: window.innerHeight};
};

HTMLElement.prototype.setAttribute = function(attr, value){
	this[attr] = value;
};

HTMLElement.prototype.getAttribute = function(attr){
	return this[attr];
};

HTMLElement.prototype.addEventListener = function(event, method){
	if (event === 'load') {
		this.onload = method;
	}
};

HTMLElement.prototype.removeEventListener = function(event, method){
	if (event === 'load') {
		this.onload = undefined;
	}
};

// The document object
window.document = {
	readyState: 'complete',
	documentElement: window,
	location: window.location,
	visibilityState: 'visible',
	hidden: false,
	style: {},
	
	head: new HTMLElement( 'head' ),
	body: new HTMLElement( 'body' ),
	
	events: {},
	
	createElement: function( name ) {
		if( name === 'canvas' ) {
			var canvas = new Canvas();
			canvas.type = 'canvas';
			return canvas;
		}
		else if( name == 'audio' ) {
			return new Audio();
		}
		else if( name == 'video' ) {
			return new Video();
		}
		else if( name === 'img' ) {
			return new window.Image();
		}
		else if (name === 'input' || name === 'textarea') {
			return new KeyInput();
 		}
		return new HTMLElement( name );
	},
	
	getElementById: function( id ){
		if( id === 'canvas' ) {
			return window.canvas;
		}
		return null;
	},
	
	getElementsByTagName: function( tagName ) {
		var elements = [], children, i;

		tagName = tagName.toLowerCase();

		if( tagName === 'head' ) {
			elements.push(document.head);
		}
		else if( tagName === 'body' ) {
			elements.push(document.body);
		}
		else {
			children = document.body.children;
			for (i = 0; i < children.length; i++) {
				if (children[i].tagName.toLowerCase() === tagName) {
					elements.push(children[i]);
				}
			}
			children = undefined;
		}
		return elements;
	},

	createEvent: function (type) { 
		return new window.Event(type); 
	},
	
	addEventListener: function( type, callback, useCapture ){
		if( type == 'DOMContentLoaded' ) {
			elv.setTimeout( callback, 1 );
			return;
		}
		if( !this.events[type] ) {
			this.events[type] = [];
			
			// call the event initializer, if this is the first time we
			// bind to this event.
			if( typeof(this._eventInitializers[type]) == 'function' ) {
				this._eventInitializers[type]();
			}
		}
		this.events[type].push( callback );
	},
	
	removeEventListener: function( type, callback ) {
		var listeners = this.events[ type ];
		if( !listeners ) { return; }
		
		for( var i = listeners.length; i--; ) {
			if( listeners[i] === callback ) {
				listeners.splice(i, 1);
			}
		}
	},
	
	_eventInitializers: {},
	dispatchEvent: function( event ) {
		var listeners = this.events[ event.type ];
		if( !listeners ) { return; }
		
		for( var i = 0; i < listeners.length; i++ ) {
			listeners[i]( event );
		}
	}
};

window.canvas.addEventListener = window.addEventListener = function( type, callback ) {
	window.document.addEventListener(type,callback);
};
window.canvas.removeEventListener = window.removeEventListener = function( type, callback ) {
	window.document.removeEventListener(type,callback);
};
window.canvas.getBoundingClientRect = function() {
	return {
		top: this.offsetTop, left: this.offsetLeft,
		width: this.offsetWidth, height: this.offsetHeight
	};
};

var eventInit = document._eventInitializers;



// Touch events

// Set touch event properties for feature detection
window.ontouchstart = window.ontouchend = window.ontouchmove = null;
document.ontouchstart = document.ontouchend = document.ontouchmove = null;

// Setting up the 'event' object for touch events in native code is quite
// a bit of work, so instead we do it here in JavaScript and have the native
// touch class just call a simple callback.
var touchInput = null;
var touchEvent = {
	type: 'touchstart',
	target: window.canvas,
	currentTarget: window.canvas,
	touches: null,
	targetTouches: null,
	changedTouches: null,
	timestamp: 0,
	preventDefault: function(){},
	stopPropagation: function(){}
};

var dispatchTouchEvent = function( type, all, changed, timestamp ) {
	touchEvent.touches = all;
	touchEvent.targetTouches = all;
	touchEvent.changedTouches = changed;
	touchEvent.type = type;
	touchEvent.timestamp = timestamp;
	
	document.dispatchEvent( touchEvent );
};
eventInit.touchstart = eventInit.touchend = eventInit.touchmove = function() {
	if( touchInput ) { return; }

	touchInput = new TouchInput(window.canvas);
	touchInput.ontouchstart = dispatchTouchEvent.bind(window, 'touchstart');
	touchInput.ontouchend = dispatchTouchEvent.bind(window, 'touchend');
	touchInput.ontouchmove = dispatchTouchEvent.bind(window, 'touchmove');
};



// DeviceMotion and DeviceOrientation events

var deviceMotion = null;
var deviceMotionEvent = {
	type: 'devicemotion',
	target: window.canvas,
	currentTarget: window.canvas,
	interval: 16,
	acceleration: {x: 0, y: 0, z: 0},
	accelerationIncludingGravity: {x: 0, y: 0, z: 0},
	rotationRate: {alpha: 0, beta: 0, gamma: 0},
	timestamp: 0,
	preventDefault: function(){},
	stopPropagation: function(){}
};

var deviceOrientationEvent = {
	type: 'deviceorientation',
	target: window.canvas,
	currentTarget: window.canvas,
	alpha: null,
	beta: null,
	gamma: null,
	absolute: true,
	timestamp: 0,
	preventDefault: function(){},
	stopPropagation: function(){}
};

eventInit.deviceorientation = eventInit.devicemotion = function() {
	if( deviceMotion ) { return; }
	
	deviceMotion = new DeviceMotion();
	deviceMotionEvent.interval = deviceMotion.interval;
	
	// Callback for Devices that have a Gyro
	deviceMotion.ondevicemotion = function( agx, agy, agz, ax, ay, az, rx, ry, rz, ox, oy, oz, timestamp ) {
		deviceMotionEvent.accelerationIncludingGravity.x = agx;
		deviceMotionEvent.accelerationIncludingGravity.y = agy;
		deviceMotionEvent.accelerationIncludingGravity.z = agz;
	
		deviceMotionEvent.acceleration.x = ax;
		deviceMotionEvent.acceleration.y = ay;
		deviceMotionEvent.acceleration.z = az;

		deviceMotionEvent.rotationRate.alpha = rx;
		deviceMotionEvent.rotationRate.beta = ry;
		deviceMotionEvent.rotationRate.gamma = rz;
 
		deviceMotionEvent.timestamp = timestamp;

		document.dispatchEvent( deviceMotionEvent );


		deviceOrientationEvent.alpha = ox;
		deviceOrientationEvent.beta = oy;
		deviceOrientationEvent.gamma = oz;
 
		deviceOrientationEvent.timestamp = timestamp;

		document.dispatchEvent( deviceOrientationEvent );
	};
	
	// Callback for Devices that only have an accelerometer
	deviceMotion.onacceleration = function( agx, agy, agz, timestamp ) {
		deviceMotionEvent.accelerationIncludingGravity.x = agx;
		deviceMotionEvent.accelerationIncludingGravity.y = agy;
		deviceMotionEvent.accelerationIncludingGravity.z = agz;
	
		deviceMotionEvent.acceleration = null;
		deviceMotionEvent.rotationRate = null;
 
		deviceMotionEvent.timestamp = timestamp;
	
		document.dispatchEvent( deviceMotionEvent );
	};
};



// Window events (resize/pagehide/pageshow)

var windowEvents = null;

var lifecycleEvent = {
	type: 'pagehide',
	target: window.document,
	currentTarget: window.document,
	timestamp: 0,
	preventDefault: function(){},
	stopPropagation: function(){}
};

var resizeEvent = {
	type: 'resize',
	target: window,
	currentTarget: window,
	timestamp: 0,
	preventDefault: function(){},
	stopPropagation: function(){}
};

var visibilityEvent = {
	type: 'visibilitychange',
	target: window.document,
	currentTarget: window.document,
	timestamp: 0,
	preventDefault: function(){},
	stopPropagation: function(){}
};

eventInit.visibilitychange = eventInit.pagehide = eventInit.pageshow = eventInit.resize = function() {
	if( windowEvents ) { return; }
	
	windowEvents = new WindowEvents();
	
	windowEvents.onpagehide = function() {
		document.hidden = true;
		document.visibilityState = 'hidden';
		visibilityEvent.timestamp = elv.performanceNow();
		document.dispatchEvent( visibilityEvent );
	
		lifecycleEvent.type = 'pagehide';
		document.dispatchEvent( lifecycleEvent );
	};
	
	windowEvents.onpageshow = function() {
		document.hidden = false;
		document.visibilityState = 'visible';
		visibilityEvent.timestamp = elv.performanceNow();
		document.dispatchEvent( visibilityEvent );
	
		lifecycleEvent.type = 'pageshow';
		document.dispatchEvent( lifecycleEvent );
	};

	windowEvents.onresize = function() {
		window.innerWidth = elv.screenWidth;
		window.innerHeight = elv.screenHeight;
		resizeEvent.timestamp = elv.performanceNow();
		document.dispatchEvent(resizeEvent);
	};
};


// Gamepad API

//var gamepadProvider = null;
//var initGamepadProvider = function() {
//	if( gamepadProvider ) { return; }
//	gamepadProvider = new GamepadProvider();
//
//	gamepadProvider.ongamepadconnected = gamepadProvider.ongamepaddisconnected = function(ev) {
//		document.dispatchEvent(ev);
//	};
//};
//
//navigator.getGamepads = function() {
//	initGamepadProvider();
//	return gamepadProvider.getGamepads();
//};
//
//eventInit.gamepadconnected = eventInit.gamepaddisconnected = initGamepadProvider;


})(this);
