#ifndef _PBR_COMMON_FXH_
#define _PBR_COMMON_FXH_

#ifndef PI
#   define  PI 3.141592653589793
#endif

//#ifdef GLTF_PBR_USE_TANGENT
//#   define  GLTF_PBR_USE_TANGENT    1
//#endif
// Lambertian diffuse
// see https://seblagarde.wordpress.com/2012/01/08/pi-or-not-to-pi-in-game-lighting-equation/
float3 LambertianDiffuse(float3 DiffuseColor)
{
    return DiffuseColor / PI;
}
static const float3x3 XYZ_TO_RGB = float3x3(2.3706743, -0.5138850, 0.0052982, 
-0.9000405, 1.4253036, -0.0146949, 
-0.4706338, 0.0885814, 1.0093968);
// The following equation models the Fresnel reflectance term of the spec equation (aka F())
// Implementation of fresnel from "An Inexpensive BRDF Model for Physically based Rendering" by Christophe Schlick
// (https://www.cs.virginia.edu/~jdl/bib/appearance/analytic%20models/schlick94b.pdf), Equation 15
float3 SchlickReflection(float VdotH, float3 Reflectance0, float3 Reflectance90)
{
    return Reflectance0 + (Reflectance90 - Reflectance0) * pow(clamp(1.0 - VdotH, 0.0, 1.0), 5.0);
}

// Visibility = G(v,l,a) / (4 * (n,v) * (n,l))
// see https://google.github.io/filament/Filament.md.html#materialsystem/specularbrdf/geometricshadowing(specularg)
float SmithGGXVisibilityCorrelated(float NdotL, float NdotV, float AlphaRoughness)
{
    float a2 = AlphaRoughness * AlphaRoughness;

    float GGXV = NdotL * sqrt(max(NdotV * NdotV * (1.0 - a2) + a2, 1e-7));
    float GGXL = NdotV * sqrt(max(NdotL * NdotL * (1.0 - a2) + a2, 1e-7));

    return 0.5 / (GGXV + GGXL);
}

// The following equation(s) model the distribution of microfacet normals across the area being drawn (aka D())
// Implementation from "Average Irregularity Representation of a Roughened Surface for Ray Reflection" by T. S. Trowbridge, and K. P. Reitz
// Follows the distribution function recommended in the SIGGRAPH 2013 course notes from EPIC Games [1], Equation 3.
float NormalDistribution_GGX(float NdotH, float AlphaRoughness)
{
    float a2 = AlphaRoughness * AlphaRoughness;
    float f = NdotH * NdotH * (a2 - 1.0)  + 1.0;
    return a2 / (PI * f * f);
}


struct AngularInfo
{
    float NdotL;   // cos angle between normal and light direction
    float NdotV;   // cos angle between normal and view direction
    float NdotH;   // cos angle between normal and half vector
    float LdotH;   // cos angle between light direction and half vector
    float VdotH;   // cos angle between view direction and half vector
    float3 H;
};

AngularInfo GetAngularInfo(float3 PointToLight, float3 Normal, float3 View)
{
    float3 n = normalize(Normal);       // Outward direction of surface point
    float3 v = normalize(View);         // Direction from surface point to camera
    float3 l = normalize(PointToLight); // Direction from surface point to light
    float3 h = normalize(l + v);        // Direction of the vector between l and v

    AngularInfo info;
    info.H=h;
    info.NdotL = clamp(dot(n, l), 0.0, 1.0);
    info.NdotV = clamp(dot(n, v), 0.0, 1.0);
    info.NdotH = clamp(dot(n, h), 0.0, 1.0);
    info.LdotH = clamp(dot(l, h), 0.0, 1.0);
    info.VdotH = clamp(dot(v, h), 0.0, 1.0);

    return info;
}

struct SurfaceReflectanceInfo
{
    float  PerceptualRoughness;
    float3 Reflectance0;
    float3 Reflectance90;
    float3 DiffuseColor;
};

struct MaterialInfo
{
    float  Anisotropic;
    float  ClearCoat;
    float  ClearCoatRounghness;
    float  IridescenceMask;
    float  FilmThickness ;
    float  BaseIor;
    float  KExtinction;
    float SheenRoughness;
    float3 SheenColor;
    float TransmissionFactor;
    float3 BaseColor;
    float4x4 worldmatrix;
    float thickness;
    float3 attenuationColor;
    float attenuationDistance;
    float sssDistortion;
    float sssPower;
    float sssScale;
    float sssAttenuation;
    
    float noise;
    float FurLength;
    float GravityStrength;
    float Thickness;
    float Blur;
    float Glossiness1;
    float Glossiness2;
    float SpecShift1;
    float SpecShift2;
    float SpecSmooth1;
    float SpecSmooth2;
    float4 SpecColor1;
    float4 SpecColor2;

};
struct FurInfo
{
    float Noise;
    float FurLength;
    float GravityStrength;
    float Thickness;
    float Blur;
    float Glossiness1;
    float Glossiness2;
    float SpecShift1;
    float SpecShift2;
    float SpecSmooth1;
    float SpecSmooth2;
    float4 SpecColor1;
    float4 SpecColor2;
};



void InitFurMat(out FurInfo furinfo,
in float noise,
in float FurLength,
in float GravityStrength,
in float Thickness,
in float Blur,
in float Glossiness1,
in float Glossiness2,
in float SpecShift1,
in float SpecShift2,
in float SpecSmooth1,
in float SpecSmooth2,
in float4 SpecColor1,
in float4 SpecColor2)
{
    
    furinfo.Noise = noise;
    furinfo.FurLength = FurLength;
    furinfo.GravityStrength = GravityStrength;
    furinfo.Thickness = Thickness;
    furinfo.Blur = Blur;
    furinfo.Glossiness1 = Glossiness1;
    furinfo.Glossiness2 = Glossiness2;
    furinfo.SpecShift1 = SpecShift1;
    furinfo.SpecShift2 = SpecShift2;
    furinfo.SpecSmooth1 = SpecSmooth1;
    furinfo.SpecSmooth2 = SpecSmooth2;
    furinfo.SpecColor1 = SpecColor1;
    furinfo.SpecColor2 = SpecColor1;
    
    
}




void InitalMateriInfo(out MaterialInfo matinfo,
in float ani,

in float clearcoat,
in float clearcoatrougness,

in float iridescenceMask,
in float filmThickness,
in float baseIor,
in float kExtinction,

in float SheenRoughness,
in float3 SheenColor,

in float TransmissionFactor,
in float3 BaseColor,
in float4x4 worldmatrix,
in float tickness,
in float3 attenuationColor,
in float attenuationDistance,
in float sssDistortion,
in float sssPower,
in float sssScale,
in float sssAttenuation,
in FurInfo furInfo
)
{
    
#ifdef PBR_Default 
    matinfo.Anisotropic = 0.0;
    matinfo.ClearCoat = 0.0;
    matinfo.ClearCoatRounghness = 0.0;
    matinfo.IridescenceMask = 0.0;
    matinfo.FilmThickness = 0.0;
    matinfo.BaseIor = 0.0;
    matinfo.KExtinction = 0.0;
    matinfo.SheenRoughness=0.0;
    matinfo.SheenColor=float3(0.0,0.0,0.0);
    matinfo.TransmissionFactor=0.0;
    matinfo.BaseColor=float3(0.0,0.0,0.0);
    matinfo.worldmatrix=worldmatrix;
    matinfo.thickness=0.0;
    matinfo.attenuationColor=float3(0.0,0.0,0.0);
    matinfo.attenuationDistance=0.0;
    matinfo.sssDistortion=0.0;
    matinfo.sssPower=0.0;
    matinfo.sssScale=0.0;
    matinfo.sssAttenuation=0.0;
    matinfo.FurLength=0.0;
    matinfo.GravityStrength=0.0;
    matinfo.Thickness=0.0;
    matinfo.Blur=0.0;
    matinfo.Glossiness1=0.0;
    matinfo.Glossiness2=0.0;
    matinfo.SpecShift1=0.0;
    matinfo.SpecShift2=0.0;
    matinfo.SpecSmooth1=0.0;
    matinfo.SpecSmooth2=0.0;
    matinfo.SpecColor1=float4(0.0,0.0,0.0,0.0);
    matinfo.SpecColor2=float4(0.0,0.0,0.0,0.0);
    matinfo.noise=0.0;
#endif

#ifdef GLTF_PBR_USE_TANGENT 
    matinfo.Anisotropic = ani;
    matinfo.ClearCoat = 0.0;
    matinfo.ClearCoatRounghness = 0.0;
    matinfo.IridescenceMask = 0.0;
    matinfo.FilmThickness = 0.0;
    matinfo.BaseIor = 0.0;
    matinfo.KExtinction = 0.0;
     matinfo.SheenRoughness=0.0;
    matinfo.SheenColor=float3(0.0,0.0,0.0);
     matinfo.TransmissionFactor=0.0;
    matinfo.BaseColor=float3(0.0,0.0,0.0);
    matinfo.worldmatrix=worldmatrix;
    matinfo.thickness=0.0;
    matinfo.attenuationColor=float3(0.0,0.0,0.0);
    matinfo.attenuationDistance=0.0;
    matinfo.sssDistortion=0.0;
    matinfo.sssPower=0.0;
    matinfo.sssScale=0.0;
    matinfo.sssAttenuation=0.0;
    matinfo.FurLength=0.0;
    matinfo.GravityStrength=0.0;
    matinfo.Thickness=0.0;
    matinfo.Blur=0.0;
    matinfo.Glossiness1=0.0;
    matinfo.Glossiness2=0.0;
    matinfo.SpecShift1=0.0;
    matinfo.SpecShift2=0.0;
    matinfo.SpecSmooth1=0.0;
    matinfo.SpecSmooth2=0.0;
    matinfo.SpecColor1=float4(0.0,0.0,0.0,0.0);
    matinfo.SpecColor2=float4(0.0,0.0,0.0,0.0);
     matinfo.noise=0.0;
#endif
#ifdef GLTF_PBR_USE_CLEARCOAT 
    matinfo.Anisotropic = 0.0;
    matinfo.ClearCoat = clearcoat;
    matinfo.ClearCoatRounghness = clearcoatrougness;
    matinfo.IridescenceMask = 0.0;
    matinfo.FilmThickness = 0.0;
    matinfo.BaseIor = 0.0;
    matinfo.KExtinction = 0.0;
    matinfo.SheenRoughness=0.0;
    matinfo.SheenColor=float3(0.0,0.0,0.0);
    matinfo.TransmissionFactor=0.0;
    matinfo.BaseColor=float3(0.0,0.0,0.0);
    matinfo.worldmatrix=worldmatrix;
    matinfo.thickness=0.0;
    matinfo.attenuationColor=float3(0.0,0.0,0.0);
    matinfo.attenuationDistance=0.0;
    matinfo.sssDistortion=0.0;
    matinfo.sssPower=0.0;
    matinfo.sssScale=0.0;
    matinfo.sssAttenuation=0.0;
    matinfo.FurLength=0.0;
    matinfo.GravityStrength=0.0;
    matinfo.Thickness=0.0;
    matinfo.Blur=0.0;
    matinfo.Glossiness1=0.0;
    matinfo.Glossiness2=0.0;
    matinfo.SpecShift1=0.0;
    matinfo.SpecShift2=0.0;
    matinfo.SpecSmooth1=0.0;
    matinfo.SpecSmooth2=0.0;
    matinfo.SpecColor1=float4(0.0,0.0,0.0,0.0);
    matinfo.SpecColor2=float4(0.0,0.0,0.0,0.0);
    matinfo.noise=0.0;
#endif
#ifdef GLTF_PBR_USE_IRR 
    matinfo.Anisotropic = 0.0;
    matinfo.ClearCoat = 0.0;
    matinfo.ClearCoatRounghness = 0.0;
    matinfo.IridescenceMask =clamp(1.0 - pow(1.0 - iridescenceMask, 3.0), 0.0, 1.0);
    matinfo.FilmThickness =clamp(filmThickness, 0.0, 1.0);
    matinfo.BaseIor = baseIor;
    matinfo.KExtinction =kExtinction;
    matinfo.SheenRoughness=0.0;
    matinfo.SheenColor=float3(0.0,0.0,0.0);
    matinfo.TransmissionFactor=0.0;
    matinfo.BaseColor=float3(0.0,0.0,0.0);
    matinfo.worldmatrix=worldmatrix;
    matinfo.thickness=0.0;
    matinfo.attenuationColor=float3(0.0,0.0,0.0);
    matinfo.attenuationDistance=0.0;
    matinfo.sssDistortion=0.0;
    matinfo.sssPower=0.0;
    matinfo.sssScale=0.0;
    matinfo.sssAttenuation=0.0;
    matinfo.FurLength=0.0;
    matinfo.GravityStrength=0.0;
    matinfo.Thickness=0.0;
    matinfo.Blur=0.0;
    matinfo.Glossiness1=0.0;
    matinfo.Glossiness2=0.0;
    matinfo.SpecShift1=0.0;
    matinfo.SpecShift2=0.0;
    matinfo.SpecSmooth1=0.0;
    matinfo.SpecSmooth2=0.0;
    matinfo.SpecColor1=float4(0.0,0.0,0.0,0.0);
    matinfo.SpecColor2=float4(0.0,0.0,0.0,0.0);
    matinfo.noise=0.0;
#endif
#ifdef GLTF_PBR_USE_SHEEN
    matinfo.Anisotropic = 0.0;
    matinfo.ClearCoat = 0.0;
    matinfo.ClearCoatRounghness = 0.0;
    matinfo.IridescenceMask =0.0;
    matinfo.FilmThickness =0.0;
    matinfo.BaseIor = 0.0;
    matinfo.KExtinction =0.0;
    matinfo.SheenRoughness=SheenRoughness;
    matinfo.SheenColor=SheenColor;
    matinfo.TransmissionFactor=0.0;
    matinfo.BaseColor=float3(0.0,0.0,0.0);
    matinfo.worldmatrix=worldmatrix;
    matinfo.thickness=0.0;
    matinfo.attenuationColor=float3(0.0,0.0,0.0);
    matinfo.attenuationDistance=0.0;
    matinfo.sssDistortion=0.0;
    matinfo.sssPower=0.0;
    matinfo.sssScale=0.0;
    matinfo.sssAttenuation=0.0;
    matinfo.FurLength=0.0;
    matinfo.GravityStrength=0.0;
    matinfo.Thickness=0.0;
    matinfo.Blur=0.0;
    matinfo.Glossiness1=0.0;
    matinfo.Glossiness2=0.0;
    matinfo.SpecShift1=0.0;
    matinfo.SpecShift2=0.0;
    matinfo.SpecSmooth1=0.0;
    matinfo.SpecSmooth2=0.0;
    matinfo.SpecColor1=float4(0.0,0.0,0.0,0.0);
    matinfo.SpecColor2=float4(0.0,0.0,0.0,0.0);
    matinfo.noise=0.0;
#endif
#ifdef GLTF_PBR_USE_TRANSMISSION
    matinfo.Anisotropic = 0.0;
    matinfo.ClearCoat = 0.0;
    matinfo.ClearCoatRounghness = 0.0;
    matinfo.IridescenceMask =0.0;
    matinfo.FilmThickness =0.0;
    matinfo.BaseIor = baseIor;
    matinfo.KExtinction =0.0;
    matinfo.SheenRoughness=0.0;
    matinfo.SheenColor=float3(0.0,0.0,0.0);;
    matinfo.TransmissionFactor=TransmissionFactor;
    matinfo.BaseColor=BaseColor;
    matinfo.worldmatrix=worldmatrix;
    matinfo.thickness=tickness;
    matinfo.attenuationColor=attenuationColor;
    matinfo.attenuationDistance=attenuationDistance;
    matinfo.sssDistortion=0.0;
    matinfo.sssPower=0.0;
    matinfo.sssScale=0.0;
    matinfo.sssAttenuation=0.0;
    matinfo.FurLength=0.0;
    matinfo.GravityStrength=0.0;
    matinfo.Thickness=0.0;
    matinfo.Blur=0.0;
    matinfo.Glossiness1=0.0;
    matinfo.Glossiness2=0.0;
    matinfo.SpecShift1=0.0;
    matinfo.SpecShift2=0.0;
    matinfo.SpecSmooth1=0.0;
    matinfo.SpecSmooth2=0.0;
    matinfo.SpecColor1=float4(0.0,0.0,0.0,0.0);
    matinfo.SpecColor2=float4(0.0,0.0,0.0,0.0);
    matinfo.noise=0.0;
#endif
#ifdef GLTF_PBR_USE_SSS
    matinfo.Anisotropic = 0.0;
    matinfo.ClearCoat = 0.0;
    matinfo.ClearCoatRounghness = 0.0;
    matinfo.IridescenceMask =0.0;
    matinfo.FilmThickness =0.0;
    matinfo.BaseIor = baseIor;
    matinfo.KExtinction =0.0;
    matinfo.SheenRoughness=0.0;
    matinfo.SheenColor=float3(0.0,0.0,0.0);;
    matinfo.TransmissionFactor=0.0;
    matinfo.BaseColor=float3(0.0,0.0,0.0);
    matinfo.worldmatrix=worldmatrix;
    matinfo.thickness=tickness;
    matinfo.attenuationColor=float3(0.0,0.0,0.0);
    matinfo.attenuationDistance=0.0;
    matinfo.sssDistortion=sssDistortion;
    matinfo.sssPower=sssPower;
    matinfo.sssScale=sssScale;
    matinfo.sssAttenuation=sssAttenuation;
    matinfo.FurLength=0.0;
    matinfo.GravityStrength=0.0;
    matinfo.Thickness=0.0;
    matinfo.Blur=0.0;
    matinfo.Glossiness1=0.0;
    matinfo.Glossiness2=0.0;
    matinfo.SpecShift1=0.0;
    matinfo.SpecShift2=0.0;
    matinfo.SpecSmooth1=0.0;
    matinfo.SpecSmooth2=0.0;
    matinfo.SpecColor1=float4(0.0,0.0,0.0,0.0);
    matinfo.SpecColor2=float4(0.0,0.0,0.0,0.0);
    matinfo.noise=0.0;
    
#endif
#ifdef GLTF_PBR_USE_FUR
    matinfo.Anisotropic = 0.0;
    matinfo.ClearCoat = 0.0;
    matinfo.ClearCoatRounghness = 0.0;
    matinfo.IridescenceMask =0.0;
    matinfo.FilmThickness =0.0;
    matinfo.BaseIor = baseIor;
    matinfo.KExtinction =0.0;
    matinfo.SheenRoughness=0.0;
    matinfo.SheenColor=float3(0.0,0.0,0.0);;
    matinfo.TransmissionFactor=0.0;
    matinfo.BaseColor=float3(0.0,0.0,0.0);
    matinfo.worldmatrix=worldmatrix;
    matinfo.thickness=tickness;
    matinfo.attenuationColor=float3(0.0,0.0,0.0);
    matinfo.attenuationDistance=0.0;
    matinfo.sssDistortion=0.0;
    matinfo.sssPower=0.0;
    matinfo.sssScale=0.0;
    matinfo.sssAttenuation=0.0;
    matinfo.FurLength=furInfo.FurLength;
    matinfo.GravityStrength=furInfo.GravityStrength;
    matinfo.Thickness=furInfo.Thickness;
    matinfo.Blur=furInfo.Blur;
    matinfo.Glossiness1=furInfo.Glossiness1;
    matinfo.Glossiness2=furInfo.Glossiness2;
    matinfo.SpecShift1=furInfo.SpecShift1;
    matinfo.SpecShift2=furInfo.SpecShift2;
    matinfo.SpecSmooth1=furInfo.SpecSmooth1;
    matinfo.SpecSmooth2=furInfo.SpecSmooth2;
    matinfo.SpecColor1=furInfo.SpecColor1;
    matinfo.SpecColor2=furInfo.SpecColor1;
    matinfo.noise=furInfo.Noise;
    
#endif
}

#define MEDIUMP_FLT_MAX    65504.0
#define MEDIUMP_FLT_MIN    0.00006103515625
#define saturateMediump(x) min(x, MEDIUMP_FLT_MAX)

float sq(float x) {
    return x * x;
}

float D_GGX_Anisotropic(float at, float ab, float ToH, float BoH, float NoH) {
    // Burley 2012, "Physically-Based Shading at Disney"
    float a2 = at * ab;
    float3 d = float3(ab * ToH, at * BoH, a2 * NoH);
    return saturateMediump(a2 * sq(a2 / dot(d, d)) * (1.0 / 3.141592627));
}

float V_SmithGGXCorrelated_Anisotropic(float at, float ab, float ToV, float BoV,
        float ToL, float BoL, float NoV, float NoL) {
    // Heitz 2014, "Understanding the Masking-Shadowing Function in Microfacet-Based BRDFs"
    // TODO: lambdaV can be pre-computed for all the lights, it should be moved out of this function
    float lambdaV = NoL * length(float3(at * ToV, ab * BoV, NoL));
    float lambdaL = NoV * length(float3(at * ToL, ab * BoL, NoV));
    float v = 0.5 / (lambdaV + lambdaL);
    return saturateMediump(v);
}

float3 F_Schlick(const float3 f0, float f90, float VoH) {
    // Schlick 1994, "An Inexpensive BRDF Model for Physically-Based Rendering"
    float f = pow(1.0 - VoH,5.0);
    return f + f0 * (f90 - f);
}

float distributionAnisotropic(float at, float ab, float ToH, float BoH, float NoH) {
    return D_GGX_Anisotropic(at, ab, ToH, BoH, NoH);
}

float visibilityAnisotropic(float linearRoughness, float at, float ab,
        float ToV, float BoV, float ToL, float BoL, float NoV, float NoL) {
    return V_SmithGGXCorrelated_Anisotropic(at, ab, ToV, BoV, ToL, BoL, NoV, NoL);
}

float3 fresnel(const float3 f0, float LoH) {
    return F_Schlick(f0, 1.0, LoH);
}

float3 anisotropicLobe(float3 anisotropicT,float3 anisotropicB,float anisotropy,
 float3 lightdirection,float3 f0, float3 h,float roughness,float3 view,
        float NoV, float NoL, float NoH, float LoH) {

    float3 l = lightdirection;
    float3 t = anisotropicT;
    float3 b = anisotropicB;
    float3 v = view;

    float ToV = dot(t, v);
    float BoV = dot(b, v);
    float ToL = dot(t, l);
    float BoL = dot(b, l);
    float ToH = dot(t, h);
    float BoH = dot(b, h);

    // Anisotropic parameters: at and ab are the roughness along the tangent and bitangent
    // to simplify materials, we derive them from a single roughness parameter
    // Kulla 2017, "Revisiting Physically Based Shading at Imageworks"
    float at = max(roughness * (1.0 + anisotropy), 0.001);
    float ab = max(roughness * (1.0 -anisotropy), 0.001);

    // specular anisotropic BRDF
    float D = distributionAnisotropic(at, ab, ToH, BoH, NoH);
    float V = visibilityAnisotropic(roughness, at, ab, ToV, BoV, ToL, BoL, NoV, NoL);
    float3  F = fresnel(f0, LoH);

    return (D * V) * F;
}
////pixel.anisotropicB pixel.anisotropicT世界空间下的切线和副切线
float3 getReflectedfloattor(float3 v, float3 n, float3 t, float3 b,float anisotropy) {
    float3  anisotropyDirection = anisotropy >= 0.0 ? b : t;
    float3  anisotropicTangent  = cross(anisotropyDirection, -v);
    float3  anisotropicNormal   = cross(anisotropicTangent, anisotropyDirection);
    float3  bentNormal          = normalize(lerp(n, anisotropicNormal, abs(anisotropy)));

    return reflect(v, bentNormal);
}

float clampedDot(float3 x, float3 y)
{
    return clamp(dot(x, y), 0.0, 1.0);
}

float3 DepolColor(float3 colS, float3 colP)
{
    return 0.5 * (colS + colP);
}



float2 sq(float2 x)
{
    return float2(x.x * x.x, x.y * x.y);
}

// Belcour & Barla 2017, "A practical extension to microfacet theory for the modeling of varying iridescence"
// Fresnel equations for dielectric/dielectric interfaces.
void F_Dielectric(in float ct1, in float n1, in float n2, out float2 R, out float2 phi)
{
  // Sinus theta1 'squared'
    float st1 = (1.0f - ct1 * ct1);
    float nr = n1 / n2;

    if (sq(nr) * st1 > 1.0) // Total Internal Reflection
    {
        R = float2(1.0,1.0);
        phi = 2.0 * atan(float2(-sq(nr) * sqrt(st1 - 1.0f / sq(nr)) / ct1, -sqrt(st1 - 1.0f / sq(nr)) / ct1));
    }
    else // Transmission & Reflection
    {
  
        float ct2 = sqrt(1.0f - sq(nr) * st1);
        float2 r = float2((n2 * ct1 - n1 * ct2) / (n2 * ct1 + n1 * ct2),
            (n1 * ct1 - n2 * ct2) / (n1 * ct1 + n2 * ct2));
        phi.x = (r.x < 1e-5) ? PI : 0.0;
        phi.y = (r.y < 1e-5) ? PI : 0.0;
        R = sq(r);
    }
}

float3 evaluateSensitivity(float opd, float shift)
{
  // OPD: Optical Path Difference
  // Use Gaussian fits, given by 3 parameters: val, pos and var
    float phase = 2.0f * PI * opd * 1.0e-6;
    float3 val = float3(5.4856e-13, 4.4201e-13, 5.2481e-13);
    float3 pos = float3(1.6810e+6, 1.7953e+6, 2.2084e+6);
    float3 var = float3(4.3278e+9, 9.3046e+9, 6.6121e+9);
    float3 xyz = val * sqrt(2.0 * PI * var) * cos(pos * phase + shift) * exp(-var * phase * phase);
    xyz.x += (9.7470e-14 * sqrt(2.0 * PI * 4.5282e+9) * cos(2.2399e+6 * phase + shift) * exp(-4.5282e+9 * phase * phase));
    return xyz / 1.0685e-7;
}

// Belcour & Barla 2017, "A practical extension to microfacet theory for the modeling of varying iridescence"
// Fresnel equations for dielectric/conductor interfaces.
void F_Conductor(in float ct1, in float n1, in float n2, in float k, out float2 R, out float2 phi)
{
    if (k < 1e-5)
    {
    // Use dielectric formula to avoid numerical issues
        F_Dielectric(ct1, n1, n2, R, phi);
        return;
    }

    float A = sq(n2) * (1.0 - sq(k)) - sq(n1) * (1.0 - sq(ct1));
    float B = sqrt(sq(A) + sq(2.0 * sq(n2) * k));
    float U = sqrt((A + B) / 2.0);
    float V = sqrt((B - A) / 2.0);

    R.y = (sq(n1 * ct1 - U) + sq(V)) / (sq(n1 * ct1 + U) + sq(V));
    phi.y = atan2(sq(U) + sq(V) - sq(n1 * ct1), 2.0 * n1 * V * ct1) + PI;

    R.x = (sq(sq(n2) * (1.0 - sq(k)) * ct1 - n1 * U) + sq(2.0 * sq(n2) * k * ct1 - n1 * V))
    / (sq(sq(n2) * (1.0 - sq(k)) * ct1 + n1 * U) + sq(2.0 * sq(n2) * k * ct1 + n1 * V));
    phi.x = atan2(sq(sq(n2) * (1.0 + sq(k)) * ct1) - sq(n1) * (sq(U) + sq(V)), 2.0 * n1 * sq(n2) * ct1 * (2.0 * k * U - (1.0 - sq(k)) * V));
}

float Depol(float2 polV)
{
    return 0.5 * (polV.x + polV.y);
}
float3 F_Airy(float ct1, float ct2, float Dinc, float eta_1, float eta_2, float eta_3, float k)
{
  // First interface
    float2 R12, phi12;
    F_Dielectric(ct1, eta_1, eta_2, R12, phi12);
    float2 R21 = R12;
    float2 T121 = 1.0 - R12;
    float2 phi21 = float2(PI, PI) - phi12;

  // Second interface
    float2 R23, phi23;
    F_Conductor(ct2, eta_2, eta_3, k, R23, phi23);
  
  // Phase shift
    float OPD = Dinc * ct2;
    float2 phi2 = phi21 + phi23;

  // Compound terms
    float3 I = float3(0, 0, 0);
    float2 R123 = R12 * R23;
    float2 r123 = sqrt(R123);
    float2 Rs = sq(T121) * R23 / (1.0 - R123);

  // Reflectance term for m=0 (DC term amplitude)
    float2 C0 = R12 + Rs;
    float3 S0 = evaluateSensitivity(0.0, 0.0);
    I += (Depol(C0) * S0);

  // Reflectance term for m>0 (pairs of diracs)
    float2 Cm = Rs - T121;
    for (int m = 1; m <= 3; ++m)
    {
        Cm *= r123;
        float3 SmS = 2.0 * evaluateSensitivity(float(m) * OPD, float(m) * phi2.x);
        float3 SmP = 2.0 * evaluateSensitivity(float(m) * OPD, float(m) * phi2.y);
        I += DepolColor(Cm.x * SmS, Cm.y * SmP);
    }

  // Convert back to RGB reflectance
    I = clamp(mul(I, XYZ_TO_RGB), 0.0, 1.0);

    return I;
}
float3 fresnelIridescence(float filmThickness, float baseIor, float kExtinction, float LoH)
{
  // iridescenceThickness unit is micrometer for this equation here. Mean 0.5 is 500nm.
    float Dinc = filmThickness * 3.0;
    float eta_0 = 1.0;
    float eta_1 = 1.0;
    float eta_2 = lerp(2.0, 1.0, filmThickness);
    float eta_3 = baseIor;
    float cosTheta1 = sqrt(1.0 + sq(eta_0 / eta_1) * (sq(LoH) - 1.0));
    float cosTheta2 = sqrt(1.0 - sq(eta_1 / eta_2) * (1.0 - sq(cosTheta1)));
    return F_Airy(cosTheta1, cosTheta2, Dinc, eta_1, eta_2, eta_3, kExtinction);
}


float D_Charlie(float sheenRoughness, float NdotH)
{
    sheenRoughness = max(sheenRoughness, 0.000001); //clamp (0,1]
    float alphaG = sheenRoughness * sheenRoughness;
    float invR = 1.0 / alphaG;
    float cos2h = NdotH * NdotH;
    float sin2h = 1.0 - cos2h;
    return (2.0 + invR) * pow(sin2h, invR * 0.5) / (2.0 * 3.1415926);
}

float lambdaSheenNumericHelper(float x, float alphaG)
{
    float oneMinusAlphaSq = (1.0 - alphaG) * (1.0 - alphaG);
    float a = lerp(21.5473, 25.3245, oneMinusAlphaSq);
    float b = lerp(3.82987, 3.32435, oneMinusAlphaSq);
    float c = lerp(0.19823, 0.16801, oneMinusAlphaSq);
    float d = lerp(-1.97760, -1.27393, oneMinusAlphaSq);
    float e = lerp(-4.32054, -4.85967, oneMinusAlphaSq);
    return a / (1.0 + b * pow(x, c)) + d * x + e;
}


float lambdaSheen(float cosTheta, float alphaG)
{
    if (abs(cosTheta) < 0.5)
    {
        return exp(lambdaSheenNumericHelper(cosTheta, alphaG));
    }
    else
    {
        return exp(2.0 * lambdaSheenNumericHelper(0.5, alphaG) - lambdaSheenNumericHelper(1.0 - cosTheta, alphaG));
    }
}
float V_Sheen(float NdotL, float NdotV, float sheenRoughness)
{
    sheenRoughness = max(sheenRoughness, 0.000001); //clamp (0,1]
    float alphaG = sheenRoughness * sheenRoughness;

    return clamp(1.0 / ((1.0 + lambdaSheen(NdotV, alphaG) + lambdaSheen(NdotL, alphaG)) *
        (4.0 * NdotV * NdotL)), 0.0, 1.0);
}
float3 BRDF_specularSheen(float3 sheenColor, float sheenRoughness, float NdotL, float NdotV, float NdotH)
{
    float sheenDistribution = D_Charlie(sheenRoughness, NdotH);
    float sheenVisibility = V_Sheen(NdotL, NdotV, sheenRoughness);
    return sheenColor * sheenDistribution * sheenVisibility;
}

float3 getPunctualRadianceSheen(float3 sheenColor, float sheenRoughness, float NdotL, float NdotV, float NdotH)
{
    return NdotL * BRDF_specularSheen(sheenColor, sheenRoughness, NdotL, NdotV, NdotH);
}

float3 getVolumeTransmissionRay(float3 n, float3 v, float thickness, float ior, float4x4 modelMatrix)
{
    // Direction of refracted light.
    float3 refractionfloattor = refract(-v, normalize(n), 1.0 / ior);

    // Compute rotation-independant scaling of the model matrix.
    float3 modelScale;
    modelScale.x = length(float3(modelMatrix[0].xyz));
    modelScale.y = length(float3(modelMatrix[1].xyz));
    modelScale.z = length(float3(modelMatrix[2].xyz));

    // The thickness is specified in local space.
    return normalize(refractionfloattor) * thickness * modelScale;
}

float applyIorToRoughness(float roughness, float ior)
{
    // Scale roughness with IOR so that an IOR of 1.0 results in no microfacet refraction and
    // an IOR of 1.5 results in the default amount of microfacet refraction.
    return roughness * clamp(ior * 2.0 - 2.0, 0.0, 1.0);
}


float3 getPunctualRadianceTransmission(SurfaceReflectanceInfo SrfInfo,
float3 normal, float3 view, float3 pointToLight, float alphaRoughness,
    float3 f0, float3 f90, float3 baseColor, float ior)
{
    float transmissionRougness = applyIorToRoughness(alphaRoughness, ior);

    float3 n = normalize(normal); // Outward direction of surface point
    float3 v = normalize(view); // Direction from surface point to view
    float3 l = normalize(pointToLight);
    float3 l_mirror = normalize(l + 2.0 * n * dot(-l, n)); // Mirror light reflection floattor on surface
    float3 h = normalize(l_mirror + v); // Halfway floattor between transmission light floattor and v


    float D = NormalDistribution_GGX(clamp(dot(n, h), 0.0, 1.0), transmissionRougness);
    float Vis = SmithGGXVisibilityCorrelated(clamp(dot(n, l_mirror), 0.0, 1.0), clamp(dot(n, v), 0.0, 1.0), transmissionRougness);
    float3 F = SchlickReflection(clamp(dot(v, h), 0.0, 1.0), SrfInfo.Reflectance0, SrfInfo.Reflectance90);
    // Transmission BTDF
    return (1.0 - F) * baseColor * D * Vis;
}

float3 applyVolumeAttenuation(in float3 radiance, float transmissionDistance, float3 attenuationColor, float attenuationDistance)
{

    if (attenuationDistance == 0.0)
    {
        // Attenuation distance is +∞ (which we indicate by zero), i.e. the transmitted color is not attenuated at all.
        return radiance;
    }
    else
    {
      
        // Compute light attenuation using Beer's law.
        float3 attenuationCoefficient = -log(attenuationColor)/ attenuationDistance;
        float3 transmittance =exp(-attenuationCoefficient * transmissionDistance); // Beer's law
       
        return transmittance * radiance;
    }
 	
}

float3 getTransmissionSample(float2 fragCoord, float roughness, float ior, Texture2D framemap, SamplerState defaultSampler,float PrefilteredCubeMipLevels)
{
    uint width, height, levels;
    framemap.GetDimensions(0, width, height, levels);
    uint specularTextureLevels = uint(PrefilteredCubeMipLevels);
    float framebufferLod = log2(float(width)) * applyIorToRoughness(roughness, ior);
    float3 transmittedLight = framemap.SampleLevel(defaultSampler, float2(fragCoord.x, 1.0 - fragCoord.y), framebufferLod).rgb;
    return transmittedLight;
}


float3 getIBLVolumeRefraction(float3 n, float3 v, float perceptualRoughness, float3 baseColor, float3 f0, float3 f90,
    float3 position, float4x4 modelMatrix, float4x4 projMatrix, float ior, float thickness, 
float3 attenuationColor, float attenuationDistance, Texture2D specularBRDF_LUT, SamplerState spBRDF_Sampler,
Texture2D framemap, SamplerState defaultSampler, float PrefilteredCubeMipLevels)
{
    float3 transmissionRay = getVolumeTransmissionRay(n, v, thickness, ior, modelMatrix);
    float3 refractedRayExit = position + transmissionRay;
  
    // Project refracted floattor on the framebuffer, while mapping to normalized device coordinates.mul(viewMatrix, )
    float4 ndcPos = mul(float4(refractedRayExit, 1.0), projMatrix);
    float2 refractionCoords = ndcPos.xy / ndcPos.w;
    refractionCoords += 1.0;
    refractionCoords /= 2.0;

    // Sample framebuffer to get pixel the refracted ray hits.
    float3 transmittedLight = getTransmissionSample(refractionCoords, perceptualRoughness, ior,
     framemap,  defaultSampler,  PrefilteredCubeMipLevels);
  
    float3 attenuatedColor = applyVolumeAttenuation(transmittedLight, length(transmissionRay), attenuationColor, attenuationDistance);
    
    // Sample GGX LUT to get the specular component.
    float NdotV = clampedDot(n, v);
    float2 brdfSamplePoint = clamp(float2(NdotV, perceptualRoughness), float2(0.0, 0.0), float2(1.0, 1.0));
    float2 brdf = specularBRDF_LUT.Sample(spBRDF_Sampler, brdfSamplePoint).rg;
    float3 specularColor = f0 * brdf.x + f90 * brdf.y;

    return (1.0 - specularColor) * attenuatedColor * baseColor;
}
float3 DirectKajiyaSpecular(float3 bitangent, float3 normalWS, float3 floatAngle, float noise, 
float SpecShift1, float SpecShift2, float SpecSmooth1, float SpecSmooth2, float Glossiness1, float Glossiness2, float4 SpecColor1, float4 SpecColor2)
{
    float3 T1 = bitangent + (-0.5 + SpecShift1) * normalWS;
    T1 = normalize(T1);

    float3 T2 = bitangent + (noise + SpecShift2) * normalWS;
    T2 = normalize(T2);

    float tdoth1 = dot(T1, floatAngle);
    float spec1 = saturate(smoothstep(-1.0, 0.0, tdoth1) * pow(sqrt(1.0 - tdoth1 * tdoth1), SpecSmooth1));

    float tdoth2 = dot(T2, floatAngle);
    float spec2 = saturate(smoothstep(-1.0, 0.0, tdoth2) * pow(sqrt(1.0 - tdoth2 * tdoth2), SpecSmooth2));

    float3 _SpecColor1_linear = max(SpecColor1.xyz, float3(0.001, 0.001, 0.001));
    float3 _SpecColor2_linear = max(SpecColor2.xyz, float3(0.001, 0.001, 0.001));

    float3 specular1 = _SpecColor1_linear * spec1 * Glossiness1;
    float3 specular2 = _SpecColor2_linear * spec2 * Glossiness2;
    float3 specular = specular1 + specular2;
    return specular * noise * noise;
}
void BRDF(in float3                 PointToLight, 
          in float3                 Normal,
		  in float3                 Tangent,
		  in float3                 BitTangent,
          in float3                 View,
          in SurfaceReflectanceInfo SrfInfo,
		  in MaterialInfo			materialInfo,
          out float3                DiffuseContrib,
          out float3                SpecContrib,
		  out float3 				clearcoat,
		  out float3 				clearcoatFresnel,
          out float3                sheencolor,
          out float3                TransmissionColor,
          out float3                sss,
          out float3                furspec,     
          out float                 NdotL)
{
    AngularInfo angularInfo = GetAngularInfo(PointToLight, Normal, View);

    DiffuseContrib = float3(0.0, 0.0, 0.0);
    SpecContrib    = float3(0.0, 0.0, 0.0);
    NdotL          = angularInfo.NdotL;  
    clearcoat = float3(1.0, 1.0, 1.0);
    clearcoatFresnel = float3(1.0, 1.0, 1.0);
    sheencolor = float3(0.0, 0.0, 0.0);
    TransmissionColor = float3(0.0, 0.0, 0.0);
    sss = float3(0.0, 0.0, 0.0);
    furspec = float3(0.0, 0.0, 0.0);
    // If one of the dot products is larger than zero, no division by zero can happen. Avoids black borders.
    if (angularInfo.NdotL > 0.0 || angularInfo.NdotV > 0.0)
    {
        //           D(h,a) * G(v,l,a) * F(v,h,f0)
        // f(v,l) = -------------------------------- = D(h,a) * Vis(v,l,a) * F(v,h,f0)
        //               4 * (n,v) * (n,l)
        // where
        //
        // Vis(v,l,a) = G(v,l,a) / (4 * (n,v) * (n,l))

        // It is not a mistake that AlphaRoughness = PerceptualRoughness ^ 2 and that
        // SmithGGXVisibilityCorrelated and NormalDistribution_GGX then use a2 = AlphaRoughness ^ 2.
        // See eq. 3 in https://blog.selfshadow.com/publications/s2013-shading-course/karis/s2013_pbs_epic_notes_v2.pdf
        float AlphaRoughness = SrfInfo.PerceptualRoughness * SrfInfo.PerceptualRoughness;
        float  D   = NormalDistribution_GGX(angularInfo.NdotH, AlphaRoughness);
        float  Vis = SmithGGXVisibilityCorrelated(angularInfo.NdotL, angularInfo.NdotV, AlphaRoughness);
        float3 F = float3(0.0, 0.0, 0.0);
#ifdef PBR_Default 
         F = SchlickReflection(angularInfo.VdotH, SrfInfo.Reflectance0, SrfInfo.Reflectance90);
   #endif
#ifdef GLTF_PBR_USE_IRR 
        F=fresnelIridescence(materialInfo.FilmThickness,
        materialInfo.BaseIor,materialInfo.KExtinction,angularInfo.NdotV);
        F = lerp(SchlickReflection(angularInfo.VdotH, SrfInfo.Reflectance0, SrfInfo.Reflectance90), F, materialInfo.IridescenceMask);
        #endif 
        DiffuseContrib = (1.0 - F) * LambertianDiffuse(SrfInfo.DiffuseColor);
        float3 iridescenceAttenuation = float3(0.0,0.0,0.0);
#ifdef GLTF_PBR_USE_IRR 
         iridescenceAttenuation = 1.0 - F *materialInfo.IridescenceMask;
        DiffuseContrib*=iridescenceAttenuation;
#endif 
		#if PBR_Default 
        SpecContrib    = F * Vis * D;
                    #ifdef GLTF_PBR_USE_IRR 
                         SpecContrib*=iridescenceAttenuation* iridescenceAttenuation;
                    #endif 
           #ifdef GLTF_PBR_USE_CLEARCOAT 
	    float ccroughness2=materialInfo.ClearCoatRounghness*materialInfo.ClearCoatRounghness;
	    float  ClearCoatD   = NormalDistribution_GGX(angularInfo.NdotH, ccroughness2);
        float  ClearCoatVis = SmithGGXVisibilityCorrelated(angularInfo.NdotL, angularInfo.NdotV, ccroughness2);
        float3 ClearCoatF   = SchlickReflection(angularInfo.VdotH, SrfInfo.Reflectance0, SrfInfo.Reflectance90);
	    clearcoat+= NdotL*ClearCoatF * ClearCoatVis *ClearCoatD;
	    clearcoatFresnel+=SchlickReflection(angularInfo.VdotH, SrfInfo.Reflectance0, SrfInfo.Reflectance90);
            #endif
		#endif

		#ifdef GLTF_PBR_USE_TANGENT 
		 SpecContrib    = anisotropicLobe(Tangent,BitTangent,materialInfo.Anisotropic,
 PointToLight,SrfInfo.Reflectance0, angularInfo.H,SrfInfo.PerceptualRoughness,View,
        angularInfo.NdotV, angularInfo.NdotL,angularInfo.NdotH, angularInfo.LdotH);
		#endif
      
#ifdef GLTF_PBR_USE_SHEEN
       sheencolor+=getPunctualRadianceSheen(materialInfo.SheenColor, materialInfo.SheenRoughness, angularInfo.LdotH,angularInfo.VdotH,angularInfo.NdotH);
#endif
       
        //float3 transmittedLight = float3(0.0, 0.0, 0.0);
#ifdef GLTF_PBR_USE_TRANSMISSION
      
        float3 transmissionRay = getVolumeTransmissionRay(Normal,View, materialInfo.thickness, materialInfo.BaseIor,materialInfo.worldmatrix);
        PointToLight-=transmissionRay;
        float3 l = normalize(PointToLight);
      
       float3 transmittedLight=getPunctualRadianceTransmission(SrfInfo,
    Normal,View,l,SrfInfo.PerceptualRoughness,
    SrfInfo.Reflectance0,SrfInfo.Reflectance90, materialInfo.BaseColor, materialInfo.BaseIor);
#ifdef GLTF_PBR_USE_VOLUME
       transmittedLight=applyVolumeAttenuation(transmittedLight,length(transmissionRay),materialInfo.attenuationColor, materialInfo.attenuationDistance);
#endif
       TransmissionColor=materialInfo.TransmissionFactor*transmittedLight;
#endif
       
#ifdef GLTF_PBR_USE_SSS
        
    float3 L = PointToLight;
    float3 V = View;
   
    float3 H = normalize(L +normalize(Normal) * materialInfo.sssDistortion);
    float VdotH = pow(saturate(dot(V, -H)), materialInfo.sssPower) *  materialInfo.sssScale;
    sss  = materialInfo.sssAttenuation * (VdotH +  0.2) * materialInfo.thickness;
#endif     
        
#ifdef GLTF_PBR_USE_FUR
      furspec=  DirectKajiyaSpecular(BitTangent, Normal, angularInfo.H, materialInfo.noise, 
materialInfo.SpecShift1,materialInfo.SpecShift2, materialInfo.SpecSmooth1, materialInfo.SpecSmooth2,materialInfo.Glossiness1,materialInfo.Glossiness2, materialInfo.SpecColor1,materialInfo.SpecColor2);
#endif
        
        
    }
}

#endif // _PBR_COMMON_FXH_
